(* finalize.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: finalize.ml,v 1.9 2003/09/26 20:44:59 rflury Exp $ *)

module HA = Hashtbl
module TP = Temp
module F = (Frame.X86_FRAME : Frame.FrameSig)
module E = Errormsg

exception FinalizeError of string

(* Transform from a list of BB to a single list of assem *)
let bb2code munchedAssem =
  List.fold_left (fun res (asmList, _) -> res @ asmList) [] munchedAssem
    

(* Replaces all jump <Return-Label> with the propper exit-sequence *)
let insertReturn assemList returnCode = 
  let tmp = List.fold_left (fun res instr -> 
    match instr with 
    | Assem.OPER("jmp\t 'l0",_,_,labels) when 
	(List.hd labels) = Temp.getReturnLabel ()-> 
	(List.rev returnCode) @ res
    | _ -> instr :: res
			   ) [] assemList
  in
  List.rev tmp

(* Returns the string representation of a register *)
let reg2string register = 
  match register with
  | 1 -> "%eax"
  | 2 -> "%ecx"
  | 3 -> "%edx"
  | 4 -> "%ebx"
  | 5 -> "%esi"
  | 6 -> "%edi"
  | 7 -> "%esp"
  | 8 -> "%ebp"
  | x -> raise (FinalizeError ("Invalid Register : " ^ string_of_int x))


(* Returns the string representation of a temporary given a 
 * function thar returns the color of a temp *)
let temp2string temp2color temp instr = 
  (* Returns the string representation of a byte register *)
  let byteReg2string register = 
    match register with
    | 1 -> "%al"
    | 2 -> "%cl"
    | 3 -> "%dl"
    | 4 -> "%bl"
    | x -> reg2string register (* called for all source-temps *)
  in
  let color = temp2color temp in
  if(String.length instr < 3) then
    reg2string color
  else 
    match String.sub instr 0 3 with
    | "sal" | "sar" -> 
	byteReg2string color
    | _ -> 
	reg2string color



let invar2string colors spilled_list lab_invar label = 
  (* takes a temp and returns a string for this temp *)
  let single2string temp = 
    let temp2color tmp = 
      (try HA.find colors tmp with Not_found -> 
	raise (FinalizeError (Printf.sprintf "Temp %i not assigned a color\n" tmp))
      )
    in
    let ((_,off),spilled) = 
      (try
	(List.find (fun (t,o) -> t = temp) spilled_list, true)
      with Not_found -> 
	((temp, 0), false)
      )
    in
    let typ =
      (try
	TP.temp2pccType temp 
      with Not_found -> 
	raise (FinalizeError (Printf.sprintf "Invalid Invariant temp: %i\n" temp))
      )
    in
    let s = 
      if(spilled) then
	Printf.sprintf "-%i(%%ebp)" off
      else
	temp2string temp2color temp ""
    in
    Printf.sprintf "%s:%s" s (Pcc.pccType2string typ)
  in

  (* must remove duplicates from invar-list *)
  let li = HA.find_all lab_invar label in
  let invar = List.fold_left (fun res l -> 
    List.fold_left (fun res2 t -> 
      if(List.mem t res2) then
	res2
      else
	t :: res2
		  ) res l 
			     ) [] li
  in
  
  if(invar = []) then
    ""
  else (
    let result = ref "" in
    let add s = result := !result ^ s in
    add "\t/* ";
    List.iter (fun s -> 
      match s with
      | Ir.TypeInvar(x) -> 
	  if(x != 7 && x != 8) then (
	    add (single2string x); add "; "
	   )
      | _ -> ()
	      ) invar;
    add "*/\n";
    !result
   )

      

let finalize_fun ((fName, munchAssem), colors, offset, 
		  spilled_list, used_colors, lab_invar) output =
  E.showProgress (Printf.sprintf "Finalize fun %s " fName);
  let addCode s = output_string output s in
  Temp.openFunLookUp fName;

  (* Pass some info to the summary module *)
  Summary.add_spills fName (offset / 4);
  Summary.add_used_regs fName 
    (List.fold_right (fun r res -> res ^ " " ^ reg2string r) used_colors "");
  
  
  
  let extName = Helpers.get_external_symbol fName in
  (* Function Entry *)
  addCode (".global " ^ extName ^ "\n");
  if(!Helpers.isWindows) then
    addCode("\t.def " ^ fName ^ "; .scl 2; .type 32; .endef\n")
  else
    addCode ("\t.type " ^ extName ^ ",@function\n");
  addCode (extName ^ ":\n");
  addCode ("\tpushl\t %ebp\n");
  addCode ("\tmovl\t %esp, %ebp\n");
  if(offset > 0) then
    addCode ("\tsubl\t $" ^ string_of_int offset ^ ", %esp\n");
  if(List.mem 4 used_colors) then
    addCode ("\tpushl\t %ebx\n");
  if(List.mem 5 used_colors) then  
    addCode ("\tpushl\t %esi\n");
  if(List.mem 6 used_colors) then    
    addCode ("\tpushl\t %edi\n");
  
  (* Returns the register (as int) given a temp *)
  let temp2reg temp = 
    (try 
      HA.find colors temp
    with Not_found -> 
      raise (FinalizeError ("Temp " ^ string_of_int temp ^ 
			    " was not assigned a color"))
    ) in

  (* The return-code *)
  let returnCode = ref [] in
  if(List.mem 6 used_colors) then    
    returnCode := !returnCode @ [Assem.OPER("popl\t %edi",[],[],[])];
  if(List.mem 5 used_colors) then  
    returnCode := !returnCode @ [Assem.OPER("popl\t %esi",[],[],[])];
  if(List.mem 4 used_colors) then
    returnCode := !returnCode @ [Assem.OPER("popl\t %ebx",[],[],[])];
  (* 'main' is supposed to return 0 on success *)
  if(0 = Pervasives.compare fName "main") then
    returnCode := !returnCode @ [Assem.OPER("movl\t $0, %eax",[],[],[])];
  returnCode := !returnCode @ [Assem.OPER("leave",[],[],[]);
			       Assem.OPER("ret",[],[],[])];
  
  let assemList = bb2code munchAssem in  
  let assemList = insertReturn assemList !returnCode in

  E.showProgress "write to file ";
  List.iter (fun i -> 
    addCode(Assem.format 
	      (temp2string temp2reg) 
	      (invar2string colors spilled_list lab_invar)
	      i
	   )
	    ) assemList;
  E.showProgress "done "
  

let finalize_prog program fileName output = 
  E.showProgress "\n\n\t\t\tFinalizing\n\n";
  let addOutput s = output_string output s in
  
  addOutput ("\t.file\t\"" ^ fileName ^ "\"\n");
  addOutput ".section .data\n";
  addOutput (Temp.label2string (F.nullPointerLabel ()) ^ ":  /* The NULL-pointer */\n");
  addOutput "\t.long 0\n";
  addOutput "\t.long 0\n";
  addOutput "\t.long 0\n";

  addOutput ("\t.section\t.rodata\n");
  HA.iter (fun l s -> 
    addOutput (Temp.label2string l ^ ":\n");
    addOutput ("\t.string \"" ^ s ^ "\"\n");
	  ) (Temp.getStringLabels ());
  
  addOutput "\n\t.text\n";
  List.iter (fun f -> finalize_fun f output) program;
  addOutput "\t.ident\t \"CMU:15-411:F03:Roland Flury\"\n"
    
    
    
