/**
 * The FFI library for the OCaml evaluator
 * 15-411:L3, Alex, Roland
 */

/**
 * Foreign functions in OCaml are declared with the 'external' keyword. 
 * For documentation, read chapter 12 of the Oreilly Book, which can be
 * found at http://caml.inria.fr/oreilly-book/ (pdf or html).
 */ 

// User defined libraries (not needed)
#include <stdio.h>
#include <math.h>

// Needed libraries
#include "caml/mlvalues.h"
#include <dlfcn.h>

#define USE_MALLOC

value ffi_call(value funname, value args) {
  void* fun;
  int retval = 0;
  int arg; 

  fun = dlsym(NULL,String_val(funname));

  if (fun == NULL) {
    printf("Function %s did not initialize. Aborting.\n",String_val(funname));
    exit(3);
  }

  switch(Wosize_val(args)) {
  case 0:
    retval = ((int (*)())fun)();
    break;
  case 1:
    retval = ((int (*)(int))fun)
      (Int32_val(Field(args,0)));
    break;
  case 2:
    retval = ((int (*)(int,int))fun)
      (Int32_val(Field(args,0)),
       Int32_val(Field(args,1)));
    break;
  case 3:
    retval = ((int (*)(int,int,int))fun)
      (Int32_val(Field(args,0)),
       Int32_val(Field(args,1)),
       Int32_val(Field(args,2)));
    break;
  case 4:
    retval = ((int (*)(int,int,int,int))fun)
      (Int32_val(Field(args,0)),
       Int32_val(Field(args,1)),
       Int32_val(Field(args,2)),
       Int32_val(Field(args,3)));
    break;
  case 5:
    retval = ((int (*)(int,int,int,int,int))fun)
      (Int32_val(Field(args,0)),
       Int32_val(Field(args,1)),
       Int32_val(Field(args,2)),
       Int32_val(Field(args,3)),
       Int32_val(Field(args,4)));
    break;
  case 6:
    retval = ((int (*)(int,int,int,int,int,int))fun)
      (Int32_val(Field(args,0)),
       Int32_val(Field(args,1)),
       Int32_val(Field(args,2)),
       Int32_val(Field(args,3)),
       Int32_val(Field(args,4)),
       Int32_val(Field(args,5)));
    break;
  default: 
    printf("FFI function '%s' called with too many arguments (%d). Aborting.\n",
	   String_val(funname), Wosize_val(args));
    exit(3);
  }
  return copy_int32(retval);
}


// The Standard Library for our language
void print_int(int i) {
  printf("%d", i);
  fflush(stdout);
}

value read_int() {
  int i;
  i = 0;
  if(!scanf("%i",&i)) {
    scanf("%*s"); // read illegal chars.
  }
  return i;
}

void print_char(int c) {
  printf("%c", (255 & c));
  fflush(stdout);
}

void print_bool(int x) {
  if (x)
    printf("true");
  else 
    printf("false");
  fflush(stdout);
}

void print_newline() {
  printf("\n");
  fflush(stdout);
}

void print_tab() {
  printf("\t");
  fflush(stdout);
}

void print_space(int n) {
  for(; n>0; n--)
    printf(" ");
  fflush(stdout);
}

void print_flush() {
  fflush(stdout);
  fflush(stdout);
}

void print_float(int f) {
  printf("%f", f);
  fflush(stdout);
}

void* _alloc (unsigned int nbytes) {
  
  void *ret;
  
#ifdef USE_MALLOC
  if ((ret = malloc (nbytes)) == NULL) 
#else
    if ((ret = GC_MALLOC (nbytes)) == NULL)  
#endif
    {
      fprintf (stderr, "out of memory!\n");
      exit (2);
    }    
    
    return ret;
}

int* _allocHeap(int size) {
  return _alloc(size*sizeof(int));
}

int** _allocHeapWithPointer(int size) {
  int* heap = _alloc(size*sizeof(int));
  int** fptr = (int**) _alloc(3*sizeof(int));
  
  fptr[0] = heap;
  fptr[1] = (int*)0;
  fptr[2] = (int*)size;
  
  return fptr;
}

int** _addressToFatPointer(int* address) {

  int** fptr = _alloc(12);
  fptr[0] = address;
  fptr[1] = (int*)0;
  fptr[2] = (int*)1;

  return fptr;
}

int** _allocFatPointer() {
  int** fptr = _alloc(12);
  
  fptr[0] = NULL;
  fptr[1] = (int*)0;
  fptr[2] = (int*)0;
  
  return fptr;
}

#define OFFSET	0
#define SIZE	1


int _pointerInfo(int* ref, int side) {
  
  if (side == OFFSET) {
    return (int)ref[1];
  }
  else {
    return (int)ref[2];
  }
   
}

int** _pointerAddition(int** fptr, int exp, int addition) {
  
  int** nfptr = _alloc(12);

  if (addition == 1)
    nfptr[1] = (int*)((int)fptr[1]+exp);
  else
    nfptr[1] = (int*)((int)fptr[1]-exp);
  
  nfptr[0] = fptr[0];
  nfptr[2] = fptr[2];

  return nfptr;
}


void _pointer_error (const char *srcfile, unsigned int line,
                     const char *errmsg)
{
  if (srcfile == NULL){
    if (errmsg == NULL)
      fprintf (stderr, "RUNTIME: PTRERROR: No message\n");
    else
      fprintf (stderr, "RUNTIME: PTRERROR: %s\n", errmsg);
  } else {
    if (errmsg == NULL)
      fprintf (stderr, "RUNTIME: %s:%d PTRERROR: No Message\n", srcfile, line);
    else fprintf (stderr, "RUNTIME: %s:%d PTRERROR: %s\n", srcfile, line, errmsg);
  }

  exit(-1);
}

int* _checkFatHeapPointer(int** fptr, int line) {

  char buffer[256];

  if ((int)fptr[1] < 0) {
      _pointer_error("_",line,"Index < 0");
      return 0;
  }	
  else if ((int)fptr[1] >= (int)fptr[2]) {
      sprintf(buffer, "Index >= Size (Index: %d Size: %d)", (int)fptr[1], (int)fptr[2]);
      _pointer_error("_",line,buffer);
      return 0;
  }
  else if (fptr[0] == NULL) {
      _pointer_error("_",line,"Attempted to dereference the NULL pointer");
      return 0;
  }
  else {
    return fptr[0] + (int)fptr[1];
  }

}

FILE* f;
FILE* out;

void fileopen() {
  f = fopen("testfile","r+");
}

int filereadint(int pos) {
  int x;
  fseek(f,pos,SEEK_SET);
  fread(&x,sizeof(int),1,f);
  return x;
}

int filereadbyte(int pos) {
  unsigned char x;
  fseek(f,pos,SEEK_SET);
  fread(&x,sizeof(unsigned char),1,f);
  return (unsigned int)x;
}

void fileclose() {
  fclose(f);
}

void outopen() {
  out = fopen("outfile","w+");
}

void outwritebyte(int byte) {
  unsigned char x;
  x = (unsigned char)byte;
  fwrite(&x,sizeof(unsigned char),1,out);
  return;
}

void outclose() {
  fclose(out);
}

// PRIMARY FFI
# ifndef NOSYM
typedef struct {
  int numargs;
  int args[10];
}ffi_group;

int ffi_init() {
  ffi_group* f = (ffi_group*)malloc(sizeof(ffi_group));
  return (int)f;
}

void set_arg_count(int id, int count) {
  ffi_group* f = (ffi_group*)id;
  f->numargs = count;
}

void set_arg(int id, int argnum, int val) {
  ffi_group* f = (ffi_group*)id;
  f->args[argnum] = val;
}

int call(int id, char* funname) {
  void* fun;
  int retval=0;
  int isfix=0;
  char* c;
  ffi_group* f = (ffi_group*)id;

  fun = dlsym(NULL,funname);
  if (fun == NULL) {
    printf("Function %s did not initialize. Aborting.\n",funname);
    exit(3);
  }
  switch(f->numargs) {
  case 0:
    retval = ((int (*)())fun)();
    break;
  case 1:
    retval = ((int (*)(int))fun)(f->args[0]);
    break;
  case 2:
    retval = ((int (*)(int,int))fun)(f->args[0],f->args[1]);
    break;
  case 3:
    retval = ((int (*)(int,int,int))fun)(f->args[0],f->args[1],f->args[2]);
    break;
  case 4:
    retval = ((int (*)(int,int,int,int))fun)(f->args[0],f->args[1],f->args[2],f->args[3]);
    break;
  case 5:
    retval = ((int (*)(int,int,int,int,int))fun)(f->args[0],f->args[1],f->args[2],f->args[3],f->args[4]);
    break;
  case 6:
    retval = ((int (*)(int,int,int,int,int,int))fun)(f->args[0],f->args[1],f->args[2],f->args[3],f->args[4],f->args[5]);
    break;
  }
  free(f);
  return retval;
}
#endif

// END OF PRIMARY FFI

// SPECIAL STUFF

int begin_string() {
  unsigned char *str;
  str = (unsigned char*)malloc(100 * sizeof(unsigned char));
  str[0] = '\0';
}

void add_string(int s, int ch) {
  int len;
  unsigned char* str = (unsigned char*)s;
  len = strlen(str);
  str[len] = (unsigned char)ch;
  str[len+1] = '\0';
}

void stop_string(int s) {
  free((unsigned char*)s);
}






