(* evaluator.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: evaluator.ml,v 1.6 2003/09/26 16:54:49 rflury Exp $ *)

module HA=Hashtbl
module A=Absyn
module C=Checker

exception EXIT

(* the last bool indicates whether the var is initialized *)
type variable = 
  | VInt of (int32 * bool)
  | VBool of (bool * bool)
  | VStruct of ( string * (string * (variable ref)) list )
             (* type-name * (fieldname * var) list *)
  | VPointer of (int * int ref * (variable ref) array ref * A.ctype * bool)
               (* num, pos,  list,           type,   ini? *)
 
(* Variable environment : string -> (variable ref) *)
let id_var = ref (HA.create 57)

(* Function environment : store the functions/procedures *)
let fpid_fun = HA.create 57

(* The point where NULL points to *)
let null_location = ref [||]
(* The offset of NULL *)
let null_offset = ref 0

(* controlFlow *)
exception Continue
exception Break
exception Returned of variable option

(* The FFI call *)
external ffi_call : string -> int32 array -> int32 = "ffi_call" ;;

(* create a new variable, only pointers initialized *)
let rec new_var pos = function
  | A.TyBool -> VBool((Random.int 2) = 0, false)
  | A.TyInt -> VInt(Int32.of_int (Random.bits ()), false)
  | A.TyPointer(t) -> VPointer(0, null_offset, null_location, A.NS, true)
  | A.NS -> VPointer(0, null_offset, null_location, A.NS, true)
  | A.TyRecord(sName) -> 
      let t = C.getStructType sName in
      VStruct(sName, List.map (fun (s,t) -> 
	(s, ref (new_var pos t))) t)
  | A.Void -> Errormsg.compiler pos 44 "Invalid type (void)"; raise EXIT

(* returns the type of a variable (expression) *)
let etypeOfvar = function
  | VInt(_,_) -> A.TyInt
  | VBool(_,_) -> A.TyBool
  | VStruct(name, _) -> A.TyRecord(name)
  | VPointer(_,_,_,t,_) -> A.TyPointer(t)

(* check whether initialized *)
let ch_ini2 pos v1 v2 =
  if( v1 = false ) then
    Errormsg.warning pos ("First expression not initialized");
  if( v2 = false ) then
    Errormsg.warning pos ("Second expression not initialized")

let ch_ini pos v = 
  if( v = false ) then
    Errormsg.warning pos ("Expression not initialized")


(* evaluates an l-val or throws an error at the given pos *)
(* returns a ref to the expression *)
let rec eval_lval pos lval = 
  match lval with
  | A.Var(A.Ident(id, p), _) -> 
      (try
	HA.find !id_var id
      with
	Not_found -> 
	  Errormsg.compiler p 80 
	    ("Compiler error: Lost variable '" ^ id ^ "'");
	  raise EXIT
      )
  | A.Deref(e, p, _) -> 
      (match (eval_exp e) with
      | VPointer(num, off, arr, etyp, valid) -> 
	  if( not valid ) then begin
	    Errormsg.error p "Segmentation Fault - Not initialized";
	    raise EXIT
	  end;
	  if( (!off >= num || !off < 0) ) then begin
	    Errormsg.error p "Segmentation Fault - Index out of bounds";
	    raise EXIT
	  end;
	  (!arr).(!off)
      | v -> Errormsg.compiler p 96 "Type mismatch"; raise EXIT
      )
  | A.Field(lval, A.Ident(id,p1), p2, _) -> 
      (match (!(eval_lval p2 lval)) with
      | VStruct(name, li) ->
	  (try
	    let (s, var) = List.find 
		(fun (s,_) -> (Pervasives.compare s id) = 0) li in
	    var
	  with Not_found ->
	    Errormsg.compiler p1 106 ("Invalid field name '" ^ 
					      id ^ "'");
	    raise EXIT
	  )
      | v -> Errormsg.compiler p2 110 "Type mismatch"; raise EXIT
      )

and eval_exp = function
  | A.ConstExp(A.IntConst(num,_), _) -> VInt(num, true)
  | A.ConstExp(A.BoolConst(b,_), _) -> VBool(b, true)
  | A.ConstExp(A.NULL(_), _) -> VPointer(0, null_offset, null_location, A.NS, true)
  | A.OpExp(e1, op, Some(e2), pos, _) -> 
      (match op with
      | A.LOGICAND | A.LOGICOR -> 
	  (match (eval_exp e1, op) with
	  | (VBool(false, v1), A.LOGICAND) -> 
	      ch_ini pos v1; VBool(false, v1)
	  | (VBool(true, v1), A.LOGICAND) -> 
	      (match (eval_exp e2) with
	      | VBool(b2, v2) -> 
	      	  ch_ini2 pos v1 v2; VBool(b2, v1 && v2)
	      | _ -> Errormsg.compiler pos 148 "Type mismatch"; raise EXIT
	      )
	  | (VBool(true, v1), A.LOGICOR) -> 
	      ch_ini pos v1; VBool(true, v1)
	  | (VBool(false, v1), A.LOGICOR) -> 
	      (match (eval_exp e2) with
	      | VBool(b2, v2) -> 
	      	  ch_ini2 pos v1 v2; VBool(b2, v1 && v2)
	      | _ -> Errormsg.compiler pos 148 "Type mismatch"; raise EXIT
	      )
	  | _ -> Errormsg.compiler pos 148 "Type mismatch"; raise EXIT
	  )
      | _ -> 
	  (match (eval_exp e1, op, eval_exp e2) with
	  | (VInt(n1, v1), A.PLUS, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VInt(Int32.add n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.MINUS, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VInt(Int32.sub n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.TIMES, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VInt(Int32.mul n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.DIVIDE, VInt(n2, v2)) -> 
	      if( (Pervasives.compare n2 Int32.zero) = 0 ) then begin
		Errormsg.error pos "Division by zero"; raise EXIT
	      end;
	      ch_ini2 pos v1 v2; VInt(Int32.div n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.MOD, VInt(n2, v2)) -> 
	      if( (Pervasives.compare n2 Int32.zero) = 0 ) then begin
		Errormsg.error pos "Division by zero"; raise EXIT
	      end;
	      ch_ini2 pos v1 v2; VInt(Int32.rem n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.BITAND, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VInt(Int32.logand n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.BITOR, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VInt(Int32.logor n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.BITXOR, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VInt(Int32.logxor n1 n2, v1 && v2)
	  | (VInt(n1, v1), A.SHIFTLEFT, VInt(n2, v2)) -> 
	      if( (Pervasives.compare n2 Int32.zero) < 0 
		|| (Pervasives.compare n2 (Int32.of_int 31)) > 0) then
		Errormsg.warning pos ("Invalid second argument (" ^
		(Int32.to_string n2) ^ ")for shift");
	      ch_ini2 pos v1 v2; 
	      VInt(Int32.shift_left n1 (Int32.to_int n2), v1 && v2)
	  | (VInt(n1, v1), A.SHIFTRIGHT, VInt(n2, v2)) -> 
	      if( (Pervasives.compare n2 Int32.zero) < 0 
		|| (Pervasives.compare n2 (Int32.of_int 31)) > 0) then
		Errormsg.warning pos ("Invalid second argument (" ^
		(Int32.to_string n2) ^ ")for shift");
	      ch_ini2 pos v1 v2; 
	      VInt(Int32.shift_right n1 (Int32.to_int n2), v1 && v2)
	  | (VPointer(num,p,li,typ,v1), A.PPLUS, VInt(n, v2)) -> 
	      ch_ini2 pos v1 v2; 
	      if(li == null_location) then
		VPointer(0, null_offset, null_location, A.NS, true)
	      else
		VPointer(num, ref (!p + (Int32.to_int n)), li, typ, v1 && v2)
	  | (VPointer(num,p,li,typ,v1), A.PMINUS, VInt(n, v2)) -> 
	      ch_ini2 pos v1 v2; 
	      if(li == null_location) then
		VPointer(0, null_offset, null_location, A.NS, true)
	      else
		VPointer(num, ref (!p - (Int32.to_int n)), li, typ, v1 && v2)
	  | (VPointer(n1,p1,l1,t1,v1), A.PEQ, VPointer(n2,p2,l2,t2,v2)) -> 
	      ch_ini2 pos v1 v2; 
	      VBool((!l1 == !l2) && (!p1 = !p2), v1 && v2)
	  | (VPointer(n1,p1,l1,t1,v1), A.PNEQ, VPointer(n2,p2,l2,t2,v2)) -> 
	      ch_ini2 pos v1 v2; 
	      VBool((!l1 != !l2) || (!p1 <> !p2), v1 && v2)
	  | (VBool(b1, v1), A.LOGICXOR, VBool(b2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool(b1 && (not b2) || (not b1) && b2, v1 && v2)
	  | (VInt(n1, v1), A.EQ, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool((Pervasives.compare n1 n2) = 0, v1 && v2)
	  | (VInt(n1, v1), A.NEQ, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool((Pervasives.compare n1 n2) <> 0, v1 && v2)
	  | (VInt(n1, v1), A.LT, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool((Pervasives.compare n1 n2) < 0, v1 && v2)
	  | (VInt(n1, v1), A.LTE, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool((Pervasives.compare n1 n2) <= 0, v1 && v2)
	  | (VInt(n1, v1), A.GT, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool((Pervasives.compare n1 n2) > 0, v1 && v2)
	  | (VInt(n1, v1), A.GTE, VInt(n2, v2)) -> 
	      ch_ini2 pos v1 v2; VBool((Pervasives.compare n1 n2) >= 0, v1 && v2)
	  | _ -> 
	      Errormsg.compiler pos 179 "Type mismatch"; raise EXIT
	  )
      )
  | A.OpExp(e, op, None, pos, _) -> 
      (match (eval_exp e, op) with
      | (VBool(b,v), A.LOGICNOT) -> ch_ini pos v; VBool(not b, v)
      | (VInt(n,v), A.UMINUS) -> ch_ini pos v; VInt(Int32.neg n, v)
      | (VInt(n,v), A.BITNOT) -> ch_ini pos v; VInt(Int32.lognot n, v)
      | _ -> Errormsg.compiler pos 185 "Type mismatch"; raise EXIT
       )
  | A.Alloc(e, p1, typ, p2, _) -> 
      (match (eval_exp e) with
      | VInt(n,v) -> 
	  if((Pervasives.compare n Int32.zero) < 0) then begin
	    Errormsg.error p1 "Invalid argument to _alloc()"; raise EXIT
	  end;
	  ch_ini p1 v; 
	  let etyp = C.typ2ctype typ in
	  let size = Int32.to_int n in 
	  let a = ref (Array.init size (fun x -> ref (new_var p1 etyp))) in
	  VPointer(size, ref 0, a, etyp, true)
      | _ -> Errormsg.compiler p1 195 "Type mismatch"; raise EXIT
      )
  | A.StackAlloc(e, p1, typ, p2, _) -> 
      (match (eval_exp e) with
      | VInt(n,v) -> 
	  if((Pervasives.compare n Int32.zero) < 0) then begin
	    Errormsg.error p1 "Invalid argument to _stack_alloc()"; raise EXIT
	  end;
	  ch_ini p1 v; 
	  let etyp = C.typ2ctype typ in
	  let size = Int32.to_int n in 
	  let a = ref (Array.init size (fun x -> ref (new_var p1 etyp))) in
	  VPointer(size, ref 0, a, etyp, true)
      | _ -> Errormsg.compiler p1 195 "Type mismatch"; raise EXIT
      )
  | A.Offset(exp, pos, _) -> 
      (match (eval_exp exp) with
      | VPointer(_, offset, _, _, v) -> 
	  ch_ini pos v;
	  VInt(Int32.of_int !offset, v)
      | _ -> Errormsg.compiler pos 202 "Type mismatch"; raise EXIT
      )
  | A.Size(exp, pos, _) -> 
      (match (eval_exp exp) with
      | VPointer(size, _, _, _, v) -> 
	  ch_ini pos v;
	  VInt(Int32.of_int size, v)
      | _ -> Errormsg.compiler pos 209 "Type mismatch"; raise EXIT
      )
(*
  | A.Inc(exp, pos, _) -> 
      (match (eval_exp exp) with
      | VPointer(size, off, list, ty, v) -> 
	  ch_ini pos v;
	  off := !off + 1;
	  VPointer(size, off, list, ty, v)
      | _ -> Errormsg.compiler pos 210 "Type mismatch"; raise EXIT
      )
  | A.Dec(exp, pos, _) -> 
      (match (eval_exp exp) with
      | VPointer(size, off, list, ty, v) -> 
	  ch_ini pos v;
	  off := !off - 1;
	  VPointer(size, off, list, ty, v)
      | _ -> Errormsg.compiler pos 210 "Type mismatch"; raise EXIT
      )
*)
  | A.Reference(lval, pos, _) -> 
      (match lval with
      | A.Var(A.Ident(id, p), _) -> 
	  let var = 
	    (try
	      HA.find !id_var id
	    with
	      Not_found -> 
		Errormsg.compiler p 219 ("Lost variable '" ^ id ^ "'");
		raise EXIT
	    )
	  in
	  let typ = etypeOfvar !var in 
	  VPointer(1, ref 0, ref [| var |], typ, true)
      | A.Deref(e, p, _) -> 
	  let old = eval_exp e in 
	  (* return new fat pointer *)
	  (match old with 
	  | VPointer(num, off, list, typ, i) -> 
	      VPointer(num, ref !off, list, typ,i)
	  | _ -> 
	      Errormsg.compiler p 219 ("Invalid lval in Reference(Deref())");
	      raise EXIT
	  )
      | A.Field(lval, A.Ident(id,p1), p2, _) -> 
	  (match (!(eval_lval p2 lval)) with
	  | VStruct(name, li) ->
	      (try
		let (s, var) = List.find 
		    (fun (s,_) -> (Pervasives.compare s id) = 0) li in
		let typ = etypeOfvar !var in
		VPointer(1, ref 0, ref [| var |], typ, true)
	      with Not_found -> 
		Errormsg.compiler p1 237 ("Invalid field name '" ^ id ^ "'");
		raise EXIT
	      )
	  | _ -> Errormsg.compiler p2 240 "Type mismatch"; raise EXIT
	  )
      )
  | A.Call(A.Ident(name,p),li,_,_) -> 
      (match (evalFunction name li) with
      | Some(exp) -> exp
      | None -> 
	  Errormsg.compiler p 278 "Proc call instead of fun call";
	  raise EXIT
      )
  | A.LVal(l) -> !(eval_lval (0,0) l)

and eval_stmt = function
  | A.Assign(e1, p1, e2, p2) -> 
      let eval2 = eval_exp e2 in
      let lval = eval_lval p1 e1 in
      lval := eval2
  | A.IfElse(exp, p1, li1, li2, p2) -> 
      begin
	match eval_exp exp with
	| VBool(true, _) -> 
	    List.iter (fun s -> eval_stmt s) li1;
	| VBool(false, _) -> 
	    List.iter (fun s -> eval_stmt s) li2;
	| _ -> Errormsg.compiler p1 298 "Type mismatch"; raise EXIT
      end
  | A.If(exp, p1, li, p2) -> 
      begin
	match eval_exp exp with
	| VBool(true, _) -> List.iter (fun s -> eval_stmt s) li
	| VBool(false, _) -> ()
	| _ -> Errormsg.compiler p1 305 "Type mismatch"; raise EXIT
      end
  | A.While(exp, p, li) -> 
      eval_stmt (A.For(A.Blank, p, exp, p, A.Blank, p, li))
  | A.For(s1,_,exp,_,s2,_,li) -> 
      eval_stmt s1;
      let condition = ref false in
      let get_condition exp = 
	let tmp = eval_exp exp in
	match tmp with
	| VBool(true,_) -> condition := true;
	| _ -> condition := false
      in
      get_condition exp;
      while !condition do
	(try
	  List.iter (fun s -> eval_stmt s) li;
	  eval_stmt s2;
	  get_condition exp;
	with 
	| Continue -> 
	    (try
	      eval_stmt s2;
	    with 
	    | Break -> condition := false;
	    | Continue -> ()
	    );
	    get_condition exp
	| Break -> 
	    condition := false;
	)
      done
  | A.StmtExp(A.Call(A.Ident(name,_), li,_,_),_) -> 
      let _ = evalFunction name li in ()
  | A.StmtExp(exp,p) -> 
      let _ = eval_exp exp in
      ()
  | A.Continue(p) -> 
      raise Continue
  | A.Break(p) -> 
      raise Break
  | A.Return(Some(exp), pos) -> 
      raise (Returned (Some(eval_exp exp)))
  | A.Return(None, pos) -> 
      raise (Returned None)
  | A.Blank -> ()

(* Evaluates a function or procedure and returns the 
   result (Some(variable) | None) *)
and evalFunction name args = 
  let new_id_var = HA.create 57 in (* create new scope *)
  let code = HA.find fpid_fun name in
  let (rType, argList) = C.getFunType name in

  let body = 
    match code with
    | A.FunDecl(_,_,_,_,b) -> b
    | A.StructDecl(_)
    | A.VarDecl(_) -> 
	Errormsg.compiler (0,0) 356 "Not a fun/proc"; raise EXIT
  in

  match body with
  | A.Body(dl, sl) -> 
      List.iter2 (fun (exp, pos) (s, ct) -> 
	HA.add new_id_var s (ref (eval_exp exp))) args argList;

      let tmp_id_var = !id_var in (* enter new scope *)
      id_var := new_id_var;
      (* add declared variables *)
      List.iter (fun s -> 
	(match s with
	| A.VarDecl(A.Ident(id,vp), typ, _) -> 
	    HA.add !id_var id (ref (new_var vp (C.typ2ctype typ)))
	|_ -> Errormsg.compiler (0,0) 370 "Wrong vardecl"; raise EXIT
	)) dl;
      
      let result =  
	(try
	  List.iter (fun s -> eval_stmt s) sl;
	  None
	with 
	  Returned(x) -> 
	    x
	) in
      id_var := tmp_id_var; (* exit scope *)
      result

  | A.Foreign(p) ->   
      let arglist = List.map (fun (e,p) -> 
	match eval_exp e with
	| VInt(n,v) -> 
	    ch_ini p v;
	    n
	| VBool(b,v) -> 
	    ch_ini p v;
	    if(b) then
	      Int32.one 
	    else
	      Int32.zero
	| _ -> Errormsg.error p 
	      ("Invalid argument type to foreign function' " ^ name ^ 
	       "' - only int and bool are allowed");
	    raise EXIT
			     ) args
      in
      let res = ffi_call name (Array.of_list arglist) in
      (match (rType,res) with
      | (A.TyInt,n) -> 
	  Some(VInt(n, true))
      | (A.TyBool, n) -> 
	  if( (Int32.compare n Int32.zero) = 0 ) then
	    Some(VBool(false, true))
	  else
    	    Some(VBool(true, true))
      | (A.Void,_) -> None
      | _ -> Errormsg.error p 
	    ("Invalid return type for foreign function '" ^ name ^ 
	     "' - only int and bool are allowed");
	  raise EXIT
      )


(* Read all functions and store them in fpid_fun *)
let readTopLevel tl = 
  match tl with
  | A.StructDecl(_) -> ()
  | A.VarDecl(_) -> ()
  | A.FunDecl(A.Ident(id,_),_,_,_,_) -> HA.add fpid_fun id tl

let evalTopLevel () = 
  let startFun = "main" in
  let startArgs = [] in
  let (rType, protoArgs) = 
    (try 
      C.getFunType startFun
    with C.CheckerError(s) -> 
      Errormsg.error (0,0) ("Link error: Function '" ^ startFun ^ "' not found"); 
      raise EXIT;
    )
  in
  if( (List.length startArgs) <> (List.length protoArgs) ) then begin
    Errormsg.error (0,0) ("Link error: Wrong number of arguments for '" ^ 
			  startFun ^ "'");
    raise EXIT
  end;
  (* call main() *)
  let result = evalFunction startFun startArgs in
  match result with
  | None -> ()
  | Some(r) ->  ()
	(*
      (match r with
      | VInt(i,_) -> 
	  print_string "Result = ";  
	  print_string (Int32.to_string i);
	  print_string "\n" 
      | VBool(true,_) -> print_string "Result = true\n"
      | VBool(false,_) -> print_string "Result = false\n"
      | _ -> print_string "Invalid return value\n";
      )
	*)
(* Evaluate a program *)
let eval_program (A.Program(pl)) = 
  Random.self_init ();
  List.iter (fun s -> readTopLevel s) pl;
  evalTopLevel ()








    

  
    
