(* errormsg.ml *)
(* 15-411 *)
(* by Benjamin Vernot, Peter Lee, Roland Flury *)
(* @version $Id: errormsg.ml,v 1.5 2003/09/26 16:54:48 rflury Exp $ *)

exception EXIT

type pos = (int * int)

(* debug printing *)
let debug = ref false

let warningFlag = ref false

let showProgressFlag = ref false

let optimizationLevel = ref 4 

(* Print a message to show the progress in the compilation process *)
let showProgress s = 
  if(!showProgressFlag) then (
    Printf.printf "%s" s; 
    flush stdout
   )
      
let print_debug s = if(!debug) then (
  Printf.printf "%s" s;
  Pervasives.flush stdout
 )


let lineNum = ref 1
let linePos = ref [1]

type linePosT =
  | Pos of int
  | PosNewFile of int * int * string

type info =
    { mutable  linenum: int      ;
      mutable  linepos: linePosT list ;
      mutable  fileName: string  ;
      mutable  errors: bool      ; }
      
let current : info = 
  { linenum  = 1   ;
    linepos  = [Pos(1)] ;
    fileName = ""  ;
    errors = false ; }
    
(* Initialize current-struct mainly with file-name *)
let startFile fname =
  current.linenum  <- 1     ;
  current.linepos  <- [Pos(0)]   ;
  current.fileName <- fname ;
  current.errors   <- false
      
(* Called for each newline *)
let startNewline n =
  current.linenum <- current.linenum + 1;
  current.linepos <- current.linepos @ [Pos(n)]
			   
(* Called when a new file is scanned *)
let startNewFile n fname startLine = 
  current.linenum <- current.linenum + 1;
  current.linepos <- current.linepos @ [PosNewFile(n, startLine, fname)]

(* Creates a string <filename>:<line:offset-line:offset> *)
let getLocation (i : info) pos pos2 = 
  let rec look line prev = function
    | [Pos(a)] -> (line+1, pos - a -1, pos2 -a -1)
    | Pos(a) :: _ when (pos<a) -> (line, pos -prev -1, pos2 -prev -1)
    | PosNewFile(a, l, n)::tail-> i.fileName <- n; look l a tail
    | Pos(a)::tail -> look (line+1) a tail
    | _ -> (0, 0, 0)
  in
  let (lin,col,col2) = look 0 0 i.linepos in
  let preloc =
    if pos2 <= 0 then
      "-" 
    else
      if col=col2 then 
	Printf.sprintf "%d.%d" lin (col)
      else 
	Printf.sprintf "%d.%d-%d.%d" lin (col) lin (col2)
  in
  Printf.sprintf "%s:%s" i.fileName preloc

(* Retrieve file-name and line-number of a position *)
let pos2info (pos, pos2) = 
  let i = current in
  let rec look line prev = function
    | [Pos(a)] -> (line+1, pos - a -1, pos2 -a -1)
    | Pos(a) :: _ when (pos<a) -> (line, pos -prev -1, pos2 -prev -1)
    | PosNewFile(a, l, n)::tail-> i.fileName <- n; look l a tail
    | Pos(a)::tail -> look (line+1) a tail
    | _ -> (0, 0, 0)
  in
  let (lin,_,_) = look 0 0 i.linepos in  
  (i.fileName, lin)

(* Error msg to user *)
let error (startpos, endpos) msg =
  current.errors <- true;
  Printf.eprintf "%s:error: %s\n" (getLocation current startpos endpos) msg

let warning (startpos, endpos) msg =
  if(!warningFlag) then
    Printf.eprintf "%s:warning: %s\n" (getLocation current startpos endpos) msg

let compiler (sp, ep) line msg =
  Printf.eprintf "%s:Compiler error (%d): %s\n" 
	(getLocation current sp ep) line msg
