module TempSet = Set.Make(struct
                           type t = Temp.temp
                           let compare = compare
                         end)

module TempMap = Map.Make (struct
                             type t = Temp.temp
                             let compare = compare
                           end)

module InstrSet = Set.Make(struct
                             type t = CFG.node * int
                             let compare = compare
                            end)
  
module InstrMap = Map.Make(struct
                             type t = CFG.node * int
                             let compare = compare
                           end)

let (++) = TempSet.union

type definition = {
  label: CFG.node;
  instr: int; (* Instruction # in its block, 0-indexed including label  *)
  assign_var: Temp.temp;
  assign_exp: Ir.exp;
}

let print_def out def =
  let print fmt = Format.fprintf out fmt in
    print "(%s:%d) t%d := %s\n"
      (CFG.string_of_node def.label)
      def.instr
      def.assign_var
      (Ir.ir_pexp2str def.assign_exp)      

module DefinitionSet = Set.Make (struct
                                   type t = definition
                                   let compare = compare
                                 end)

(* index a set of reaching definitions by the temp that they assign -wjl *)
let temp_map_of_def_set set =
    let get map v =
      try
        TempMap.find v map
      with
      | Not_found -> DefinitionSet.empty
    in
    let var def = def.assign_var in
  DefinitionSet.fold
    (fun def map ->
       let set = DefinitionSet.add def (get map (var def)) in
         TempMap.add (var def) set map)
    set
    TempMap.empty

let rec freetemps exp =
  match exp with
  | Ir.TEMP l -> TempSet.singleton l
  | Ir.MEM e -> freetemps e
  | Ir.BINOP(op, e1, e2) -> TempSet.union (freetemps e1) (freetemps e2)
  | Ir.CALL(f, args) ->
      List.fold_right
         (fun e set -> TempSet.union (freetemps e) set)
         args
         TempSet.empty
  | Ir.CONST _ | Ir.NAME _ -> TempSet.empty
  | Ir.ESEQ(_, _) | Ir.PHI _ -> assert false (* not implemented *)

let stmt_freetemps stmt =
    match stmt with
    | Ir.MOVE (Ir.TEMP _, e2) -> freetemps e2
    | Ir.MOVE (e1, e2) -> freetemps e1 ++ freetemps e2
    | Ir.CJUMP (_, e1, e2, _, _) -> freetemps e1 ++ freetemps e2
    | Ir.EXP e -> freetemps e
    | Ir.LABEL _
    | Ir.JUMP _
    | Ir.COMMENT (_, _)
    | Ir.INVARIANT _ -> TempSet.empty
    | Ir.SEQ (_, _) -> assert false

let stmt_alltemps stmt =
    match stmt with
    | Ir.MOVE (Ir.TEMP t, e2) -> TempSet.singleton t ++ freetemps e2
    | _ -> stmt_freetemps stmt

let block_alltemps instrs =
    List.fold_left
        (fun set stmt -> set ++ stmt_alltemps stmt)
        TempSet.empty
        instrs

(* NB: no nested expressions at this point, so no recursion -wjl *)
let pure exp =
    match exp with
    | Ir.CONST (_, _)
    | Ir.NAME _
    | Ir.TEMP _ -> true
    | Ir.MEM _ -> false (* ??? not sure -- is a load an effect? -wjl *)
                        (* seems so -- test case 123 fails otherwise *)
    | Ir.BINOP (op, _, e2) ->
        (match op with
        | Ir.DIV | Ir.MOD ->
            (match e2 with
            | Ir.CONST (n, _) -> n <> 0l (* division by non-zero const: pure *)
            | _ -> false (* division by anything else: not so sure *))
        | _ -> true (* non-division == Server OK *))
        (* false (* seems it's not safe to eliminate these? *) *)
    | Ir.CALL (_, _) -> false
    | Ir.ESEQ (_, _)
    | Ir.PHI _ -> assert false

let impure exp = not (pure exp)

(* list functions *)

let mapi f list =
  let rec loop i list = 
    match list with
    | [] -> []
    | x :: xs -> (f i x) :: (loop (i+1) xs)
  in
    loop 0 list

let iteri f list =
    let rec loop i list =
        match list with
        | [] -> ()
        | x :: xs -> let () = f i x in loop (i+1) xs
    in
    loop 0 list

let fold_righti f list base =
    let rec loop i list =
        match list with
        | [] -> base
        | x :: xs -> f i x (loop (i+1) xs)
    in
    loop 0 list

let filteri p list =
    fold_righti (fun i x xs -> if p i x then x :: xs else xs) list []

