module NodeMap = CFG.NodeMap
module NodeSet = CFG.NodeSet

let (%%) map (key, value) = NodeMap.add key value map 
let (<=) key value = (key, value)
let (@@) map key = NodeMap.find key map

let one = NodeSet.singleton
let (++) set elt = NodeSet.add elt set
let (--) set set' = NodeSet.diff set set'
let empty = NodeSet.empty

  
	      
let dominators cfg =
  let entry = CFG.entry cfg in
  let pred node = List.fold_right NodeSet.add (CFG.pred cfg node) empty in
  let nodes = CFG.nodes cfg in
  let nodes' = nodes -- (one entry) in
  let dom = NodeMap.empty %% (entry <= (one entry)) in
  let dom = NodeSet.fold (fun node dom -> dom %% (node <= nodes)) nodes' dom in
  let rec loop dom = 
      let update n (changed, dom) =
         (* t = inter_{p in pred(n)} dom(p) *)
         let t =
           NodeSet.fold
             (fun p t -> NodeSet.inter t (dom @@ p))
             (pred n)
             nodes
         in
         (* d = t union {n} *)
         let d = t ++ n in
         let changed = changed || not (NodeSet.equal d (dom @@ n)) in
         let dom = dom %% (n <= d) in
           (changed, dom)
      in
      let (changed, dom) = NodeSet.fold update nodes' (false, dom)
      in
        if changed then
            loop dom
        else
            dom
  in
  (* remove nodes "dominated" by the exit -- these represent dead code *)
  let cleanup dom =
    NodeMap.mapi
      (fun node dominators ->
	 let exit = CFG.exit cfg in
	   if node <> exit && NodeSet.mem exit dominators then
	     NodeSet.empty
	   else
	     dominators)
      dom
  in
    cleanup (loop dom)

let dom d a b = NodeSet.mem a (d @@ b)

let sdom d a b = (a <> b) && (dom d a b)

let idom d a b =
  not (NodeSet.exists (fun c -> (sdom d a c) && (sdom d c b)) (d @@ b))

let backedges d cfg =
  NodeSet.fold
    (fun start backedges ->
       List.fold_right
       (fun finish backedges ->
	  if dom d finish start then
	    (start, finish) :: backedges
	  else
	    backedges)
       (CFG.succ cfg start)
       backedges)
    (CFG.nodes cfg)
    []

let set_of_list list =
  List.fold_right NodeSet.add list NodeSet.empty

let has_path cfg a b exclude =
  let succ x =
    set_of_list
      (List.filter (fun n -> not (NodeSet.mem n exclude)) (CFG.succ cfg x))
  in
  let rec loop reachable visited =
    if NodeSet.is_empty reachable then
      false
    else
      let node = NodeSet.choose reachable in
	if node = b then
	  true
	else
	  let reachable = NodeSet.remove node reachable in
	  let successors = (succ node) -- visited in
	    loop (NodeSet.union successors reachable) (visited ++ node)
  in
    loop (one a) empty
      
let natural d cfg (n, h) =
  NodeSet.filter
    (fun x -> (dom d h x) && (has_path cfg x n (one h)))
    (CFG.nodes cfg)  

let natural_loops d cfg =
  List.map (fun edge -> (edge, natural d cfg edge)) (backedges d cfg)
    
let print_nodeset out set =
  let print fmt = Format.fprintf out fmt in
  let string = CFG.string_of_node in
    begin
      print "{ ";
      NodeSet.iter (fun node -> print "%s " (string node)) set;
      print "}";
    end

let print_nodemap print_elt out map =
  let print fmt = Format.fprintf out fmt in
  let string = CFG.string_of_node in
    NodeMap.iter
      (fun node elt ->
	 begin
	   print "%s --> " (string node);
	   print_elt out elt;
	   print "\n";
	 end)
      map

let print_dominators = print_nodemap print_nodeset

let print_loop out ((n, h), nodes) = 
  let print fmt = Format.fprintf out fmt in
  let string = CFG.string_of_node in
    begin
      print "(%s -> %s) : " (string n) (string h);
      print_nodeset out nodes;
      print "\n";
    end

let print_loops out list =
  let print fmt = Format.fprintf out fmt in
  let string = CFG.string_of_node in
    List.iter (print_loop out) list
