open Sigs

module IDA (A : ANALYSIS) =
struct
  let (%%) map (key, value) = CFG.NodeMap.add key value map
  let (<=) key value = (key, value)
  let (@@) map key = CFG.NodeMap.find key map

  let build_worklist entry next =
    let marked_set = ref CFG.NodeSet.empty in
    let mark node = marked_set := CFG.NodeSet.add node (!marked_set) in
    let marked node = CFG.NodeSet.mem node (!marked_set) in
    (* careful: building two queues: "queue" controls the breadth-first
       search, while "worklist" is the one we intend to return *)
    let worklist = Queue.create () in
    let queue = Queue.create () in
    Queue.add entry queue;
      (* invariant: if n \in queue, then (marked n) *)
      while not (Queue.is_empty queue) do
          let node = Queue.take queue in
          Queue.add node worklist;
          List.iter
              (fun n -> if not (marked n) then (mark n; Queue.add n queue))
              (next node)
      done;
    worklist

    (* (* omg, imperative code is more readable@!!1 *)
    let rec loop visited =
        if Queue.empty queue then
            ()
        else
            let node = Queue.take q in
            Queue.add node worklist;
            let visited = List.fold_left (fun set n ->
                                            if NodeSet.mem n visited then
                                                visited
                                            else
                                                (Queue.add n queue;
                                                 NodeSet.add n set))
                                         visited
                                         (next node)
            in
            loop visited
    *)

  let analyze cfg =
    let module Flow = A.MakeFlow (struct let cfg = cfg end) in
    let module Lattice = Flow.Lattice in
    let (++) = Flow.combine in
    let (--) = Lattice.diff in
    let init_map = CFG.initial_nodemap cfg Flow.initial in
    let inflow = init_map in
    let outflow = init_map in
    let w = build_worklist Flow.entry Flow.next in
    let rec loop inflow outflow =
      match try Some (Queue.take w) with Queue.Empty -> None with
        | None -> (inflow, outflow) (* which one...? *) 
        | Some b ->
            let ps = List.map (fun p -> outflow @@ p) (Flow.prev b) in
            let inflow = inflow %% (b <= Flow.combine_list ps) in
            let old_b_out = outflow @@ b in
            let new_b_out = (Flow.gen b) ++ ((inflow @@ b) -- (Flow.kill b)) in
            let outflow = outflow %% (b <= new_b_out)
            in
              if Lattice.equal old_b_out new_b_out then
                loop inflow outflow
              else
                let () = List.iter (fun p -> Queue.add p w) (Flow.next b) in
                loop inflow outflow
    in
    let (inflow, outflow) = loop inflow outflow in
    Flow.interpret'
        (CFG.NodeMap.map Flow.interpret inflow,
         CFG.NodeMap.map Flow.interpret outflow)
end
