(* Constant folding:

   Take a triple-ized list of instructions, and then eliminate constant
   ops.

*)

let opify op =
  match op with
  | Ir.PLUS -> Int32.add
  | Ir.MINUS -> Int32.sub
  | Ir.DIV -> Int32.div
  | Ir.MUL -> Int32.mul
  | Ir.MOD -> Int32.rem
  | Ir.AND -> Int32.logand
  | Ir.OR  -> Int32.logor
  | Ir.XOR -> Int32.logxor
  | Ir.SHIFTL -> (fun x y -> Int32.shift_left x (Int32.to_int y))
  | Ir.SHIFTR -> (fun x y -> Int32.shift_right x (Int32.to_int y))
  | Ir.NOT -> (fun x y -> Int32.lognot x)

let rec cfold_exp exp =
  match exp with 
  | Ir.CONST(_, _) -> exp
  | Ir.BINOP(op, e1, e2) ->
      let c1 = cfold_exp e1 in
      let c2 = cfold_exp e2 in
	(match c1, op, c2 with
         (* leave division by 0 alone *)
         | _,               Ir.DIV, Ir.CONST (0l, _)
         | _,               Ir.MOD, Ir.CONST (0l, _) -> exp
	 | Ir.CONST(n1, _), op,     Ir.CONST(n2, _) -> Ir.CONST(opify op n1 n2,
                                                                Pcc.Bogus)
	 | c1,              op,     c2 -> Ir.BINOP(op, c1, c2))
  | _ -> exp
      

let relopify op n1 n2 =
  let c = Int32.compare n1 n2 in
    match op with
    | Ir.EQ -> c = 0
    | Ir.NE -> c <> 0
    | Ir.LT -> c < 0
    | Ir.GT -> c > 0
    | Ir.LE -> c <= 0
    | Ir.GE -> c >= 0 
      
let cfold_stmt stmt =
  match stmt with
  | Ir.MOVE(e1, e2) -> 
      let e1 = cfold_exp e1 in
      let e2 = cfold_exp e2 in
	Ir.MOVE(e1, e2)
  | Ir.EXP e -> Ir.EXP (cfold_exp e)
  | Ir.CJUMP(relop, e1, e2, lt, lf) ->
      let c1 = cfold_exp e1 in
      let c2 = cfold_exp e2 in
	(match c1, c2 with
	 | Ir.CONST(n1, _), Ir.CONST(n2, _) ->
	     if relopify relop n1 n2 then
	       Ir.JUMP lt
	     else
	       Ir.JUMP lf
	 | c1, c2 -> Ir.CJUMP(relop, c1, c2, lt, lf))
  | Ir.COMMENT _ 
  | Ir.LABEL _
  | Ir.JUMP _ -> stmt
  | _ -> assert false (* Nonlinear statements *)
	
let cfold_block stmts = List.map cfold_stmt stmts

let constant_fold stmts_list = List.map cfold_block stmts_list
