type t = int * string

exception Bitset

let bytes n = (n / 8) + if 0 = n mod 8 then 0 else 1

let init c n =
  let b = bytes n in
    (n, String.make b c)

let top n = init (Char.chr 255) n

let bot n = init (Char.chr 0) n

let make n f =
  let len = bytes n in
  let s = String.make len (Char.chr 0) in
  begin
    for i = 0 to len - 1 do
      let c = ref 0 in
      for b = 0 to 7 do
	let j = b + 8 * i in
	if j < n && f j then c := !c lor (1 lsl b)
      done;
      s.[i] <- Char.chr !c;
    done;
    (n, s)
  end

let get (n,s) i =
  if i >= 0 && i < n then
    0 != Char.code s.[i/8] land (1 lsl (i mod 8))
  else
    raise Bitset

let size (n, s) = n

let fold f init set =
  let len = size set in
  let rec loop i acc =
    if i = len then
      acc
    else
      loop (i+1) (f i (get set i) acc)
  in
    loop 0 init

let print stream set =
  begin
    Format.fprintf stream "%d:" (size set);
    Format.fprintf stream "{";
    fold
      (fun _ b () ->
	 if b then Format.fprintf stream "1" else Format.fprintf stream "0")
      ()
      set;
    Format.fprintf stream "}";
  end

let cand c c' = Char.chr (Char.code c land Char.code c')
let cor c c' = Char.chr (Char.code c lor Char.code c')
let cxor c c' = Char.chr (Char.code c lxor Char.code c')
let cnot c = Char.chr (lnot (Char.code c) land 255)

let copy (n, s) = (n, String.copy s)

let binop' cop ((n, s) : t) (n', s') =
  if n = n' then
    for i = 0 to (String.length s) - 1 do
      try
	s.[i] <- cop s.[i] s'.[i]
      with
	e ->
	  (print_int n;
	   print_string " {";
	   print_string (String.escaped s);
	   print_string "}\n";
	   print_int n';
	   print_string " {";
	   print_string (String.escaped s');
	   print_string "}\n";
	   raise e)
    done 
  else
    raise Bitset

let unop' cop (n, s) =
  for i = 0 to (String.length s) - 1 do
    s.[i] <- cop s.[i]
  done

let intersect' = binop' cand 
let union' = binop' cor
let complement' = unop' cnot

let intersect set set' = let set = copy set in (intersect' set set'; set)
let union set set' = let set = copy set in (union' set set'; set)
let complement set = let set = copy set in (complement' set; set)

let oplist op' set list =
  let n = size set in
    if List.for_all (fun set -> n = size set) list then
      let init = copy set in
	(List.iter (op' init) list;
	 init)
    else
      raise Bitset

let intersect_list n list =
  match list with
  | [] -> top n
  | set :: sets -> oplist intersect' set sets

let union_list n list = 
  match list with
  | [] -> bot n
  | set :: sets -> oplist union' set sets

let equal set set' =
  if size set = size set' then
    fold (fun i b acc -> acc && (b = get set' i)) true set
  else
    false


module BV(Size : sig val n : int end) =
struct
  type dummy = t
  type t = dummy
      
  let top = top Size.n

  let bot = bot Size.n

  let meet = intersect 

  let join = union 

  let meet_list = intersect_list Size.n

  let join_list = union_list Size.n

  let diff s1 s2 = meet s1 (complement s2)

  let equal = equal

  let leq s s' = equal s (meet s s')
end

		    
