(* bitlist.mli *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: bitlist.mli,v 1.2 2003/08/11 13:04:42 rflury Exp $ *)

(* description:
   Implements a module that provieds a bitList and some functions.
   BitLists are mutable objects, i.e. the object itself does not get
   copied all the time, but is just like an object pointed by a pointer. 
   
   Available functions
   -------------------
   - Create a new Bit_list with at least n entries
   - set a bit to 1 (bit specified with the place in the bit-list)
   - set a bit to 0 (bit specified with the place in the bit-list)
   - Read content of bit (bit specified with the place in the bit-list)
   - AND / OR / NOT / XOR two BitArrays;
*)

type bitList_t = int array ref

(* Creates a new BitList and returns a pointer to it. (is a reference)
 * and initilizes all entries to 0. *)
val newBitList : int -> bitList_t
    
(* set the bit number n to true == 1, first element has number 0 
 * bli: int Array ref      n: int
 * Raises Invalid_argument if n is out of range *)
val setBit : bitList_t -> int -> unit
    
(* set the bit number n to false == 0, first element has number 0 
 * bli: int Array ref        n: int
 * Raises Invalid_argument if n is out of range *)
val resetBit : bitList_t -> int -> unit

(* reads the bit number n and returns either true (for 1) or false (for 0) 
 * bli: int Array ref        n: int
 * Raises Invalid_argument if n is out of range *)
val getBit : bitList_t -> int -> bool

(* returns the maximum number of bits storable in a bitList
 * => is at least as big as indicated in the constructor newBitList *)
val getLength : bitList_t -> int

(* sets all entries to 0 *)
val nullBitList : bitList_t -> unit

(* copies a BitList to a second BitList 
 * Raises Invalid_argument if length(toL) < length(fromL) *)
val copyBitList : bitList_t -> bitList_t -> unit

(* makes a copy of a BitList and returns a new BitList *)
val copyBitListNew : bitList_t -> bitList_t

(* Returns a list of temps that are set in the bit list *)
val getTempList : bitList_t -> (int -> 'a) -> 'a list

val isEqual : bitList_t -> bitList_t -> bool

(*-----------------------------------------------------------------------*)
(* AND OR XOR NOT creating a new bitList *)

val andBLnew : bitList_t -> bitList_t -> bitList_t
val orBLnew : bitList_t -> bitList_t -> bitList_t 
val xorBLnew : bitList_t -> bitList_t -> bitList_t
val notBLnew : bitList_t -> bitList_t

(*-----------------------------------------------------------------------*)
(* AND OR XOR NOT writing the result in the second operand *)

val andBLto2 : bitList_t -> bitList_t -> unit
val orBLto2 : bitList_t -> bitList_t -> unit 
val xorBLto2 : bitList_t -> bitList_t -> unit
val notBLto1 : bitList_t -> unit

(*-----------------------------------------------------------------------*)
(* sped up functions for or-ing *)

(* orBLtoR_ABCalter R A B C calculates R = A or (B and (not C))
 * Raises Invalid_argument if the lengths of R, A, B, C are not appropriate
 * returns true if R was changed, else false *)
val orBLtoR_ABCalter : bitList_t -> bitList_t -> 
  bitList_t -> bitList_t -> bool

(* or's the two bitLists 'a' and 'b' and writes the result in 'b'.
 * a, b: int Array ref
 * Raises Invalid_argument if length(a) < length(b)
 * returns true if b was changed, else false  *)
val orBLto2alter : bitList_t -> bitList_t -> bool


(* Iterate over set bits in a bitlist *)
val iterTrue : (int -> unit) -> bitList_t -> unit

(* Iterate over all bits in a bitlist *)
val iter : (int -> bool -> unit) -> bitList_t -> unit

(* Find the first set bit, returns -1 if none was found *)
val findSet : bitList_t -> int

(* Executes the given function for every offset that is set to true *)
val fold : ('a -> int -> 'a) -> 'a -> bitList_t -> 'a
