(* absyn.ml *)
(* 15-411 *)
(* by Roland Flury *)
(* @version $Id: absyn.ml,v 1.4 2003/09/17 18:54:06 rflury Exp $ *)

module E=Errormsg

type ident = Ident of string * E.pos

(* Type representation, added in the checker *)
type ctype = 
  | TyInt
  | TyBool
  | TyRecord of string
  | TyPointer of ctype
  | NS (* Nonsens, for Null *)
  | Void (* Void *)

type program = Program of decl list

and decl = 
  | FunDecl of ident * typ * E.pos * (ident * typ * E.pos) list * body
  | StructDecl of ident * (ident * typ * E.pos) list
  | VarDecl of ident * typ * E.pos

and body = 
  | Body of decl list * stmt list
  | Foreign of E.pos
	
and stmt =
  | Assign of (lval * E.pos * exp * E.pos)
  | IfElse of (exp * E.pos * stmt list * stmt list * E.pos)
  | If of (exp * E.pos * stmt list * E.pos) 
  | For of (stmt * E.pos * exp * E.pos * stmt * E.pos * stmt list)
  | While of (exp * E.pos * stmt list)
  | StmtExp of (exp * E.pos) (* evaluated and discarded *)
  | Return of (exp option * E.pos)
  | Continue of E.pos
  | Break of E.pos
  | Blank (* No Code *)

and exp = 
  (* The last field is the type of the evaluated expression.
     It is filled in during type checking and set to VOID in the parser *)
  | ConstExp of (const * ctype)
  | OpExp of (exp * op * exp option * E.pos * ctype) 
  | Alloc of (exp * E.pos * typ * E.pos * ctype)
  | StackAlloc of (exp * E.pos * typ * E.pos * ctype)
  | Offset of (exp * E.pos * ctype)
  | Size of (exp * E.pos * ctype)
  | Reference of (lval * E.pos * ctype)
  | Call of (ident * (exp * E.pos) list * E.pos * ctype)
  | LVal of lval

and lval = 
  | Var of (ident * ctype)
  | Deref of (exp * E.pos * ctype)
  | Field of (lval * ident * E.pos * ctype)

and typ = 
  | Int
  | Bool
  | Pointer of typ
  | User of ident
  | VOID

and op = 
  | PLUS | MINUS | TIMES | DIVIDE | MOD | BITAND | BITOR | BITXOR
  | SHIFTLEFT | SHIFTRIGHT
  | PPLUS | PMINUS | LOGICAND | LOGICOR | LOGICXOR 
  | EQ | NEQ | LT | LTE | GT | GTE
  | PEQ | PNEQ
  | LOGICNOT | UMINUS | BITNOT  (* unop, but represented as binop *)
and const = 
  | IntConst of (int32 * E.pos)
  | BoolConst of (bool * E.pos)
  | NULL of E.pos


(******* Pretty Print AST *******)

(* a counter for identation *)
let pp_level = ref 0

let ps text = print_string text

(* Print New Line && indentation *)
let ppnl () = 
  let rec iter = function
  | 0 -> ()
  | x -> ps " "; iter (x-1)
  in
  print_newline();
  iter !pp_level

(* print open bracket and increment pp_level *)
let pprint_br_open () =
  pp_level := !pp_level + 2; 
  ps "{"

(* print close bracket and lower pp_level *)
let pprint_br_close () =
  pp_level := !pp_level - 2;
  ppnl ();
  ps "}"

let pprint_binop = function
  | PLUS -> ps " + " 
  | MINUS -> ps " - " 
  | TIMES -> ps " * " 
  | DIVIDE -> ps " / " 
  | MOD -> ps " % " 
  | BITAND -> ps " & " 
  | BITOR -> ps " | " 
  | BITXOR -> ps " ^ " 
  | SHIFTLEFT -> ps " << " 
  | SHIFTRIGHT -> ps " >> " 
  | PPLUS -> ps " *+ " 
  | PMINUS -> ps " *- " 
  | LOGICAND -> ps " && " 
  | LOGICOR -> ps " || " 
  | LOGICXOR -> ps " ^^ " 
  | EQ -> ps " == " 
  | NEQ -> ps " != " 
  | LT -> ps " < " 
  | LTE -> ps " <= " 
  | GT -> ps " > " 
  | GTE -> ps " >= "  
  | PEQ -> ps " *== " 
  | PNEQ -> ps " *!= " 
  | LOGICNOT -> ps " !"
  | UMINUS -> ps " -"
  | BITNOT -> ps " ~"

let rec pprint_type = function
  | User(Ident(id,_)) -> ps id
  | Bool -> ps "bool"
  | Int -> ps "int"
  | Pointer(t) -> pprint_type t; ps "*" 
  | VOID -> ps "void"
let rec pprint_exp= function
  | ConstExp(IntConst(x,_),_) -> ps (Int32.to_string x)
  | ConstExp(BoolConst(true,_),_) -> ps "true"
  | ConstExp(BoolConst(false,_),_) -> ps "false"
  | ConstExp(NULL(_),_) -> ps "NULL"

  | OpExp(e1, op, None , _,_) -> 
      ps "("; pprint_binop op; pprint_exp e1; ps ")"
  | OpExp(e1, op, Some(e2), _,_) -> 
      ps "("; pprint_exp e1; pprint_binop op; pprint_exp e2; ps ")"
  | Alloc(e,_,t,_,_) -> ps "alloc( "; pprint_exp e; ps ", "; pprint_type t; ps ") "
  | StackAlloc(e,_,t,_,_) -> ps "stack_alloc( "; pprint_exp e; ps ", "; pprint_type t; ps ") "
  | Offset(e,_,_) -> ps "offset( "; pprint_exp e; ps ") "
  | Size(e,_,_) -> ps "size( "; pprint_exp e; ps ") "
  | Reference(l,_,_) -> ps "&("; pprint_lval l; ps ")"
  | Call(Ident(s,_), li,_, _) -> 
      ps (s ^ "("); pprint_arglist li; ps ")"
  | LVal(l) -> pprint_lval(l)

(* print argument list of functions / procedures *)
and pprint_arglist = function
  | [] -> ()
  | [ (e,_) ] -> 
      pprint_exp e
  | (e,_) :: tail -> 
      pprint_exp e; ps ", "; pprint_arglist tail


and pprint_lval = function
  | Var(Ident(s,_),_) -> ps s
  | Deref(e,_,_) -> ps "*("; pprint_exp e; ps ")"
  | Field(l, Ident(i,_),_,_) -> 
      ps "("; pprint_lval l; ps ")."; ps i 
	
let rec pprint_stmt2 newLine = function
    | Assign(l, _, e2, _) -> 
	if(newLine) then ppnl (); 
	pprint_lval l; ps " = "; pprint_exp e2; 
	if(newLine) then ps ";"
    | IfElse(e, _, block1, block2, _) -> 
	ppnl (); ps "if("; pprint_exp e; ps ") "; 
	if(not (pprint_block block1)) then 
	  ppnl ()
	else
	  ps " ";
	ps "else ";
	let _ = pprint_block block2 in ()
    | If(e, _, block, _) -> 
	ppnl (); ps "if("; pprint_exp e; ps ") "; 
	let _ = pprint_block block in ()
    | For(s1, _, e, _, s2, _, block) -> 
	ppnl (); ps "for("; pprint_stmt2 false s1; ps "; ";
	pprint_exp e; ps "; ";  
	pprint_stmt2 false s2; ps ") ";
	let _ = pprint_block block in ()
    | While(e, _, block) -> 
	ppnl (); ps "while("; pprint_exp e; ps ") "; 
	let _ = pprint_block block in ()
    | StmtExp(e, _) -> 
	if(newLine) then ppnl ();
	pprint_exp e; 
	if(newLine) then ps ";"
    | Continue(_) -> 
	if(newLine) then ppnl ();
	ps "continue";
	if(newLine) then ps ";"	
    | Break(_) -> 
	if(newLine) then ppnl ();
	ps "break";
	if(newLine) then ps ";"
    | Return(Some(e), _) -> 
	if(newLine) then ppnl () ;
	ps "return "; pprint_exp e; 
	if(newLine) then ps ";"
    | Return(None, _) -> 
	if(newLine) then ppnl () ;
	ps "return";
	if(newLine) then ps ";"
    | Blank -> () 
	  
and pprint_stmt s = 
  pprint_stmt2 true s

and pprint_block b = 
  match b with
  | [] -> ps " {} "; true (* parentheses printed *)
  | head :: [] ->  
      pp_level := !pp_level + 2;
      pprint_stmt head; 
      pp_level := !pp_level - 2;
      false (* no parenteses printed  *)
  | head :: tail -> 
      pprint_br_open ();
      List.iter (fun x -> pprint_stmt x) b;
      pprint_br_close();
      true (* parentheses printed *)

(* print typelist of fun/proc declaration *)
let rec pprint_typelist = function
  | [] -> ()
  | [ (Ident(id,_),t,_) ] -> 
      ps id; ps " : "; pprint_type t
  | (Ident(id,_),t,_) :: tail -> 
      ps id; ps " : "; pprint_type t; ps ", "; pprint_typelist tail

(* print typelist of struct declaration *)
let pprint_structlist tl =
  List.iter (fun (Ident(id,_),t,_) -> 
    ppnl(); ps id; ps " : "; pprint_type t; ps ";") tl

let rec pprint_body = function
  | Foreign(_) -> ps " foreign"
  | Body(dl, sl) -> 
      pprint_br_open ();
      List.iter (fun d -> pprint_decl d) dl;
      List.iter (fun s -> pprint_stmt s) sl;
      pprint_br_close ()
	
and pprint_decl = function 
  | FunDecl(Ident(id,_),ty, _, li, b) -> 
      pprint_type ty; ps " "; ps id; ps "("; 
      pprint_typelist li; ps ")";
      pprint_body b;
      ppnl (); ppnl ()
  | StructDecl(Ident(id,_), li) -> 
      ps "struct "; ps id; ps " "; pprint_br_open ();
      pprint_structlist li;
      pprint_br_close (); ps ";";
      ppnl(); ppnl ()
  | VarDecl(Ident(id,_), ty, _) -> 
      ppnl(); ps "var "; ps id; ps " : "; pprint_type ty; ps ";"

let pprint_ast (Program dl) =
  ps "//=============== PP AST ================\n";
  pp_level := 0;
  List.iter (fun d -> pprint_decl d) dl;
  ppnl ();
  ps "//============= END PP AST===============\n"



(*********** dump AST ***********)

let dump_ctype_flag = ref false

let dump_inc x =
  pp_level := !pp_level + x

let dump_dec x =
  pp_level := !pp_level - x   

let dump_ident (Ident(s,p)) = ps "Ident("; ps s; ps ")"

let dump_op op = 
    pprint_binop op
    
let dump_const = function
  | IntConst(i,_) -> ps "IntConst("; ps (Int32.to_string i); ps ")"
  | BoolConst(true,_) -> ps "BoolConst(true)"
  | BoolConst(false,_) -> ps "BoolConst(false)"
  | NULL(_) -> ps "NULL"

let rec dump_ctype ty = 
  let rec dump_ctyp = function
    | TyInt -> ps "TyInt"
    | TyBool -> ps "TyBool"
    | TyRecord(s) -> ps "TyRecord("; ps s; ps ")"
    | TyPointer(ctyp) -> ps "TyPointer("; dump_ctyp ctyp; ps ")"
    | NS -> ps "NS"
    | Void -> ps "Void"
  in
  if(!dump_ctype_flag) then (
    ps ", "; dump_ctyp ty
   ) else 
    ()
    
let rec dump_typ = function
  | Int -> ps "Int"
  | Bool -> ps "Bool"
  | Pointer(typ) -> ps "Pointer("; dump_typ typ; ps ")"
  | User(id) -> ps "User("; dump_ident id; ps ")"
  | VOID -> ps "VOID"

let rec dump_lval = function
  | Var(i, ct) -> ps "Var("; dump_ident i; dump_ctype ct; ps ")"
  | Deref(e, _, ct) -> 
      dump_inc 6; 
      ps "Deref("; dump_exp e; dump_ctype ct; ps ")";
      dump_dec 6
  | Field(l, i, _, ct) -> 
      dump_inc 6;
      ps "Field("; dump_lval l; ps ", "; dump_ident i; dump_ctype ct; ps ")";
      dump_dec 6

and dump_exp = function
  | ConstExp(c,ct) -> ps "ConstExp("; dump_const c; dump_ctype ct; ps ")"
  | OpExp(e1, op, Some(e2), _, ct) -> 	
      dump_inc 6;
      ps "OpExp("; dump_exp e1; ps ","; ppnl ();
      dump_op op; ps ","; ppnl ();
      dump_exp e2; dump_ctype ct; ps ")";
      dump_dec 6;
  | OpExp(e1, op, None, _, ct) ->
      dump_inc 6;
      ps "OpExp("; dump_exp e1; ps ","; ppnl ();
      dump_op op; ps ", None ";
      dump_ctype ct; ps ")";
      dump_dec 6;
  | Alloc(e, _, typ, _, ct) -> 
      dump_inc 6;
      ps "Alloc("; dump_exp e; ps ","; ppnl ();
      dump_typ typ; dump_ctype ct; ps ")";
      dump_dec 6;
  | StackAlloc(e, _, typ, _, ct) -> 
      dump_inc 11;
      ps "StackAlloc("; dump_exp e; ps ","; ppnl ();
      dump_typ typ; dump_ctype ct; ps ")";
      dump_dec 11;
  | Offset(e, _, ct) -> 
      dump_inc 7;
      ps "Offset("; dump_exp e; dump_ctype ct; ps ")";
      dump_dec 7;
  | Size(e, _, ct) -> 
      dump_inc 5;
      ps "Size("; dump_exp e; dump_ctype ct; ps ")";
      dump_dec 5;
  | Reference(l, _, ct) -> 
      dump_inc 10;
      ps "Reference("; dump_lval l; dump_ctype ct; ps ")";
      dump_dec 10;
  | Call(i, li, _, ct) -> 
      dump_inc 5; 
      ps "Call("; dump_ident i; 
      List.iter (fun (e,_) -> ps ","; ppnl (); dump_exp e) li;
      ps ")";
      dump_dec 5;
  | LVal(l) -> 
      dump_inc 5;
      ps "LVal("; dump_lval l; ps ")"; 
      dump_dec 5;
      
and dump_stmt = function
  | Assign(l, _, e, _) -> 
      dump_inc 7;
      ps "Assign("; dump_lval l; ps ","; ppnl ();
      dump_exp e; ps ")";
      dump_dec 7;
      ppnl ()
  | IfElse(e, _, sl1, sl2, _) -> 
      dump_inc 7;
      ps "IfElse("; dump_exp e; ppnl ();
      ps "["; dump_inc 1;
      List.iter (fun s -> dump_stmt s) sl1;
      ps "]"; dump_dec 1;ppnl ();
      ps "["; dump_inc 1;
      List.iter (fun s -> dump_stmt s) sl2;
      ps "]"; ppnl ();
      dump_dec 7;
      ppnl ()
  | If (e, _, sl, _) -> 
      dump_inc 3;
      ps "If("; dump_exp e; ppnl ();
      ps "["; dump_inc 1;
      List.iter (fun s -> dump_stmt s) sl;
      ps "]"; dump_dec 1; 
      dump_dec 3;
      ppnl ()
  | For(s1, _, e, _, s2, _, sl) -> 
      dump_inc 4;
      ps "For("; dump_stmt s1; 
      dump_exp e; ppnl ();
      dump_stmt s2;  
      ps "["; dump_inc 1;
      List.iter (fun s -> dump_stmt s) sl;
      ps "]"; dump_dec 1; 
      dump_dec 4;
      ppnl ()
  | While(e, _, sl) -> 
      dump_inc 6;
      ps "While("; dump_exp e; ps ", "; ppnl ();
      ps "["; dump_inc 1;
      List.iter (fun s -> dump_stmt s) sl;
      ps "]"; dump_dec 1; 
      dump_dec 6;
      ppnl ()
  | StmtExp(e,_) -> 
      dump_inc 8;
      ps "StmtExp("; dump_exp e; ps ")"; 
      dump_dec 8;
      ppnl ()
  | Return(Some(e), _) -> 
      dump_inc 7;
      ps "Return("; dump_exp e; ps ")"; 
      dump_dec 7;
      ppnl ()
  | Return(None,_) -> 
      ps "Return(None)"; ppnl ()
  | Continue(_) -> ps "Continue"; ppnl ()
  | Break(_) -> ps "Break"; ppnl ()
  | Blank -> ()

and dump_body = function
  | Body(dl, sl) -> 
      ps "Body("; dump_inc 2; ppnl ();
      List.iter (fun d -> dump_decl d) dl;
      List.iter (fun s -> dump_stmt s) sl;
      dump_dec 2; 
      ps ")";
      ppnl()
  | Foreign(_) -> ps "Foreign"; ppnl ()

and dump_decl = function
  | FunDecl(i, t, _, p, b) -> 
      dump_inc 8;
      ps "FunDecl("; dump_ident i; ps ", "; 
      dump_typ t; ps ", "; 
      ps "[ "; 
      List.iter (fun (i, t, _) -> 
	ps "("; dump_ident i; ps ", "; 
	dump_typ t; ps ") "; 
		) p;
      ps "], "; 
      dump_dec 6; 
      dump_body b;
      dump_dec 2;
      ps "       )";
      ppnl ()
  | StructDecl(i, li) -> 
      dump_inc 11;
      ps "StructDecl("; dump_ident i; ps ", ["; dump_inc 1; ppnl ();
      List.iter (fun (i, t, _) -> 
	ps "("; dump_ident i; ps ", "; 
	dump_typ t; ps ")"; ppnl ()
		) li;
      ps "]"; dump_dec 1; ppnl ();
      dump_dec 11;
      ps "          )";
      ppnl ()
  | VarDecl(i, t, _) -> 
      ps "VarDecl("; dump_ident i; ps ", "; dump_typ t; ps ")"; ppnl ()

and dump_ast (Program(dl)) = 
  ps "//=============== DUMP AST ================\n";
  pp_level := 0;
  dump_ctype_flag := false;
  List.iter (fun d -> dump_decl d) dl;
  ppnl ();
  ps "//============= END DUMP AST===============\n"
  
and dump_c_ast (Program(dl)) = 
  ps "//=============== DUMP CHECKED AST ================\n";
  pp_level := 0;
  dump_ctype_flag := true;
  List.iter (fun d -> dump_decl d) dl;
  ppnl ();
  ps "//============= END DUMP CHECKED AST===============\n"
