/* tests.c */

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <assert.h>

#include "tests.h"
#include "parameters.h"

#ifdef E_ASTAR
#include "astar.h"
#endif

int counter1;
int counter2;


void setAllMarksInSimpleGraph(Node** arSimpleGraph, int numNodes, int nodeMark, int edgeMark, int markOnlyNodes)
{
	int i;

	for(i=0; i<numNodes; ++i)
	{
		Node* node;
		Edge* edge;
		
		node = arSimpleGraph[i];
		
		node->nodeMark = nodeMark;
		
		counter1++;
		
		if(!markOnlyNodes)
		{
			edge=node->pEdges;
			while(edge!=NULL)
			{
				edge->edgeMark = edgeMark;
				edge = edge->pNext;
				counter2++;
			}
		}
		
	}
}

void setAllMarksInOptimizedGraph(ONode* arOptimizedGraph, int numNodes, int nodeMark, int edgeMark, int markOnlyNodes)
{
	int i;

	for(i=0; i<numNodes; ++i)
	{
		int* onode;	
		int j;
		
		onode = arOptimizedGraph[i];
		
		
		onode[OFFSET_NODEMARK] = nodeMark;
		
		counter1++;
		
		if(!markOnlyNodes)
		{
			j=0;
			while(j<onode[OFFSET_NUMEDGES])
			{
				onode[OFFSET_EDGEMARK(j)] = edgeMark;
				j++;
				counter2++;
			}
		}
			
		
	}
}

/*
Runs a DFS, on the assumption that all nodes and edges as if they had been visited with 0s and 1s.
Will mark them with the visiting order, wch scounter 1 and counter 2.
Starts at node 0 
*/
void runDfsInSimpleGraph(Node** arSimpleGraph, int numNodes, int curNode)
{

		Node* cur;
		Edge* edge;
		
		cur = arSimpleGraph[curNode];
		if(cur->nodeMark>0)
			return;
		
		cur->nodeMark = counter1++;
		
		edge=cur->pEdges;
		
		while(edge!=NULL)
		{
			edge->edgeMark = counter2++;
			
			runDfsInSimpleGraph(arSimpleGraph, numNodes, edge->pTarget->serial);
			
			edge = edge->pNext;
		}
}


void runDfsInOptimizedGraph(ONode* arOptimizedGraph, int numNodes, int curNode)
{

		int* onode;
		int j;
		
		onode = arOptimizedGraph[curNode];
/*		
		printf("%d %d %d %d\n",onode[0], onode[1], onode[2], onode[3]);
*/		
		if(onode[OFFSET_NODEMARK]>0)
			return;
			
		
		onode[OFFSET_NODEMARK] = counter1++;


		j = 0;
		while(j<onode[OFFSET_NUMEDGES])
		{
			onode[OFFSET_EDGEMARK(j)] = counter2++;
			
			runDfsInOptimizedGraph(arOptimizedGraph, numNodes, onode[OFFSET_EDGETARGET(j)]);
			
			j++;
		}	

	return;
}


void runBfsInSimpleGraph(Node** arSimpleGraph, int numNodes, int startNode, int* arWorklist)
{
		Node* cur;
		Edge* edge;
		int insertion;
		int deletion;
		
		int count;
		
		/* Initialize the work list */
		count = 0;
		deletion = 0;
		arWorklist[deletion]=startNode;
		insertion = 1;
		
		/* Main loop */
		while(deletion<insertion)
		{
			int curindex;
			
			curindex = arWorklist[deletion];
			deletion++;
			cur = arSimpleGraph[curindex];
			
			if(cur->nodeMark>0)
				continue;
				
			cur->nodeMark = counter1++;
		
			edge=cur->pEdges;
			while(edge!=NULL)
			{
				int targetNode;
				
				edge->edgeMark = counter2++;
				
				targetNode = edge->pTarget->serial;
				
				if(arSimpleGraph[targetNode]->nodeMark==0)
				{
					arWorklist[insertion++]=targetNode;
				}
			
				edge = edge->pNext;
			}

			count++;
		}

		//printf("*** BFS-- %d ***\n", count);
}


/*
Runs a BFS, on the assumption that all nodes and edges as if they had been visited with 0s and 1s.
Will mark them with the visiting order, wch scounter 1 and counter 2.
Starts at node 0 
*/
void runBfsInOptimizedGraph(ONode* arOptimizedGraph, int numNodes, int startNode, int* arWorklist)
{
		int* cur;
		int insertion;
		int deletion;	
		int j;	

		/* Initialize the work list */
		deletion = 0;
		arWorklist[deletion]=startNode;
		insertion = 1;
		
		/* Main loop */
		while(deletion<insertion)
		{
			int curindex;
			
			curindex = arWorklist[deletion];
			deletion++;
			cur = arOptimizedGraph[curindex];
			/*printf("%d\n",cur);*/
			
			if(cur[OFFSET_NODEMARK]>0)
				continue;
				
			cur[OFFSET_NODEMARK] = counter1++;
		
		j = 0;
		while(j<cur[OFFSET_NUMEDGES])
		{
				int targetNode;
				
				cur[OFFSET_EDGEMARK(j)] = counter2++;
				
				targetNode = cur[OFFSET_EDGETARGET(j)];
				
				if((arOptimizedGraph[targetNode])[OFFSET_NODEMARK]==0)
				{
					arWorklist[insertion++]=targetNode;
				}
			
				j++;
		}
		}
}


void runAStarInSimpleGraph(Node** arSimpleGraph, int numnodes) {
#ifdef E_ASTAR
    int n1, n2;

    // pick two nodes at random
    n1 = (int) (((double) rand() / (double) RAND_MAX) * (double) numnodes);

    n2 = n1;
    while (n2 == n1)
	n2 = (int) (((double) rand() / (double) RAND_MAX) * (double) numnodes);
	
    // find the shortest path between them
    find_path_simple(arSimpleGraph, numnodes,
		     arSimpleGraph[n1], arSimpleGraph[n2]);
#endif
} // runAStarInSimpleGraph


void runAStarInOptimizedGraph(ONode* arOptimizedGraph, int numnodes) {
#ifdef E_ASTAR
    int n1, n2;
    
    // pick two nodes at random
    n1 = (int) (((double) rand() / (double) RAND_MAX) * (double) numnodes);

    n2 = n1;
    while (n2 == n1)
	n2 = (int) (((double) rand() / (double) RAND_MAX) * (double) numnodes);
	
    // find the shortest path between them
    find_path_opt(arOptimizedGraph, numnodes,
		     arOptimizedGraph[n1], arOptimizedGraph[n2]);
#endif
} // runAStarInOptimizedGraph


void runTests(Node** arSimpleGraph, ONode* arOptimizedGraph, int numNodes)
{
	long earlyClock, laterClock, delta;
	int i;
	int markOnlyNodes = MARK_ONLY_NODES;
	int markruns = NUM_RUNS;
	int worklistsize;
	int* arWorklist;
	
	/* ========== ORIGIANL GRAPH ========
	printf("Starting tests on original graph \n");
	*/
	
	worklistsize = WORKLIST_SIZE;
	arWorklist = (int*)malloc(worklistsize*sizeof(int));
	
	earlyClock = clock();

	// seed the random number generator
	if (RUN_ASTAR)
	{
	    srand(1);
	}
	for(i=0; i<markruns; ++i)
	{
		counter1=0;
		counter2=0;

		if (!RUN_ASTAR)
		{
		    setAllMarksInSimpleGraph(arSimpleGraph,numNodes, 0, i,markOnlyNodes);
		}
#if PRT
		printf("MARK RESULTS: %d %d\n",counter1,counter2);
#endif
		counter1=0;
		counter2=0;
		
		if(RUN_BFS)
		{
			runBfsInSimpleGraph(arSimpleGraph,numNodes,0,arWorklist);
		}
		
		if(RUN_DFS)
		{
			runDfsInSimpleGraph(arSimpleGraph,numNodes,0);
		}

		if(RUN_ASTAR) {
		    runAStarInSimpleGraph(arSimpleGraph, numNodes);
		}
		       
#if PRT
		printf("ALGORITHM RESULTS: %d %d\n",counter1,counter2);
#endif
	}
	
	laterClock = clock();
	delta = laterClock - earlyClock;
	printf("Ending tests\n");
	
	printf("Ran original tests between %ld and %ld resulting in delta of %ld\n",earlyClock,laterClock,delta);



	/* ========= OPTIMIZED GRAPH ========= */
	printf("Starting tests on optimized graph \n");
	earlyClock = clock();
	/*runRepeatedDfs(arSimpleGraph);*/
	
	// seed the random number generator
	if (RUN_ASTAR)
	    srand(1);
	
	for(i=0; i<markruns; ++i)
	{
		counter1=0;
		counter2=0;

		if (!RUN_ASTAR)
		{
		    setAllMarksInOptimizedGraph(arOptimizedGraph,numNodes, 0, i,markOnlyNodes);
		}
#if PRT
		printf("MARK RESULTS: %d %d\n",counter1,counter2);
#endif

	
		counter1=0;
		counter2=0;
		if(RUN_BFS)
		{
			runBfsInOptimizedGraph(arOptimizedGraph,numNodes,0,arWorklist);
		}
		if(RUN_DFS)
		{
			runDfsInOptimizedGraph(arOptimizedGraph,numNodes,0);
		}

		if (RUN_ASTAR) {
		    runAStarInOptimizedGraph(arOptimizedGraph, numNodes);
		}
#if PRT
		printf("ALGORITHM RESULTS: %d %d\n",counter1,counter2);
#endif
	}
	
	laterClock = clock();
	delta = laterClock - earlyClock;
	printf("Ending tests\n");
	
	printf("Ran new tests between %ld and %ld resulting in delta of %ld\n",earlyClock,laterClock,delta);

}
