/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GLib Team and others 1997-2000.  See the AUTHORS
 * file for a list of people on the GLib Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GLib at ftp://ftp.gtk.org/pub/gtk/.
 */

#include <glib.h>
#include "gpriq.h"

#define g_heaparray_index(h, i) g_array_index(h, HeapEntry, i)

typedef struct _HeapEntry HeapEntry;

struct _HeapEntry
{
    gpointer key;
    gpointer data;
};

struct _GPriq
{
    GCompareFunc   key_compare;
    gboolean       key_compare_func_data_p;/* if FALSE, key_compare is a
					      GCompareFunc */
    gpointer       key_comp_data;
    GDestroyNotify key_destroy; /* pointer to a function that destroys  keys */
    GDestroyNotify data_destroy; /* pointer to a function that destroys
				    destroy data */
    GArray        *heap;
};

static inline guint
parent (guint index)
{
    return index / 2;
}

static inline guint
left (guint index)
{
    return index * 2;
}

static inline guint
right (guint index)
{
    return (index * 2) + 1;
}

static inline gint
compare (GPriq * priq, const HeapEntry * a, const HeapEntry * b)
{
    return ((GCompareFunc) (priq->key_compare)) (a->key, b->key);
}

static inline void
upheap (GPriq * priq, guint index)
{
    HeapEntry he = g_heaparray_index (priq->heap, index);
    while (index > 1 &&
	   compare (priq, &(g_array_index (priq->heap,
					   HeapEntry,
					   parent (index))), &he) > 0)
	{
	    g_heaparray_index (priq->heap, index)
		= g_heaparray_index (priq->heap, parent (index));

	    index = parent (index);
	}

    g_heaparray_index (priq->heap, index) = he;
    return;
}

static inline void
downheap (GPriq * priq, guint index)
{
    guint child;
    HeapEntry he = g_heaparray_index (priq->heap, index);
    while (index <= (priq->heap->len - 1) / 2)
	{

	    if (left (index) < (priq->heap->len - 1)
		&& (compare (priq,
			     &(g_heaparray_index (priq->heap, left (index))),
			     &(g_heaparray_index (priq->heap, right (index)))) > 0))
		{
		    child = right (index);
		}
	    else
		{
		    child = left (index);
		}

	    if (compare (priq, &he, &(g_heaparray_index (priq->heap, child))) <= 0)
		{
		    break;
		}

	    g_heaparray_index (priq->heap, index)
		= g_heaparray_index (priq->heap, child);

	    index = child;
	}

    g_heaparray_index (priq->heap, index) = he;

    return;
}

static inline GPriq *
g_priq_new_full_real (GCompareFunc key_compare_func,
		      gboolean key_compare_func_has_data,
		      gpointer key_compare_data,
		      GDestroyNotify key_destroy_func,
		      GDestroyNotify data_destroy_func)
{
    HeapEntry he = { NULL, NULL };

    GPriq *priq;

    g_return_val_if_fail (key_compare_func != NULL, NULL);

    priq = g_new (GPriq, 1);
    priq->key_compare = key_compare_func;
    priq->key_compare_func_data_p = key_compare_func_has_data;
    priq->key_comp_data = key_compare_data;
    priq->key_destroy = key_destroy_func;
    priq->data_destroy = data_destroy_func;

    priq->heap = g_array_new (FALSE, FALSE, sizeof (HeapEntry));

    g_array_append_val (priq->heap, he);

    return priq;
}


static inline void
traverse_fast (GPriq * priq, GTraverseFunc traverse_func, gpointer user_data)
{
    int i;
    HeapEntry *he;
	
    for (i = 1; i < priq->heap->len; i++)
	{
	    he = &g_heaparray_index (priq->heap, i);
	    if (traverse_func (he->key, he->data, user_data))
		{
		    break;
		}
	}
	
    return;
}

/*
 * creating a quickie heap for doing an inorder traversal
 * no copying of data, so we don't want to destroy it on
 * g_priq_remove
 */

static inline GPriq *
duplicate_no_destructors (GPriq * priq)
{
    int i;
    GPriq *dup = g_new (GPriq, 1);

    dup->key_compare = priq->key_compare;
    dup->key_comp_data = priq->key_comp_data;
    dup->key_compare_func_data_p = priq->key_compare_func_data_p;
    dup->key_destroy = NULL;
    dup->data_destroy = NULL;
    dup->heap = g_array_new (FALSE, FALSE, sizeof (HeapEntry) );
    g_array_set_size (dup->heap, priq->heap->len);
    //    dup->heap = g_array_sized_new (FALSE,
    //				   FALSE, sizeof (HeapEntry), priq->heap->len);
    for (i = 0; i < priq->heap->len; i++)
	{
	    g_array_append_val (dup->heap, g_heaparray_index (priq->heap, i));
	}

    return dup;
}

static inline void
traverse_inorder (GPriq * priq,
		  GTraverseFunc traverse_func, gpointer user_data)
{
    int i;
    gpointer key, data;

    GPriq *copy = duplicate_no_destructors (priq);


    for (i = g_priq_nnodes (copy); i > 0; i--)
	{
	    g_priq_remove_extended (copy, &key, &data);

	    if (traverse_func (key, data, user_data))
		{
		    break;
		}

	}

    g_priq_destroy (copy);
}

GPriq *
g_priq_new (GCompareFunc key_compare_func)
{
    return g_priq_new_full_real ((GCompareFunc) key_compare_func, FALSE,
				 NULL, NULL, NULL);
}

void
g_priq_set_key_destroy(GPriq *priq, GDestroyNotify key_destroy) {
    priq->key_destroy = key_destroy;
}
    

/*
GPriq *
g_priq_new_with_data (GCompareDataFunc key_compare_func,
		      gpointer key_compare_data)
{
    return g_priq_new_full_real (key_compare_func, TRUE, key_compare_data, NULL,
				 NULL);
}
*/

/*
GPriq *
g_priq_new_full (GCompareDataFunc key_compare_func,
		 gpointer key_compare_data,
		 GDestroyNotify key_destroy, GDestroyNotify data_destroy)
{
    return g_priq_new_full_real (key_compare_func,
				 TRUE,
				 key_compare_data, key_destroy, data_destroy);
}
*/

gint
g_priq_nnodes (GPriq * priq)
{
    g_return_val_if_fail (priq, 0);
    return (priq->heap->len) - 1;
}

void
g_priq_destroy (GPriq * priq)
{
    int i;

    if (priq->key_destroy)
	{
	    for (i = 1; i < priq->heap->len; i++)
		{
		    (priq->key_destroy) ((g_array_index (priq->heap, HeapEntry, i)).
					 key);
		}
	}

    if (priq->data_destroy)
	{
	    for (i = 1; i < priq->heap->len; i++)
		{
		    (priq->data_destroy) ((g_array_index (priq->heap, HeapEntry, i)).
					  data);
		}
	}

    g_array_free (priq->heap, TRUE);
    g_free (priq);
    return;
}

gpointer
g_priq_minimum (GPriq * priq)
{
    g_return_val_if_fail (priq->heap->len > 1, NULL);
    return (g_heaparray_index (priq->heap, 1)).data;
}

void
g_priq_minimum_extended (GPriq * priq, gpointer * key, gpointer * value)
{
    HeapEntry *he ;
    *key = NULL;
    *value = NULL;
    g_return_if_fail (priq->heap->len > 1);
    he = &(g_heaparray_index (priq->heap, 1));
    *key = he->key;
    *value = he->data;
    return;
}

void
g_priq_remove_data (GPriq * priq, gpointer * value)
{
    gint index;

    // sanity check
    g_return_if_fail(priq->heap->len > 1);
    
    // first, verify the data is in the queue
    index = g_priq_find(priq, value);
    if (index == -1) return;

    // replace the given node with the last node
    g_heaparray_index(priq->heap, index)
	= g_heaparray_index(priq->heap, priq->heap->len - 1);
    g_array_remove_index(priq->heap, priq->heap->len - 1);
    
    // rebalance heap
    downheap(priq, index);
    return;
} // g_priq_remove_data

void
g_priq_remove_extended (GPriq * priq, gpointer * key, gpointer * value)
{
    HeapEntry min;
    *key = NULL;
    *value = NULL;
    g_return_if_fail (priq->heap->len > 1);

    min = g_heaparray_index (priq->heap, 1);

    g_heaparray_index (priq->heap, 1)
	= g_heaparray_index (priq->heap, priq->heap->len - 1);

    g_array_remove_index (priq->heap, priq->heap->len - 1);

    downheap (priq, 1);

    *key = min.key;
    *value = min.data;
    return;
}

gpointer
g_priq_remove (GPriq * priq)
{
    gpointer key, data;

    g_priq_remove_extended (priq, &key, &data);

    if (priq->key_destroy)
	{
	    (priq->key_destroy) (key);
	}

    return data;
}

void
g_priq_insert (GPriq * priq, gpointer key, gpointer data)
{

    HeapEntry he;

    he.key = key;
    he.data = data;

    g_return_if_fail (priq);

    g_array_append_val (priq->heap, he);
    upheap (priq, priq->heap->len - 1);
    return;
}

gint
g_priq_find (GPriq * priq, gpointer data)
{
    HeapEntry *he;
    int i;
    gint index;
    
    // for each element in the array, check if it is equal to data
    index = -1;
    for (i=0;i<priq->heap->len;i++)
	{
	    he = &g_array_index(priq->heap, HeapEntry, i);
	    if (he->data == data) index = i;
	} // for

    return index;
} // g_priq_find

void
g_priq_foreach (GPriq * priq,
		gboolean inorder,
		GTraverseFunc traverse_func, gpointer user_data)
{
    g_return_if_fail (priq && traverse_func);
    if (inorder)
	{
	    traverse_inorder (priq, traverse_func, user_data);
	}
    {
	traverse_fast (priq, traverse_func, user_data);
    }
    return;
}
