#include <glib.h>
#include <stdio.h>
#include "gpriq.h"

#include "astar.h"
#include "simplegraph.h"
#include "optimizedgraph.h"

// uncommment to debug A* search
//#define D_ASTAR

void init_simple(Node **graph, int numnodes) {
    int i;

    for (i=0; i<numnodes; ++i) {
	Node *node;

	node = graph[i];
	node->nodeKey = 0; // 1 if in closed list, 0 otherwise
	node->nodeMark = 0; // 1 if in open list, 0 otherwise
	node->nodeData = 0; // used to track current cost to this node
    } /* for */    
} /* init_simple */

void init_opt(ONode *graph, int numnodes) {
    int i;

    for (i=0; i<numnodes; ++i) {
	ONode node;

	node = graph[i];
	node[OFFSET_NODEKEY] = 0; // 1 if in closed list, 0 otherwise
	node[OFFSET_NODEMARK] = 0; // 1 if in open list, 0 otherwise
	node[OFFSET_NODEDATA] = 0; // used to track current cost to this node
    } // for    
} // init_opt

gint compare_int(gconstpointer a, gconstpointer b) {
    return (a < b) ? -1 : ((a == b) ? 0 : 1);
} /* compare_simple */

void find_path_simple(Node **graph, int numnodes, Node *n1, Node *n2) {
    GPriq *open;
    // TODO closed should be a heap or something, not a list
    //GSList *closed, *ele, *path;
    
    Node *cur, *neighbor;
    Edge *iter;
    
    int done, final;
    int touched;
    
    touched = 0;
#ifdef D_ASTAR
    printf("@ %4d *** Searching for n%d from n%d\n",
	   touched, n2->serial, n1->serial);
#endif

    /* initialize */
    open = g_priq_new(compare_int);
    //closed = NULL;

    /* clear all marks and set all data to 0 */
    init_simple(graph, numnodes);
    
    /* insert the starting node in the priority queue */
    g_priq_insert(open, 0, n1);
    
    done = 0;
    final = BIGINT;
    while (!done) {
	// expand the node with the current least cost
	cur = (Node *) g_priq_remove(open);

#ifdef D_ASTAR     
	printf("@ %4d --- Expanding node n%d\n", touched, cur->serial);
#endif
	// for each adjoining node, check if it is
	// - the destination-- if so, determine the cost, and continue
	// - a new node-- if so, determine the cost and add to open
	// - an old node with lower cost-- if so, remove from closed and
	//   add to open
	iter = cur->pEdges;
	while (iter && !done) {
	    neighbor = iter->pTarget;
	    
	    // check if neighbor is destination
	    if (neighbor == n2) {
		// make sure this solution is better
		if (cur->nodeData + 1 < final) {
		    // update the final cost
		    final = cur->nodeData + 1;		    
		}

#ifdef D_ASTAR
		printf("@ %4d xxx Found n%d with cost %d\n", touched,
		       neighbor->serial, final);
#endif	
		// node not destination, check if it is in the open list
	    } else if (neighbor->nodeMark == 1) {
		// node is in the open list, check if cur yields a
		// lower cost path
		if (cur->nodeData + 1 < neighbor->nodeData) {
		    // remove node from open list
		    g_priq_remove_data(open, (gpointer) neighbor);
		    
		    // add node to list once more
		    neighbor->nodeData = neighbor->nodeData + 1;
		    g_priq_insert(open, neighbor->nodeData, neighbor);
		    
#ifdef D_ASTAR		    
		    printf("@ %4d +++ Re-inserted n%d in queue with cost %d\n",
			   touched, neighbor->serial, neighbor->nodeData);
#endif		    
		} // if

		// node not in open list, check if it is in closed list
	    } else if (neighbor->nodeKey == 0) {
		// node not in closed list, so calculate cost and
		// add it to the open list
		neighbor->nodeMark = 1;
		neighbor->nodeData = cur->nodeData + 1;
		g_priq_insert(open, neighbor->nodeData, neighbor);

#ifdef D_ASTAR		
		printf("@ %4d +++ Added n%d to queue with cost %d\n",
		       touched, neighbor->serial,
		       neighbor->nodeData);
#endif
		// node is in the closed list, but check to see if
		// the path to it this time is of lower cost
	    } else if (cur->nodeData + 1 < neighbor->nodeData) {
		// insert node in open list
		neighbor->nodeMark = 1;
		neighbor->nodeData = cur->nodeData + 1;
		g_priq_insert(open, neighbor->nodeData, neighbor);

		// remove item from closed list
		neighbor->nodeKey = 0;

#ifdef D_ASTAR		
		printf("@ %4d +++ Re-inserted n%d in queue with cost %d\n",
		       touched, neighbor->serial,
		       neighbor->nodeData);
#endif		
	    } // if-else-if...
	    
	    iter = iter->pNext;
	} // while

	// for debugging purposes
	touched++;
	
	// add this node to the closed list
	cur->nodeMark = 0;
	cur->nodeKey = 1;
	
	// check if we're done-- if final < lowest cost on the queue, or
	// there are no more nodes in the queue
	if (g_priq_nnodes(open) == 0 ||
	    final <= ((Node *) (g_priq_minimum(open)))->nodeData) {
	    done = 1;

#ifdef D_ASTAR
	    printf("@ %4d === Finished with ", touched);
	    if (g_priq_nnodes(open) == 0)
		printf("empty queue and final cost %d\n", final);
	    else
		printf("final cost %d and least cost item %d\n",
		       final, ((Node *) (g_priq_minimum(open)))->nodeData);
#endif
	} // if
    } // while
    
    // clean up
    g_priq_destroy(open);
} /* find_path_simple */

void find_path_opt(ONode *graph, int numnodes, ONode n1, ONode n2) {
    GPriq *open;
    // TODO closed should be a heap or something, not a list
    //GSList *closed, *ele, *path;
    
    ONode cur, neighbor;
    int target;
    
    int i, done, final;
    int touched;
    
    touched = 0;
#ifdef D_ASTAR
    printf("@ %4d *** Searching for n%d from n%d\n",
	   touched, n2[OFFSET_SERIAL], n1[OFFSET_SERIAL]);
#endif

    /* initialize */
    open = g_priq_new(compare_int);
    //closed = NULL;

    /* clear all marks and set all data to 0 */
    init_opt(graph, numnodes);
    
    /* insert the starting node in the priority queue */
    g_priq_insert(open, 0, n1);
    
    done = 0;
    final = BIGINT;
    while (!done) {
	// expand the node with the current least cost
	cur = (ONode) g_priq_remove(open);

#ifdef D_ASTAR     
	printf("@ %4d --- Expanding node n%d\n", touched, cur[OFFSET_SERIAL]);
#endif
	// for each adjoining node, check if it is
	// - the destination-- if so, determine the cost, and continue
	// - a new node-- if so, determine the cost and add to open
	// - an old node with lower cost-- if so, remove from closed and
	//   add to open

	for (i=0; i<cur[OFFSET_NUMEDGES]; ++i) {
	    // check if we're done-- if so, break
	    if (done)
		break;

	    target = cur[OFFSET_EDGETARGET(i)];
	    neighbor = graph[target];
	    
	    // check if neighbor is destination
	    if (neighbor == n2) {
		// make sure this solution is better
		if (cur[OFFSET_NODEDATA] + 1 < final) {
		    // update the final cost
		    final = cur[OFFSET_NODEDATA] + 1;		    
		}

#ifdef D_ASTAR
		printf("@ %4d xxx Found n%d with cost %d\n", touched,
		       neighbor[OFFSET_SERIAL], final);
#endif	
		// node not destination, check if it is in the open list
	    } else if (neighbor[OFFSET_NODEMARK] == 1) {
		// node is in the open list, check if cur yields a
		// lower cost path
		if (cur[OFFSET_NODEDATA] + 1 < neighbor[OFFSET_NODEDATA]) {
		    // remove node from open list
		    g_priq_remove_data(open, (gpointer) neighbor);
		    
		    // add node to list once more
		    neighbor[OFFSET_NODEDATA] = neighbor[OFFSET_NODEDATA] + 1;
		    g_priq_insert(open, neighbor[OFFSET_NODEDATA], neighbor);
		    
#ifdef D_ASTAR		    
		    printf("@ %4d +++ Re-inserted n%d in queue with cost %d\n",
			   touched, neighbor[OFFSET_SERIAL],
			   neighbor[OFFSET_NODEDATA]);
#endif		    
		} // if

		// node not in open list, check if it is in closed list
	    } else if (neighbor[OFFSET_NODEKEY] == 0) {
		// node not in closed list, so calculate cost and
		// add it to the open list
		neighbor[OFFSET_NODEMARK] = 1;
		neighbor[OFFSET_NODEDATA] = cur[OFFSET_NODEDATA] + 1;
		g_priq_insert(open, neighbor[OFFSET_NODEDATA], neighbor);

#ifdef D_ASTAR		
		printf("@ %4d +++ Added n%d to queue with cost %d\n",
		       touched, neighbor[OFFSET_SERIAL],
		       neighbor[OFFSET_NODEDATA]);
#endif
		// node is in the closed list, but check to see if
		// the path to it this time is of lower cost
	    } else if (cur[OFFSET_NODEDATA] + 1 < neighbor[OFFSET_NODEDATA]) {
		// insert node in open list
		neighbor[OFFSET_NODEMARK] = 1;
		neighbor[OFFSET_NODEDATA] = cur[OFFSET_NODEDATA] + 1;
		g_priq_insert(open, neighbor[OFFSET_NODEDATA], neighbor);

		// remove item from closed list
		neighbor[OFFSET_NODEKEY] = 0;

#ifdef D_ASTAR		
		printf("@ %4d +++ Re-inserted n%d in queue with cost %d\n",
		       touched, neighbor[OFFSET_SERIAL],
		       neighbor[OFFSET_NODEDATA]);
#endif		
	    } // if-else-if...
	} // for

	// for debugging purposes
	touched++;
	
	// add this node to the closed list
	cur[OFFSET_NODEMARK] = 0;
	cur[OFFSET_NODEKEY] = 1;
	
	// check if we're done-- if final < lowest cost on the queue, or
	// there are no more nodes in the queue
	if (g_priq_nnodes(open) == 0 ||
	    final <= ((ONode) (g_priq_minimum(open)))[OFFSET_NODEDATA]) {
	    done = 1;

#ifdef D_ASTAR
	    printf("@ %4d === Finished with ", touched);
	    if (g_priq_nnodes(open) == 0)
		printf("empty queue and final cost %d\n", final);
	    else
		printf("final cost %d and least cost item %d\n",
		       final, ((ONode) (g_priq_minimum(open)))[OFFSET_NODEDATA]);
#endif
	} // if
    } // while
    
    // clean up
    g_priq_destroy(open);
} /* find_path_opt */
