#! /bin/python3

##########
## Version: Apr, 2017
## Author: Tsubasa Takahashi
## Note: This work is done in author's visiting at Carnegie Mellon University.
##########

import numpy as np
from numpy.linalg import svd, pinv
from sktensor import dtensor, ktensor, sptensor
import scipy.stats as stats

from tenfact_tools import sptensor_diff, identity_tensor

def efficient_corcondia(X, Fac):
    """
    Input
    ----------
    X: original tensor
    Fac: PARAFAC of X

    Output
    ---------
    c: CORCONDIA diagnostic value (max: 100)

    """

    Z = []
    for Ui in Fac.U:
        Z.append(Ui)
    Z[0] = Z[0] * Fac.lmbda
    ndim = len(Fac.U)
    rank = Z[0].shape[1]

    U = []
    S = []
    V = []
    for i in range(ndim):
        Ui, si, VTi = svd(Z[i], full_matrices=False)
        U.append(Ui.T)
        S.append(pinv(np.diag(si)))
        V.append(VTi.T)

    p1 = kron_mat_vec(U, X)
    p2 = kron_mat_vec(S, p1)
    G = kron_mat_vec(V, p2)

    T = identity_tensor(rank,ndim)

    ## CORCONDIA diagnosis value
    c = 100 * ( 1- np.sum((G-T)**2)/rank)

    return float(c)

def kron_mat_vec(Alist, X):
    K = len(Alist)
    for k in range(K):
        mode = K-k-1
        A = Alist[mode]
        Y = X.ttm(A, mode)
        X = Y
    return Y


def relfit(X, Fac):
    M = Fac.toarray()

    ## RELFIT value
    if isinstance(X,sptensor):
        Z = sptensor_diff(X,M,is_sptensor=True).vals
        Y = X.vals
    else:
        Z = X-M
        Y = X

    f1 = np.sum(np.sum(np.sum((Z)**2,axis=0),axis=0))
    f2 = np.sum(np.sum(np.sum((Y)**2,axis=0),axis=0))
    rfit = 100 * (1 - f1/f2)

    return float(rfit)
