% success = useKB(kbDirectory, <depth>, <caseSensitiveEntityFiles>)
%
% This function tells THEO to use the kb stored on directory kbDirectory, and informs it that
% entities are stored on subdirectories nested "depth" levels deep.  
%
% If the optional argument 'depth' is not provided, it defaults to THEO.kbSubdirDepth.
%
% The optional argument 'caseSensitiveEntityFiles' determines whether filenames in the KB use the
% same font case as entity names (if caseSensitiveEntityFiles=1), or whether filenames are all
% lower case regardless of entity case (if caseSensitiveEntityFiles=0).  The default value for this
% optional argument is THEO.caseSensitiveEntityFiles.
%
% Example: useKB('/Users/tommitchell/Documents/MATLAB/Theo/RTW_KB_2009_03_19_ORS/')
% Example: useKB('http://rtw.ml.cmu.edu/sslnlp09/', 0, 1)
%
% HISTORY: created 3/28/09 Tom
% 9/7/09 Tom added check that kbDirectory ends with filesep, and if not, adds it
% 9/7/09 Tom added 'depth' and 'caseSensitiveEntityFiles as optional arguments
%
%
% Possible extensions: 
% 1. might want to clear THEO.entitiesInRAM 
% 2. should check whether URL directory exists?

function success = useKB(kbDirectory, depth, caseSensitiveEntityFiles)
global THEO

if nargin<2
  depth=THEO.kbSubdirDepth;
end

if nargin<3
  caseSensitiveEntityFiles=THEO.caseSensitiveEntityFiles;
end

% add fileseparator to end of KBdirectory if there isn't one already
if length(kbDirectory)>=5  && strcmp('http:',kbDirectory(1:5))
  % it's a URL, so be sure to end with '/'
  fileOrURL='URL';
  if strcmp('html',kbDirectory(end-3:end))  % quick error check since we're here
    fprintf('WARNING: you cannot give a URL to an individual file as a KBdirectory.');
    crash  % sorry, but we must stop this before it goes any further
  end
  if ~isequal(kbDirectory(end),'/')
    kbDirectory(end+1)='/';
  end
else
  fileOrURL='file';
  if ~isequal(kbDirectory(end),filesep)
    kbDirectory(end+1)=filesep;
  end
end

if strcmp(fileOrURL,'file') && ~exist(kbDirectory,'file')
  success=0;
  fprintf('WARNING: CALL useKB AGAIN WITH AN EXISTING DIRECTORY. THIS ONE DOES NOT EXIST\n');
else
  THEO.kbdir=kbDirectory;
  THEO.kbSubdirDepth=depth;
  THEO.caseSensitiveEntityFiles=caseSensitiveEntityFiles;
  THEO.readDiskKB=1;
  success=1;
end

