% swapInEntity(entity, <dir THEO.kbdir>) 
%
% reads in entity from dir (default=THEO.kbdir).  Returns 1 if successful, else 0.  Entity may be a
% char or cell array of strings.  If a cell array, it swaps in the first element of the array (i.e.,
% the top level entity). As a side effect, it add entity to THEO.entitiesInRAM.  Note this works if
% entity is stored locally, or on the web.
%
%  Matlab's xmlread, xmlwrite are based on Java org.apache.xerces.dom.DeferredDocumentImpl
%
% HISTORY
% Created 3/28/09 by Tom, copy and edit of loadEntity
% 5/29/09 Tom: added code to swap out oldest entities if number in RAM exceeds THEO.maxEntitiesInRAM
% 5/29/09 Tom: added call to entToFilename, to support nested subdirectories
% 5/31/09 Tom: added functionality for THEO.traceSwapInEntity
% 7/31/09 Tom: added more print statements when error caught

function success = swapInEntity(entity, kbdir)
global THEO
debug=0;  % set to 1 to print diagnostics
if nargin<2
  kbdir=THEO.kbdir;
end
if debug
  fprintf('** inside swapInEntity(%s)\n',v2p(entity));
end

if THEO.traceSwapInEntity
  fprintf('swapping in entity %s\n',v2p(entity));
end

if strcmp('http:', kbdir(1:5))
  location='URL';
else
  location='LocalFile';
end

if strcmp(location,'URL')
  exist = isEntityOnWeb(entity);
else
  exist = isEntityOnDisk(entity);
end

if ~exist
  success=0;
  fprintf('WARNING: swapInEntity could not locate %s\n',entity);
  return
end


fileName=entToFilename(entity,kbdir);
try
  xDoc = xmlread(fileName); % read in the entity DOM
catch ME
  fprintf('WARNING1: swapInEntity could not read file for entity %s\n',v2p(entity));
  fprintf('WARNING2: swapInEntity could not read file %s\n',fileName);
  fprintf('%s\n',v2p(ME));
  crash;
end
success=1;  % somewhat optimistic...

xRoot = xDoc.getDocumentElement; % its top level obj must be an "entity"
if ~strcmp('entity',xRoot.getTagName)
  fprintf('** ERROR: root node of ~s should be an ENTITY DOM object ***\n', ...
          fileName);
  return
end

% Java assigns DOM nodes integer types, so redefine them here
textNode=3;
element=1;

% child 0 of the entity is empty (presumably the empty text node that
% goes with the <entity> tag.  Child 1 contains the entityName

if xRoot.hasChildNodes % it would be unusual to have no child nodes...
  entChildren = xRoot.getChildNodes;
  % next statement uses 'char' to coerce java.lang.String to matlab string
  entityName = char(xRoot.getChildNodes.item(1).getFirstChild.getData);
  
  %%%%%% create entity data structure, and mark that it is in RAM
  entityStruc.generalizations.val={};
  if ~memberOf(entity,THEO.entitiesInRAM)
    THEO.entitiesInRAM{end+1}=entity;
  end
  % next 2 lines define a new variable in the global workspace, called
  % THEO_<entityname>, eg., THEO_joe, and assigns its slots, subslots,
  % etc. as a struct
  entityVarName=sprintf('%s%s', THEO.varPrefix, entity); 
  assignin('base',entityVarName,entityStruc);
  %%%%%% END create entity data structure
  
  % now go through remaining children, which should be slots and
  % their values (last "child" is the empty array, so ignore it).
  % For each slot, call createSlotFromDOMnode(node, entityName),
  % where entityName can be a list indicating a subslot.
  for k=2:(entChildren.getLength-1)
    if (entChildren.item(k).getNodeType == element) && strcmp('slot',char(entChildren.item(k).getNodeName))
      % only process element nodes that define slots - others are empty
      createSlotBareFromDOMnode(entChildren.item(k),entityName);
    end
  end
end

if length(THEO.entitiesInRAM)>THEO.maxEntitiesInRAM
  % we have too many items in RAM, so swap out the oldest (least recently swapped in) half
  swapOutN=round(length(THEO.entitiesInRAM)/2)-1;
  fprintf('THEO swapInEntity function is swapping out %d entities..', swapOutN);
  for k=swapOutN:-1:1
    swapOutEntity(THEO.entitiesInRAM{k});
  end
  fprintf('..\n');
end



% create slot, its value, and its subslots, recursively
function [] = createSlotBareFromDOMnode(node,entityName)
global THEO
debug=1;
% following are Java-defined integers for node types
textNode=3;
element=1;

nodeChildren = node.getChildNodes;
for k=1:(nodeChildren.getLength-1)
  if (nodeChildren.item(k).getNodeType == element) 
    % only process element nodes - others are empty
    child=nodeChildren.item(k);
    tagName= char(child.getTagName);
    if strcmp(tagName,'slot')
      if debug
      fprintf('createSlotBareFromDOMnode(somenode,%s,%s)\n',v2p(entityName),v2p(slotName));
      end
      
      createSlotBareFromDOMnode(child, entSlot2entAddr(entityName, slotName));
    elseif strcmp(tagName,'name')
      slotName= char(child.getFirstChild.getData);
    elseif strcmp(tagName, 'val')
      valNode=child.getFirstChild;
      val = createMatlabTypeFromXMLtype(valNode);
      if debug 
        fprintf('pv(%s,%s,%s)\n', slotName, value2printString(entityName), value2printString(val));
      end
      putValueBareBare(slotName, entityName, val);
    end
  end
end

  

%%%%%%%%%%%%%%%%%%%
% Just testing below here.
%
% HINTS FROM http://www.w3schools.com/xml/xml_dom.asp
% NOTE: you must use Java numbering for lists, begin with index=0.
% NOTE: use  xxx.getClass to find out the Java class returned to matlab.
% 
% In the examples below we use the following DOM reference to get the text from the <to> element:
%
% xmlDoc.getElementsByTagName("to")[0].childNodes[0].nodeValue
%
% xmlDoc - the XML document created by the parser.
% getElementsByTagName("to")[0] - the first <to> element
% childNodes[0] - the first child of the <to> element (the text node)
% nodeValue - the value of the node (the text itself)
% You can learn more about the XML DOM in our XML DOM tutorial.
%%%%%%%%%%%%%%%%%%%%
%if 0
%  if xRoot.hasChildNodes
%    firstChild=xRoot.getFirstChild()
%    firstChild.getNodeName
%    firstChild.getNodeType
%    firstChild.getNextSibling
%    firstChild.getNextSibling.getNodeName  %% this returns: 'entityName'
%    childNodes=xRoot.getChildNodes()
%    xRoot.getElementsByTagName('entityName').getLength 
%    xRoot.getElementsByTagName('slot').getLength 
%    xRoot.getElementsByTagName('val').getLength 
%    xRoot.getElementsByTagName('s').getLength 
%    
%  end
%  
%  % WHAT WE WANT: xmlDoc.getElementsByTagName("to")[0].childNodes[0].nodeValue
%  z=xRoot.getElementsByTagName('entityName').getLength
%  z=xRoot.getElementsByTagName('entityName').item(1).childNodes
%  z=xRoot.getElementsByTagName('entityName').item(0).getFirstChild
%  z=xRoot.getElementsByTagName('entityName').item(0).getChildNodes.item(0).getClass
%  z=xRoot.getElementsByTagName('entityName').item(0).getChildNodes.item(0).getData
%  
%  z=xRoot.getElementsByTagName('entityName').item(0).getChildNodes.item(0).getData
%  
%end
%
%%%%%%%%%%%%%% ABOVE HERE ALL WORK CORRECTLY IN MATLAB....  %%%%%%%%%%%%%%%%
