% directory = saveKB(rootOrList, directory < kbSubdirDepth>, <caseSensitiveEntityFiles> <alreadySaved>)
%
% The basic function for storing a KB on a directory.  If rootOrList is a string, treats it as an
% entity, and saves it and every one of its descendants on directory.  If rootOrList is a cell array
% of entity names, it saves each of these entities.  Also creates an index.html file to support
% browsing the KB.  If directory exists, it will be deleted and recreated.
%
% Note this function does not change the current THEO knowledge base directory as defined by
% THEO.kbdir.  To change it to this new directory, invoke the function useKB.
%
% kbSubdirDepth and caseSensitiveEntityFiles are optional arguments, and default to
% THEO.kbSubdirDepth and THEO.caseSensitiveEntityFiles, respectively.  They allow temporarily
% overriding these values during execution of this function.
%
% Example: 
%   saveKB('everything','/Users/tommitchell/Documents/MATLAB/Theo/userKBs/2009_02_26/')
%   saveKB({'plays_sport','plays_for'},'/Users/tommitchell/Documents/MATLAB/Theo/userKBs/2009_02_26/')
%   saveKB('everything','/Users/tommitchell/kb/',3,1)
%
% IMPLEMENTATION NOTE: this uses RAM linear in size of KB.  POOR scaling!  Could easily instead
% use disk space linear in size of KB -- MUCH better.
%
% HISTORY
% created by Tom 
% 3/15/09 Tom: extended to allow first arg to be a list
% 5/20/09 Tom: added call to entToFilename to calculate where to store each entity
% 6/22/09 Tom: added code to create directory if it doesn't already exist
% 9/8/09  Tom: added optional kbSubdirDepth and THEO.caseSensitiveEntityFiles
% 9/8/09  Tom: added code to append a file separator to end of input 'directory' if it is missing
% 9/12/09 Tom: added code to make 'directory' argument default to THEO.kbdir
% 9/12/09 Tom: added code to invoke webdisplayHierarchy to create index.html after saving
% 9/12/09 Tom: rewrote to use transClosure('rootOrList') to get entities to save

function directory = saveKB(rootOrList, directory, kbSubdirDepth, caseSensitiveEntityFiles)
global THEO

if nargin<2
  directory=THEO.kbdir;
end
if nargin<3
  kbSubdirDepth=THEO.kbSubdirDepth;
end
if nargin<4
  caseSensitiveEntityFiles=THEO.caseSensitiveEntityFiles;
end
if nargin<5
  alreadySaved={};
end

% in case input directory doesn't end with a filesep, add one
if ~isequal(directory(end),filesep)
    directory(end+1)=filesep;
end

% it directory already exists, delete and recreate it
if exist(directory)==7
  rmdir(directory,'s')
end
mkdir(directory);

% copy this file so that the saved KB can be viewed with web browser
copyfile([THEO.TheoKBdir 'displayWithSubslots.xsl'], [directory 'displayWithSubslots.xsl']);

if ischar(rootOrList) % input is name of root entity
  allEnts=transClosure(rootOrList,'specializations');
  for k=1:length(allEnts)
    saveFile=entToFilename(allEnts{k},directory,kbSubdirDepth,caseSensitiveEntityFiles);
    saveEntity(allEnts{k},saveFile,kbSubdirDepth);
  end
elseif iscellstr(rootOrList) % input is list of entities to save
  for k=1:length(rootOrList)
    saveFile=entToFilename(rootOrList{k},directory,kbSubdirDepth,caseSensitiveEntityFiles);
    saveEntity(rootOrList{k},saveFile,kbSubdirDepth);
  end
end

if ischar(rootOrList)
  webdisplayHierarchy(rootOrList,0,{},[directory 'index.html']);
end
