%  filename = saveEntity(entity, <filename>, <kbSubdirDepth>)
%
% Saves ENTITY in XML format on FILE. FILE is an optional argument. If not given, it defaults to
% entToFilename(entity,THEO.kbdir).  It returns the file name with full path.  If the file already
% exists, it is overwritten.  If entity is a cell array, such as {'domain' 'whenToCache'} then it
% saves the entire top-level entity (e.g., 'domain').   The optional argument kbSubdirDepth
% (default = THEO.kbSubdirDepth) specifies how deeply the subdirectories are nested, so that the
% correct number of "../"s can be added to the relative location of the style sheet (whoo!). 
%
% Slot values are stored in xml format with tags for the following
% data types:
%
%      <l>,</l> = list
%      <s> = string
%      <r> = real value
%      <in> = integer value
%
%
% Example: 
%
% saveEntity('person'); % saves the entity 'person' to person.xml
% saveEntity({'person','specializations}); % saves the entity 'person' to person.xml (in general,
%                                            saves the top-level entity)
% saveEntity('person','/Users/tommitchell/Documents/MATLAB/inramTheo/userKBs/person.xml'); % saves the entity 'person' to person.xml
%
% TBD: Should probably add the write date, source info.
% TBD: Should probably change second input arg to be a top level directory, not a specific filename.
% TBD: check why this is using THEO.TheoKBdir, but useKB is using THEO.kbdir...
%
%
% HISTORY
% Created 10/23/2008 by Tom
% 3/21/09 Tom - replaced inefficient fileprintf by fprintf and opening file just once
% 5/20/09 Tom - added call to entoToFilename to calculate default file location 
% 5/29/09 Tom - upgraded so if THEO.readDiskKB=1 and entity not in RAM, it first loads it
% 5/31/09 Tom - added ability to accept input entities which are lists (just saves top level entity)
% 6/17/09 Tom - added code to create directory if it doesn't exist
% 9/9/09  Tom - added optional argument kbSubdirDepth 

function filename = saveEntity(entity, filename, kbSubdirDepth)
global THEO
rslt='';
DEBUG=0;
if DEBUG
  fprintf('saveEntity(%s)\n',v2p(entity));
end

if iscell(entity)  % if entity is a slot or subslot, save the entire top level entity
  entity=entity{1};
end

if THEO.readDiskKB && ~isEntityInRAM(entity) % first check whether entity is in RAM
  if ~swapInEntity(entity) % and if not, try swapping it in from disk
    fprintf('**WARNING: saveEntity cannot find %s on disk or in memory.\n',v2p(entity));
    crash;
  end
end

if nargin<2
  filename=entToFilename(entity,THEO.kbdir);
end
if nargin<3
  kbSubdirDepth=THEO.kbSubdirDepth;
end


% if directory doesn't yet exist, create it
[dir, name, ext, versn] = fileparts(filename) ;
if ~(exist(dir)==7) % directory doesn't yet exist..
  mkdir(dir);
end

% 0. open the file for printing
fid = fopen(filename,'w+');  % open or create file, discarding existing contents
% fprintf(fid,printCommand, varargin{4:end});


% 1. print file heading XML version and stylesheet
fprintf(fid,'<?xml version="1.0" encoding="utf-8"?>\n');
% I can't seem to get the .dtd file to work...
%fprintf(fid,'<!DOCTYPE entity SYSTEM "entity.dtd">\n');


% create stylesheet href location
styleSheet='displayWithSubslots.xsl';

for k=1:kbSubdirDepth
  styleSheet=['../',styleSheet];
end

fprintf(fid,'<?xml-stylesheet type="text/xsl" href="%s"?>\n',styleSheet);

% 1.1 add name of entity
fprintf(fid,'<entity>\n   <entityName>%s</entityName>\n',entity);
indentStr='   ';


% 1.2 for each slot (and its sub...slots) add new element
slotNames=getCachedSlots(entity);
for k=1:length(slotNames)
  if ~strcmp('val',slotNames{k})
    s=slotNames{k};
    v=getValue(s,entity);
    if isTheoValue(v) 
      addSlotChildElement(s,entity,v,fid,indentStr);
    end
  end
end

% close entity tag
fprintf(fid,'</entity>\n<!-- created by saveEntity-->\n');

% close the file
fclose(fid);
