%  [rslt idx] =removeValue(slot, entity, value)
%
% Remove value from the values in slot of entity, along with its supporting information in the
% subslots 'source' and 'probability' if they happen to exist.  If value does not exist in slot of
% entity, it does nothing.  Slot must be a string.  Entity may be a string naming a top-level
% entity, or it may be a cell array of strings representing a (sub)slot of an entity.  If the
% latter, the slot of that nested entity will be deleted.
%
% Returns:
%  rslt=1 if value was found in slot's values, else 0
%  idx= an integer indicating which position value appeared in, within slot's values
%
% Note that if deleting the slot's value results in a totally empty slot with no subslots or value,
% then the slot itself is also deleted by calling deleteSlotBare.
%
% Implements Layer 1 addValue function.  Adds to layer 0:
%
% 1. if you removeValue('specializations',x,y) it also
% removeValue('generalizations',y,x), and vice versa.
%
% 2. if slot has a source and/or probability subslot, these are maintained by removing the
% corresponding source/probability entries
%
%
% Examples:
%  removeValue('hasOperationsIn','seagate','pittsburgh')
%
% Implementation note: optional recursiveCall argumt is only used by removeValue calls itself,
% and wants to signal this fact.
%
% HISTORY:
% Created Tom 2/20/2009
% 9/12/09 Tom: added functionality to maintain generalizations/specializations hierarchy

function [rslt idx] = removeValue(slot, entity, value, recursiveCall)
global THEO
rslt=0;
v=getValueBare(slot,entity);
[brslt, idx]=removeValueBare(slot,entity,value);

if brslt % success, so update source and probability subslots if they exist
  rslt=brslt;
  se=entSlot2entAddr(entity,slot); 
  if isTheoValue(getValueBare('source',se))
    removeNthValueBare('source',se,idx);
  end
  if isTheoValue(getValueBare('probability',se))
    removeNthValueBare('probability',se,idx);
  end

  if nargin<4 || ~recursiveCall  % remove backpointer in gen/spec hierarchy
    if strcmp('specializations',slot)
      removeValue('generalizations',value,entity,1);
    elseif strcmp('generalizations',slot)
      removeValue('specializations',value,entity,1);
    end
  end
end
