% rslt = putValueC(s,e,v,c, source, probability)
%
% level 3 putValue function.  Puts value v into slot s of entity e, and if the optional source and
% probability are provided, stores these in its subslots.  Input argument c determines whether it
% adds a candidateValues subslot in case s already has a different value.  In particular, if c=0
% then it just overwrites the old value of s.  However, if c=1 and nrOfValues(s)=1 and slot s
% already has a different value, then v is inserted as the new value, but a candidateValues subslot
% is also added and both the old value and v are inserted as its values, and it is assigned a source
% and probability subslot if these are provided.  If candidateValues already exists, the new value
% is added to it, and is also asserted as the value of the slot s.  Note if v equals the existing
% slot value, then its source and probability update the current subslot values.
%
% Note that source must be just a single source, e.g., {'manualInput' 0.2}.  If you have multiple
% sources, just call this function multiple times.
%
% Example: the two commands
%   putValue('spouse','tom','joan'); 
%   putValueC('spouse','tom','bob', 0,{'manual' 0.1}, 0.1);
% results in the following:
%
% tom:
%  generalizations = {male} 
%  father = bob
%  wife = joan
%    source = {{{manualInputTom, 1} } } 
%    probability = {1} 
%  spouse = bob
%    candidateValues = {joan, bob} 
%      source = {{{unknown} } , {{manual, 0.1} } } 
%      probability = {unknown, 0.1} 
%    source = {{{mostRecentCandidateValue, 0.1} } } 
%    probability = {0.1} 
%
%
% HISTORY
% Created 2/23/2009 Tom   (tested and seems to work, but test it some more...)
%  2/25/2009 Tom - fixed parenthesis bug in creating source for candidateValues
%

% DEBUG:
% s='spouse'; e='tom'; v='bob'; c=1; source={'manual' 0.1}; probability=0.1;
% s='father'; e='tom'; v='louise'; c=1; source={'manual2' 0.2}; probability=0.2;
% putValueC(s,e,v,c, source, probability);

function rslt = putValueC(s,e,v,c, source, probability)
global THEO
display(nargin)
if nargin<5  
  source={'unknown'};
end

oldVal=getValueBare(s,e);
if isequal(oldVal, v)
  % then update source and probability, then exit
  esAddr=entSlot2entAddr(e,s);
  if nargin>4
    createSlotSP(s,e,1,0); % create source subslot unless already exists
    src=getValueBare('source',esAddr);
    src=src{1}; % assumes nrOfValues=1
    src{end+1}=source;
    putValue('source',esAddr,{src});
  end
  if nargin>5  % create probability subslot unless already exists
    createSlotSP(s,e,0,1);
    putValue('probability',esAddr,{probability});    
  end
  return
end


if (c && isequal(getValue('nrOfValues',s),1) && isTheoValue(oldVal))
  % we need to use candidateValues subslot to store values
  esAddr=entSlot2entAddr(e,s);
  escAddr=entSlot2entAddr(esAddr,'candidateValues');
  cv=getValueBare('candidateValues',esAddr);
  if ~isTheoValue(cv) 
    %  need to create a list of candiateValues, its source, and perhaps its probability subslot
    addValue('candidateValues',esAddr,oldVal);
    oldS=getValueBare('source',esAddr);
    oldP=getValueBare('probability',esAddr);
    if isTheoValue(oldS)
      addValue('source',escAddr,oldS{1}); 
    else
      addValue('source',escAddr,{{'unknown'}});
    end
    if isTheoValue(oldP)
      addValue('probability',escAddr,oldP{1}); 
    elseif nargin==6
      addValue('probability',escAddr,'unknown');
    end
  end
  if nargin==6
    addValueSP('candidateValues',esAddr,v,source,probability);
    putValueSP(s,e,v,{'mostRecentCandidateValue' probability},probability);
  else
    addValueS('candidateValues',esAddr,v,source);
    putValueS(s,e,v,{'mostRecentCandidateValue'});
  end
else % just put the value v into slot s, overwriting as needed
  rslt=putValueBare(s,e,v);
  if nargin>4
    ea=entSlot2entAddr(e,s); 
    putValueBare('source',ea,{{source}});
  end
  if nargin>5
    putValueBare('probability',ea,{probability});
  end
end
