% printHierarchy(rootentity <slotsToPrint>)
%
% Prints the hierarchy of specializations under rootentity, as well as any known values for slots
% included in the optional argument "slotsToPrint" (which defaults to the empty list {}). 
%
% For typing convenience, the function prh(rootentity, <slotsToPrint>) is defined as a synonym.
%
% Examples:
%  prh('person' {'mother' 'father'}) : prints tree of specializations under "person"
%                                      including any cached slot values of "mother" or "father"
%  prh('slot') : print the tree of specializations of 'slot'
%  prh('relationships' 'all') : prints specializations of "relationships", 
%                               including ALL known slot values of every entity printed.
%
% HISTORY:
% Created 10/20/2008 -Tom


function rslt = printHierarchy(entity, slotsToPrint)
global THEO
rslt={};
printAllSlots=0; % if input sets this to 1, print vals of all slots

if nargin<2
  slotsToPrint={};
end
if strcmp('all',slotsToPrint)
  printAllSlots=1;
end

indentStr=''; % add spaces for nesting
if isEntity(entity)
  rslt=prHelp(entity,slotsToPrint,printAllSlots,{},indentStr);
else
  fprintf('sorry, %s does not exist.\n',entity);
end
fprintf('\n');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% recursive fn to actually do printing -- ALSO ADD SLOT PRINTING.
% accepts an argument 'alreadyPrinted' listing slots allready
% printed.  It returns the list of slots it has printed (not
% including alreadyPrinted).
function printedEnts = prHelp(entity,slotsToPrint,printAllSlots,alreadyPrinted,indentStr)
printedEnts={entity};

% 1. print the entity (including '**' if already printed)
fprintf('%s%s', indentStr,entity);
if ismember(entity,alreadyPrinted)
  fprintf(' ***\n');
  return
else
  fprintf('\n');
end

% 1. print slots if instructed
if printAllSlots
  slotsToPrint=getCachedSlots(entity);
end
if ~isempty(slotsToPrint)
  for k=1:length(slotsToPrint)
    if ~(strcmp('generalizations',slotsToPrint{k})| ...
         strcmp('specializations',slotsToPrint{k}))
      
      val=getValueBare(slotsToPrint{k},entity);
      if isTheoValue(val)
        valStr=value2printString(val);
        fprintf('%s    | %s = %s\n',indentStr,slotsToPrint{k},valStr);
      end
    end
  end
end

% 2. for each specialization, call recursively
specs=getValueBare('specializations',entity);
if isTheoValue(specs)  % it's not THEO_NO_VALUE
  for k=1:length(specs)
    s=specs{k};
    prEnts = prHelp(s,slotsToPrint,printAllSlots,alreadyPrinted,[indentStr,'  ']);
    alreadyPrinted=[alreadyPrinted,prEnts];
    printedEnts=[printedEnts,prEnts];
  end
end

