% deleteEntity(ent).  Also support de(entity) for typing convenience.
% 
% Deletes entity ent.  Ent must be a top level Theo entity described by a string.  Deletes the
% entity from RAM, and if THEO.mainainDiskKB=1 then also from the directory THEO.kbdir
%
% Side effects: if ent has specializations, these become specializations of ent's
% generalizations.
%
%%%%% Implementation Strategy %%%%%%
% 1. if a top level entity, remove it from it's generalizations and
% specializations, and remove it from THEO.entitiesInRAM.
%
% 2. delete the (sub)struct from memory, and if THEO.maintainDiskKB, then remove file from
% directory.
%
%%%%%%
%
% EXAMPLES:
%   de('tom')
%
% IMPLEMENTATION NOTE: can get into surprising behavior if THEO.readDiskKB=1 but
% THEO.maintainDiskKB=0.  In this case, deleteEntity(e) will delete e from RAM, but not update
% disk (because THEO.maintainDiskKB=0).  But other functions might read it from disk later,
% because it's still out there.  So do a saveKB or set THEO.maintainDiskKB=1 if you want the
% delete to be permanent.
%
% HISTORY: created 3/14/09 by Tom
% 5/31/09 Tom: changed from using THEO.entities to THEO.entitiesInRAM
% 5/31/09 Tom: changed over to use entToFilename to find entity on disk
% 9/12/09 Tom: added debugging code

function rslt = deleteEntity(ent)
global THEO
rslt=0;
debug=0;

if THEO.readDiskKB && ~isEntityInRAM(ent) % first check whether entity is in RAM
  if ~swapInEntity(ent) % and if not, try swapping it in from disk
    fprintf('**WARNING: putValueBare cannot find %s on disk or in memory.\n',v2p(ent));
    crash;
  end
end

if isEntity(ent)
  if debug fprintf('debug1: %s\n',ent); end
  gs=getValueBare('generalizations',ent);
  ss=getValueBare('specializations',ent);
  if isTheoValue(ss)
    for k=1:length(ss) % connect each specialization to the generalizations of ent
      removeValueBare('generalizations',ss{k},ent);
      if isTheoValue(gs)
        for m=1:length(gs)
          % note addValue will automatically also add ss{k} to specializations of gs{m}
          addValue('generalizations',ss{k},gs{m});
        end
      end
    end
  end
  if isTheoValue(gs)
    for k=1:length(gs) % remove ent from specializations of its parents
      removeValueBare('specializations',gs{k},ent);
      if debug fprintf('debug2: removeValueBare(specializations,%s,%s)\n',gs{k},ent); end
    end
  end
  
  THEO.entitiesInRAM=remove(ent,THEO.entitiesInRAM);
  if debug fprintf('debug3: memberOf(%s,THEO.entitiesInRam)=%d\n',ent,memberOf(ent,THEO.entitiesInRAM)); end
  
  % remove the RAM represention of ent
  entityVarName=sprintf('%s%s', THEO.varPrefix, ent);
  cmdStr=['clear ',entityVarName];
  evalin('base',cmdStr);

  % remove the disk representation of ent
  if THEO.maintainDiskKB
%    filename=sprintf('%s%s.xml',THEO.kbdir,ent);  
    filename=entToFilename(ent,THEO.kbdir);
    if exist(filename)==2  % delete any existing file with this name
      delete(filename);
    end
  end
  if THEO.maintainDiskHierarchy
    webdisplayHierarchy('everything',0,'all');
  end
end

