import java.util.List;
import java.util.Iterator;

// Theo KB stuff
import edu.cmu.ml.rtw.kb.RTWValue;
import edu.cmu.ml.rtw.kb.RTWListValue;
import edu.cmu.ml.rtw.kb.RTWIntegerValue;
import edu.cmu.ml.rtw.kb.RTWRealValue;
import edu.cmu.ml.rtw.kb.RTWStringValue;
import edu.cmu.ml.rtw.course.TheoKB;


/**
 * Quick demo for 10-709 Fall 2009 to accompany Theo_API.txt, showing examples of the Java
 * version of the same API.
 *
 * This is meant to be compiled and run as a standalone file using the "-course" jar file.
 * If this file is in . and your jar files are in ./lib, then build and run like so:
 *
 * javac -source 1.5 -target 1.5 -encoding UTF-8 -g -O -classpath ".:lib/OntologyLearner-course.jar:lib/log4j-1.2.14.jar" RTWJavaDemo.java
 * java -cp ".:lib/OntologyLearner-course.jar:lib/log4j-1.2.14.jar" RTWJavaDemo <KB directory>
 */
public class RTWJavaDemo {

    public static void main(String[] args) throws Exception {

        // Obtain KB directory and initialize Theo for use
        //
        // The TheoKB object provides all of the functions that we use.  They are all static,
        // meaning that you are limited to using one KB at a time.
        //
        if (args.length != 1) {
            System.out.println("Usage: RTWJavaDemo <KB directory>");
            System.exit(1);
        }
        String kbDirectory= args[0];
        TheoKB.initializeTheo();

        // Note that the Java version of useKB will not accept depth and case-sensistivy parameters.
        // They are fixed at their default values, which means that you cannot use
        // http://rtw.ml.cmu.edu/sslnlp09/ as shown in Theo_API.txt because that KB requires a depth
        // setting of 0 (default is 3).
        //
        // Note also that the Java code does not yet support using a KB over http.  You'll need to
        // get a copy of the KB on your filesystem.
        //
        TheoKB.useKB(kbDirectory);  

        // Here's how to use RTWValue*, the datatypes we use to emulate Matlab's ability to store in
        // any variable string value, number, or list of other variables.
        //
        RTWIntegerValue iv = new RTWIntegerValue(-3);
        RTWRealValue rv = new RTWRealValue(4.5);
        RTWStringValue sv = new RTWStringValue("NO_THEO_VALUE");
        RTWListValue lv = new RTWListValue();

        // Build a simple list
        lv.addValue(iv);
        lv.addValue(rv);
        lv.addValue(sv);
        lv.addValue(new RTWListValue());

        // RTWValue is the abstract parent of all these.  getTypeString allows us to differentiate.
        // This also shows how to turn RTWValue into one of Java's primitive types.
        //
        System.out.println("Checking RTWValue type:");
        RTWValue v = rv;
        if (v.getTypeString().equals("in")) {
            int i = (int)((RTWIntegerValue)v).getVal();  // getVal actually returns a double here.  You can cast it to int or just always use double.
            System.out.println("v is the integer " + i);
        } else if (v.getTypeString().equals("r")) {
            double d = ((RTWRealValue)v).getVal();
            System.out.println("v is the real " + d);
        } else if (v.getTypeString().equals("s")) {
            String s = ((RTWStringValue)v).getVal();
            System.out.println("v is the string " + s);
        } else if (v.getTypeString().equals("l")) {
            List<RTWValue> l = ((RTWListValue)v).getVal();
            System.out.println("l is a list");  // See below for an easier way to print it
        }
            
        // Add something to the list in the list to show list access and nesting.  Note the cast.
        //
        List<RTWValue> l = lv.getVal();
        ((RTWListValue)l.get(3)).addValue(new RTWStringValue("I_M_IN_A_LIST_IN_A_LIST"));

        // TheoKB has a method that will render any RTWValue into a human-readable string
        //
        System.out.println("\nUsing RTWValueToString:");
        String humanString = TheoKB.RTWValueToString(lv);
        System.out.println("The whole value of lv is " + humanString);

        // Now to show some TheoKB methods.

        // isEntity
        System.out.println("\nUsing isEntity:");
        boolean b;
        b = TheoKB.isEntityKB("everything");
        System.out.println("everything is an entity:" + b);
        b = TheoKB.isEntityKB("dizzledong");
        System.out.println("dizzledong is an entity:" + b);

        // getValueKB -- note how we use a space-delimited String where Matlab would use a list
        System.out.println("\nUsing getValueKB:");
        v = TheoKB.getValueKB("source", "mets teamPlaysAgainstTeam candidateValues");
        System.out.println("{mets, teamPlaysAgainstTeam, candidateValues, source} = " + TheoKB.RTWValueToString(v));

        // getEntitySlotsKB
        System.out.println("\nUsing getEntitySlotsKB:");
        String slot = "mets generalizations candidateValues";         // Space-delimited string where Matlab uses a list
        List<String> slotList = TheoKB.getEntitySlotsKB(slot);
        for (Iterator<String> it = slotList.iterator(); it.hasNext();)
            System.out.println("'" + slot + "' has a slot named " + it.next());

        // printEntity
        System.out.println("\nUsing printEntity:");
        System.out.print(TheoKB.printEntity("mets", 4));               // Up to a depth of 4.  Note that printEntity returns a string already with terminating newlines
        System.out.print(TheoKB.printEntity("mets generalizations"));  // Default depth of 1.  Note the space-delimited string were Matlab uses a list
        slotList.clear();
        slotList.add("generalizations");
        slotList.add("literalString");
        System.out.print(TheoKB.printEntity("mets", 4, slotList));  // Limit the set of slots to emit
        System.out.print(TheoKB.printEntity("dizzledong"));         // Empty string when entity does not exist

        // printHierarchy
        System.out.println("\nUsing printHierarchy:");
        slotList.add("specializations");
        System.out.print(TheoKB.printHierarchy("slotSlot", slotList));  // slotList is optional.  By default, no slot vaules are printed.  As with printEntity, we use print instead of println
        
    }
}