function [features] = get_features(img, img_cracks, sigma_0, foreground_mask)
%
% [features] = get_features(img, <img_cracks>, <sigma_0>, <foreground_mask>)
%
%  Gets scale-invariant features from an image.  If occlusion boundary
%  information is also provided (in the form of "cracks" or a foreground
%  mask), the computed features are also background-invariant.  See below
%  for further details.
%
% -------
% INPUTS:
%
%  (Note that there are several other parameters at the top of the code)
% 
%  img: the input image
% 
%  img_cracks: (optional) A matrix the same size as the input image with
%     boundary information encoded as pixel "cracks" (the edges between
%     individuatal pixels).  This should be of type UINT8. Each of the 8
%     bits signifies whether a crack is present in a given direction:
%
%       Bit:        1    2   3     4     5      6      7         8
%       Direction:  Up Down Left Right UpLeft UpRight DownLeft DownRight
%
%     If no crack/boundary information is available, set this to [], or
%     leave it off.
%
%  sigma_0: (optional) the base level smoothing used to build the Gaussian
%     (and Laplacian) pyramids.  Default is 1.
%
%  foreground_mask: (optional) a binary mask with 1's at pixel locations
%     designated to be foreground.  Use this OR img_cracks but but not both
%     (i.e. set img_cracks to [] if providing a foreground mask).  The
%     foreground mask will be converted to crack information by the code.
%
% --------
% OUTPUTS:
%
%  If no ouput is requested, the features are displayed (using
%  DrawFeatures) overlaid on the input image, and with any supplied crack
%  information.
%
%  features: a struct array containing all information about the features.
%     Each element of the array contains the following fields:
%
%       'x' and 'y'   - the feature's (possibly floating point) coordinates
%       'scale'       - the detected scale of the feature
%       'scale_index' - the index into the level of the pyramid octave
%                       corresponding to the feature's scale (probably not 
%                       terribly useful outside this function)
%       'angle'       - the feature's dominant orienation angle in radians
%       'descriptor'  - the feature's descriptor, stored as UINT8's. 
%
% -------------
% DEPENDENCIES:
%  
%  Note that this code relies on _many_ other functions, several of them
%  mex functions.  These include (but may not be limited to):
%   
%      mask2cracks
%      ScaleSpaceRepresentation
%        mexDiffuse_cracks
%        myGaussian
%        downsample_cracks
%      mexLocalMax
%      refine_int_pts
%      eliminate_edge_responses
%      CreateDescriptors
%        mexFastMarchMask_cracks
%        CreateGaussianMask
%        mexWeighted_hist
%        smoothgradient
%        mexInterp2
%        mexComputeDescriptor_simple
%
%
% ---------------
% Andrew Stein
%
% Please reference:
%
% "Incorporating Background Invariance into Feature-Based Object
%  Recognition"
%     A. Stein and M. Hebert
%     IEEE Workshop on Applications of Computer Vision (WACV), January 2005.
%




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARAMETERS:
%

% Turn on displaying of results (if no output arguments are requested, this
% will be turned on by the code)
DISPLAY = false;
    
% ==== DETECTION ====

do_image_doubling = false;

% number of levels per scale octave (3 or 4 usually)
num_samples_per_scale_octave = 3;    

% smoothing of each level of the gaussian pyramid before building the
% scale-space representation and finding local maxima (to turn off, set
% this to empty)
scale_space_pre_smooth = [];

% "low-contrast" threshold (Lowe claims to use 0.03)
laplacian_threshold = 0.01;   

% Max ratio of eigenvalues of the Hessian around an interest point to allow
% before removing it for being a response to an edge.
eigenvalue_ratio_thresh = 8;

% % Smoothing timestep (needs to be less than .25)
% timestep = 0.1;

% % ratio of derivative scale to integration scale 
% deriv_scale_factor = 1;              


% ==== DESCRIPTION ====

% descriptor samples:
%  number of samples (in each dimension) that are taken from within the
%  nearby window patch to generate the descriptor.  Lowe uses 16. My code
%  needs this to be an odd number.
descriptor_samples = 17;

% % this is sketchy: the characteristic scale of a keypoint gets multiplied 
% % by this factor to determine the patch size for that keypoint (where the
% % patch size is the size of the patch of samples extracted to create that 
% % keypoint's descriptor)
% % evidently in Lowe's code, the _diameter_ of the patch is:
% %  scale*MagFactor*1.414*(IndexSize+1), where MagFactor is 3-5 and
% %     IndexSize = 4.  Thus the multiplier turns out to be about 20 (for _diameter_).
% %      But i think this assumes the image was doubled first, so i should
% %      probably divide by two since I don't do that.  So then I get 5 (for _radius_).
% patch_size_multiplier = 3;  
% %patch_size_multiplier = 3*descriptor_samples/2;

                        
% % Two points nearby spatially have to differ by this percentage
% % (i.e. (1-scale_ratio_threshold) < scale1/scale2  or 
% %       scale1/scale2 > (1+scale_ratio_threshold)
% scale_ratio_threshold = .25; 



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INPUT ARGUMENTS
%

if(nargout==0)
    DISPLAY = true;
end

if(DISPLAY)
    figure(gcf), hold off
    imagesc(img), axis image, colormap(gray)
    hold on
    drawnow
end

% if we're given a color image, make it grayscale
if(size(img,3) == 3)
    disp('Converting color image to grayscale.')
    img = rgb2gray(img);
end

if(isa(img, 'uint8'))
    disp('Converting UINT8 image to DOUBLE and scaling to be between 0 and 1.')
    img = double(img)/255;
end

max_val = max(img(:));
if(max_val > 1)
    if(max_val <= 255)
        disp('Image is DOUBLE, but was likely a UINT8 originally.  Scaling by 255.')
        img = img / 255;
    else
        disp(['Image is DOUBLE and max value is ' num2str(max_val) '. Scaling by max value.']);
        img = img / max_val;
    end
end

% If we're not given any edges, use plain old SIFT
if(nargin==1 | isempty(img_cracks))
    disp('No boundary edges provided, regular SIFT will be used.')
    img_cracks = [];
    
elseif(~all(size(img_cracks) == size(img)))
    error('Input image and crack image must be the same size!');
end

% If user did not supply boundary edge information, but did supply a
% foreground mask, use the mask to get boundary edges
if(isempty(img_cracks) & nargin > 3)
    img_cracks = mask2cracks(foreground_mask);
end

if(nargin < 4)
    foreground_mask = [];
end

if(~isempty(img_cracks) & ~isa(img_cracks, 'uint8'))
    disp('Converting boundary edges to UINT8.')
    img_cracks = uint8(img_cracks);
end

% Min and max smoothing for Gaussian pyramid creation:
if(nargin<3)
    sigma_0 = 1;
%     sigma_min_max = 20;
end

% Adjust parameters if we're doubling the image before processing
if(do_image_doubling)
    sigma_0 = 2*sigma_0;
%     sigma_min_max = 2*sigma_min_max;
    patch_size_multiplier = 2*patch_size_multiplier;
    scale_space_pre_smooth = 2*scale_space_pre_smooth;
end

[nrows, ncols] = size(img);

% If after all that input checking, we have crack information, and display
% is turned on, draw the cracks:
if(DISPLAY && ~isempty(img_cracks))
    drawcracks(img_cracks, 'r', 2);
end


% Definitions for crack encoding:
DOWN  = 2;
LEFT  = 4;
RIGHT = 8;
UPLEFT    = 16;
UPRIGHT   = 32;
DOWNLEFT  = 64;
DOWNRIGHT = 128;

                            
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% FEATURE POINT DETECTION:
%

if(do_image_doubling)
    tic
    img = imresize(img, 2, 'bilinear');
    if(isempty(foreground_mask))
        img_cracks = imresize(img_cracks,2,'nearest');
    else
        foreground_mask = imresize(foreground_mask, 2, 'nearest');
        img_cracks = mask2cracks(foreground_mask);
    end
        
    
    [nrows, ncols] = size(img);
    disp(['Done doubling the image and edge map. (' num2str(toc) ' seconds)'])
end

% tic
% % Create a Gaussian pyramid by heat diffusion.  Also handle the
% % downsampling of edges and compute derivative magnitudes and orientations
% % at each level:
% [gaussian_pyramid, crack_pyramid, mag_pyramid, orient_pyramid, scales_list] = ...
%     diffusion_pyramid(img, img_cracks, sigma_0, num_samples_per_scale_octave);
% 
% % % Downsample the foreground mask as well, if one is available:
% % if(~isempty(foreground_mask))
% %     foreground_mask_pyramid{1} = foreground_mask;
% %     for(i_octave = 2:length(gaussian_pyramid))
% %         foreground_mask_pyramid{i_octave} = imresize(foreground_mask_pyramid{i_octave-1}, size(gaussian_pyramid{i_octave}(:,:,1)), 'nearest');
% %     end
% % end
% disp(['Done smoothing and creating pyramids. (' num2str(toc) ' seconds)'])
% 
% tic
% % Now compute a differnce-of-Gaussian pyramid to approximate a Laplacian
% % pyramid and find local extrema in that pyramid to determine potential
% % interest point locations and scales.  (Note that this function will not
% % return interest points at the boundaries of the image). This requires an
% % adjustment to the list of scales: assume the scale of a given level in
% % the laplacian pyramid corresponds to the scale of the higher of the two
% % Gaussian pyramid levels from which it was derived.
% [int_pts, lap_pyramid] = scale_space_interest_points(gaussian_pyramid, crack_pyramid, laplacian_threshold, ...
%     scale_space_pre_smooth);
% scales_list = scales_list(1:end-1);
% disp(['Done building and finding ' num2str(length(vertcat(int_pts{:}))) ' extrema in the Laplacian pyramid. (' num2str(toc) ' seconds)'])

tic
if(isempty(img_cracks))
    [gaussian_pyramid, lap_pyramid, dx_pyramid, dy_pyramid, scales_list] = ScaleSpaceRepresentation(img, [], sigma_0);
    for(i=1:length(gaussian_pyramid))
        crack_pyramid{i} = uint8(zeros(size(gaussian_pyramid{i}(:,:,1))));
    end
else
    [gaussian_pyramid, lap_pyramid, crack_pyramid, dx_pyramid, dy_pyramid, scales_list] = ...
        ScaleSpaceRepresentation(img, img_cracks, sigma_0);
end

% % UPDATE, 5/23/05: Throw out small-scale features and use derivative info
% % from one octave below to build descriptors.
% lap_pyramid = lap_pyramid(2:end);
% crack_pyramid = crack_pyramid(2:end);


% Compute pyramids of gradient magnitude and orientation.  Also find
% potential interest points (extrema above a threshold) in the Laplacian
% pyramid:
for(i = 1:length(lap_pyramid))

    mag_pyramid{i} = sqrt(dx_pyramid{i}.^2 + dy_pyramid{i}.^2);
    
    % The negative sign on the dy values below MUST NOT BE DELETED!!!! It is
    % absolutely crucial for rotation invariance and has to do with the
    % fact that image coordinate system's y axis is flipped: (0,0) is in
    % the upper left hand corner and positive y values point down.  (Thus
    % this coordinate system is not right-handed.)
    orient_pyramid{i} = 180/pi * atan2(-dy_pyramid{i}, dx_pyramid{i}) + 180;
    
    extrema = mexLocalMax(abs(lap_pyramid{i}));
    
    % Store those extrema that are also above the threshold 
    int_pts{i} = find(extrema & abs(lap_pyramid{i}) > laplacian_threshold);
end
disp(['Done building pyramids and finding ' num2str(length(vertcat(int_pts{:}))) ' initial interest points. (' num2str(toc) ' seconds)'])


if(isempty(vertcat(int_pts{:})))
    disp('No scale space extrema found... Nothing to do!');
    features = [];
    return;
end

tic
% Convert the interest points from 3D indices to spatial positions and
% scales in each octave.  At the same time, remove any interest points that
% fall _on_ image edges or in a region marked as background (if foreground mask is available):
for(i_octave = 1:length(int_pts))
    if(~isempty(int_pts{i_octave}))
        [y{i_octave},x{i_octave},s{i_octave}] = ind2sub(size(lap_pyramid{i_octave}), int_pts{i_octave});
                     
        %%% This is now done by scale_space_interest_points()
        % Not using scale_space_interest_points:
        temp_index = sub2ind(size(crack_pyramid{i_octave}), y{i_octave}, x{i_octave});
        to_remove = find(crack_pyramid{i_octave}(temp_index));
                
        %         if(~isempty(foreground_mask))
        %             to_remove = [to_remove; find(foreground_mask_pyramid{i_octave}(temp_index) == 0)];
        %         end
        
        int_pts{i_octave}(to_remove) = [];
        x{i_octave}(to_remove) = [];
        y{i_octave}(to_remove) = [];
        s{i_octave}(to_remove) = [];
    else 
        x{i_octave} = [];
        y{i_octave} = [];
        s{i_octave} = [];   
        scale{i_octave} = [];
    end
end

% Make adjustments to the potential interest points before computing
% descriptors:
for(i_octave = 1:length(x))

    if(~isempty(x{i_octave}))
        % Do subvoxel refinement:
        [x{i_octave},y{i_octave},s{i_octave},scale{i_octave}] = ...
            refine_int_pts(x{i_octave}, y{i_octave}, s{i_octave}, lap_pyramid{i_octave}, scales_list);
        
        % Remove interest points that are just responses to edges:
        [x{i_octave},y{i_octave},s{i_octave}, to_remove] = eliminate_edge_responses( ...
            x{i_octave},y{i_octave},s{i_octave}, lap_pyramid{i_octave}, crack_pyramid{i_octave}, eigenvalue_ratio_thresh);
        scale{i_octave}(to_remove) = [];
        
%         % UPDATE 5/23/05: Handle the fact that we're now finding interest points in
%         % a different octave of the laplacian pyramid than the octave of the
%         % derivative pyramids we're building descriptors in.  So the coordinates
%         % need to be scaled.
%         x{i_octave} = 2*(x{i_octave}-1) + 1;
%         y{i_octave} = 2*(y{i_octave}-1) + 1;
%         scale{i_octave} = 2*scale{i_octave};
        
    else
        scale{i_octave} = [];
    end
    
    
    
%     octave_sizes(i_octave,:) = size(lap_pyramid{i_octave});
    

end


% % Get rid of interest points that are too close together in scale or space:
% [x,y,s,scale] = prune_int_pts(x, y, s, scale, octave_sizes, scales_list, scale_ratio_threshold);

% % For using detections from one of Lowe's .key files:
% % keyfilename = 's:\vmr50\stein\BSIFT\img_rotated.key';
% keyfilename = 's:\vmr50\stein\BSIFT\totoro.key';
% disp(['READING SIFT KEY FILE: ' keyfilename]);
% x = {};
% y = {};
% scale = {};
% s = {};
% sift_keypoints = ReadKeyFile(keyfilename);
% if(do_image_doubling)
%     x_sift = 2*[sift_keypoints(:).x];
%     y_sift = 2*[sift_keypoints(:).y];
%     scale_sift = 2*[sift_keypoints(:).scale];
% else
%     x_sift = [sift_keypoints(:).x];
%     y_sift = [sift_keypoints(:).y];
%     scale_sift = [sift_keypoints(:).scale];
% end
% angles_sift = [sift_keypoints(:).angle];
%     
% for(i=1:length(x_sift))
%     ctr = 1;
%     temp = scale_sift(i);
%     while(temp > scales_list(end))
%         temp = temp/2;
%         ctr = ctr + 1;
%     end
%     octave(i) = ctr;
%     
% end
% num_octaves = max(octave);
% x = cell(1,num_octaves);
% y = x;
% scale = x;
% s = x;
% angle = x;
% for(i=1:length(x_sift))
%     factor = 2^(octave(i)-1);
%    
%     scale{octave(i)}(end+1) = scale_sift(i) / factor;
%     [temp, s{octave(i)}(end+1)] = min(abs(scale{octave(i)}(end) - scales_list));
%     x{octave(i)}(end+1) = (x_sift(i)-1) / factor + 1;
%     y{octave(i)}(end+1) = (y_sift(i)-1) / factor + 1;
%     angle{octave(i)}(end+1) = angles_sift(i);
% end

disp(['Done refining and pruning. Initially found ' num2str(length([x{:}])) ' interest points. (' num2str(toc) ' seconds)'])

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% DESCRIPTOR CREATION:
%

tic
features = [];
for(i_octave = 1:length(x))
  
    if(~isempty(x{i_octave}))
        % Create a descriptor for each interest point (or multiple ones if there
        % are more than one dominant orientations) and return the full set
        % of features
        features{i_octave} = CreateDescriptors(x{i_octave},y{i_octave},s{i_octave},scale{i_octave}, ...
            mag_pyramid{i_octave}, orient_pyramid{i_octave}, crack_pyramid{i_octave}, descriptor_samples);%, angle{i_octave});
        
        % Now adjust the positions and scales to be relative to the original
        % image size:
        if(i_octave > 1)
            for(i=1:length(features{i_octave}))
                factor = 2^(i_octave-1);
                features{i_octave}(i).x = (features{i_octave}(i).x-1) * factor + 1;
                features{i_octave}(i).y = (features{i_octave}(i).y-1) * factor + 1;
                features{i_octave}(i).scale = features{i_octave}(i).scale * factor;
            end
        end
    end
    
end

if(~isempty(features))
	% Concatenate all the interest points from each octave into one list:
	features = [features{:}];

	% % Prune interest points that are too close in position and/or scale
	% features = prune_int_pts(features, [nrows ncols], scale_ratio_threshold);

	disp(['Done computing descriptors.  Total interest points = ' num2str(length(features)) '. (' num2str(toc) ' seconds)'])

	% If we doubled the image, halve the locations and scales of each interest
	% point before returning
	if(do_image_doubling)
	    for(i=1:length(features))
        	features(i).x = features(i).x/2;
	        features(i).y = features(i).y/2;
        	features(i).scale = features(i).scale/2;
	    end
	end
end

if(DISPLAY)
    % Display interest points:
    if(length(features)>0)
        DrawFeatures(features);
    end
end
