function [img_sampled, crack_sampled, varargout] = downsample_cracks(img, crack_img, sigma, varargin)
%
% [img_sampled, crack_sampled, <other_img_sampled, ...>] = ...
%     downsample_cracks(img, crack_img, <other_img, ...>)
%
%  Downsamples and image and it's associated cracks by a factor of 2.
%
%  If extra images (of the same size as img) are provided, they will be
%  downsampled as well.
%
% ------------
% Andrew Stein
%

% Force the input to have even dimensions
[oldrows,oldcols] = size(img);
if(mod(oldrows,2))
    img = img(1:end-1,:);
    crack_img = crack_img(1:end-1,:);
end
if(mod(oldcols,2))
    img = img(:,1:end-1);
    crack_img = crack_img(:,1:end-1);
end

if(nargin >= 3 && ~isempty(sigma))
    timestep = 0.05;
    img = mexDiffuse_cracks(img, crack_img, sigma^2/(2*timestep), timestep);
    %     g = myGaussian(sigma);
%     img = imfilter( imfilter( img, g, 'replicate'), g', 'replicate');

end

% Resampling the image is simple: Just take every other pixel
img_sampled = img(1:2:end,1:2:end);
[nrows,ncols] = size(img_sampled);

% If other images are provided (such as derivatives), downsample those too:
for(i=1:length(varargin))
    varargout{i} = varargin{i}(1:2:size(img,1),1:2:size(img,2));
end

% Resample the cracks by looking at the pixels between the ones used in
% resampling the image and see if there are edges in those cracks:
crack_sampled = uint8(zeros(nrows, ncols));

UP_BIT = 1;
DOWN_BIT = 2;
LEFT_BIT = 3;
RIGHT_BIT = 4;
UPLEFT_BIT    = 5;
UPRIGHT_BIT   = 6;
DOWNLEFT_BIT  = 7;
DOWNRIGHT_BIT = 8;

% Clear any problematic bits around the edges of the image
crack_img(end,:) = 0;
crack_img(:,end) = 0;

% If the pixel row between the samples has _either_ (but not both) a crack
% above or below it, set a crack between the samples
index = find(xor(bitget(crack_img(2:2:end,1:2:end), UP_BIT), bitget(crack_img(2:2:end,1:2:end), DOWN_BIT)));
crack_sampled(index) = bitset(crack_sampled(index), DOWN_BIT);
crack_sampled(index+1) = bitset(crack_sampled(index+1), UP_BIT);

% If the pixel column between the samples has _either_ (but not both) a crack
% to the left or right of it, set a crack between the samples
index = find(xor(bitget(crack_img(1:2:end,2:2:end), LEFT_BIT), bitget(crack_img(1:2:end,2:2:end), RIGHT_BIT)));
crack_sampled(index) = bitset(crack_sampled(index), RIGHT_BIT);
crack_sampled(index+nrows) = bitset(crack_sampled(index+nrows), LEFT_BIT);

% If the pixel row between the samples has _both_ a crack above and below
% it, only set a crack between the samples if there is not also a crack to
% the left or right.  Repeat for the left/right case as well.
% This avoids problems in the case below, where X is a sample:
%
%    Original:                    After sampling, we get little vertical
%                                  cracks sticking out:
%
%       -----   -----
%     X |   | X |   | X      ===>    X | X | X
%   -----   -----   -----           ------------
index = find(bitget(crack_img(2:2:end,1:2:end), UP_BIT) & bitget(crack_img(2:2:end,1:2:end), DOWN_BIT) & ...
    ~(bitget(crack_img(2:2:end,1:2:end), LEFT_BIT) | bitget(crack_img(2:2:end,1:2:end), RIGHT_BIT)) );
crack_sampled(index) = bitset(crack_sampled(index), DOWN_BIT);
crack_sampled(index+1) = bitset(crack_sampled(index+1), UP_BIT);

index = find(bitget(crack_img(1:2:end,2:2:end), LEFT_BIT) & bitget(crack_img(1:2:end,2:2:end), RIGHT_BIT) & ...
    ~(bitget(crack_img(1:2:end,2:2:end), UP_BIT) | bitget(crack_img(1:2:end,2:2:end), DOWN_BIT)) );
crack_sampled(index) = bitset(crack_sampled(index), RIGHT_BIT);
crack_sampled(index+nrows) = bitset(crack_sampled(index+nrows), LEFT_BIT);

% Also need to preserve diagonal crack information:
index = find(xor(bitget(crack_img(2:2:end,2:2:end), UPLEFT_BIT), bitget(crack_img(2:2:end,2:2:end), DOWNRIGHT_BIT)));
crack_sampled(index) = bitset(crack_sampled(index), DOWNRIGHT_BIT);
crack_sampled(index+1+nrows) = bitset(crack_sampled(index+1+nrows), UPLEFT_BIT);

index = find(xor(bitget(crack_img(2:2:end,2:2:end), DOWNLEFT_BIT), bitget(crack_img(2:2:end,2:2:end), UPRIGHT_BIT)));
crack_sampled(index+1) = bitset(crack_sampled(index+1), UPRIGHT_BIT);
crack_sampled(index+nrows) = bitset(crack_sampled(index+nrows), DOWNLEFT_BIT);