#!/usr/bin/perl
#
# genconfig.pl
#
# Initial Author: Jeff Pang <jeffpang+441@cs.cmu.edu>
# Class: 15-441 (Spring 2004)
#
# This script takes in a specified network description and generates the
# appropriate config files you need for each node in the network. You should
# run this script with your goup number, the host you will run your nodes on
# (e.g., localhost), and the specification of the network, like this:
#
# ./genconfig.pl <grp_num> <hostname> <network_spec>
#
# Your group number is used to assign ports to your nodes. If you find that
# some of the ports are in use, try a random group number 120 < x < 4500.
#
# The format if the network specification is the network graph in
# adjacency list format. In otherwords, each line is of the format
# 'nodeID neighborID1 neighborID2 ...' where nodeID is the current node
# and neighborIDi are its neighbors. For example, take the following graph:
#
#             1 ----- 2 --- 5 ---- 6   
#              \     /      |
#               \   /       |
#                 3 ------- 4
#
# This graph would have the network specification:
#
# 1 2 3
# 2 1 3 5
# 3 1 2 4
# 4 3 5
# 5 2 4 6
# 6 5
#
# Make sure that the edges you specify are bi-directional! This script will
# NOT check that your network is valid --- so make sure your network spec
# is correct before using it!
#
# It outputs the config file for node i as nodei.conf.
#
###############################################################################

use strict;

if (@ARGV != 3) {
    print STDERR "usage: ./genconfig.pl <grp_num> <hostname> <network_spec>\n";
    exit(1);
}

our $grp     = $ARGV[0];
our $host    = $ARGV[1];
our $netspec = $ARGV[2];

if ($grp < 0 || $grp > 4500) {
    die "ERROR: bad group number $grp\n";
}
# use the port range as defined in the handout
our $port = 20000 + $grp*100;

our %portmap;

open(SPEC, "<$netspec") or die "ERROR: can't open $netspec: $!\n";
my $lineno = 1;
my $line;
while ($line = <SPEC>) {
    chomp $line;
    next if ! $line; # skip blank lines
    my ($id, @neighbors) = split(/\s+/, $line);
    die "ERROR: missing nodeID on line $lineno\n" if ! defined $id;
    map { die "ERROR: bad nodeID $_ on line $lineno\n" if ($_ !~ /^\d+$/);
          $_; } ($id, @neighbors);
    open(CONF, ">node$id.conf") or die "ERROR: can't create node$id.conf: " .
	"$!\n";
    foreach my $i ($id, @neighbors) {
	my ($rport, $lport, $iport) = getports($i);
	print CONF "$i $host $rport $lport $iport\n";
    }
    close(CONF);

    $lineno++;
}
close(SPEC);


sub getports {
    my $id = shift;
    if (! defined $portmap{$id}) {
	$portmap{$id} = [ $port++, $port++, $port++ ];
    }
    return @{$portmap{$id}};
}

