/************************************
 * clients.c
 * 
 * Contains definitions for client data, and can connect to and keep track of clients.
 * Also knows how to deal with SSL.
 ************************************/

#include "clients.h"
#include "client_read.h"
#include "client_write.h"
#include "util.h"

struct clientnode *new_clientnode() {
  struct clientnode *temp = (struct clientnode*)malloc(sizeof(struct clientnode));
  temp->data.socklen = sizeof(struct sockaddr_in);
  temp->data.requests = NULL;
  temp->data.reqlen = 0;
  temp->data.readpending = 0;
  /*temp->data.keepalive = 1;*/
  temp->data.lastseen = 0;
  temp->data.isSSL = 0;
  temp->data.ssl_connection = NULL;
  temp->data.ssl_client_bio = NULL;
  return temp;
}

void delete_clientnode(struct clientnode *client) {
  if (client->data.requests) free(client->data.requests);
  free(client);
}

int write_client(struct clientdata *client, char *buf, int len) {
  int retval;
  if (!client->isSSL) {
    retval = write(client->fd, buf, len);
  } else {
    retval = BIO_write(client->ssl_client_bio, buf, len);
    if (retval != -1) {
      retval = BIO_flush(client->ssl_client_bio);
    }
  }
  return retval;
}

void close_client(struct clientdata *client) {
  if (client->isSSL) {
    ssl_close_connection( client->fd, client->ssl_connection, client->ssl_client_bio );
  } 
  close(client->fd);
}

struct requestinfo *new_requestinfo() {
  struct requestinfo *request = (struct requestinfo*)malloc(sizeof(struct requestinfo));
  request->host = NULL;
  request->URI = NULL;
  strcpy(request->contenttype, "text/html");
  request->content = NULL;
  request->relpath = NULL;
  request->keepalive = 0;
  request->contentlength = 0;
  request->httpversion11 = 0;
  request->requesttype = REQUEST_UNFIN;
  request->responsetype = 0;
  return request;
}

void delete_requestinfo(struct requestinfo *request) {
  if (request->host != NULL)    free(request->host);
  if (request->URI != NULL)     free(request->URI);
  if (request->content != NULL) free(request->content);
  if (request->relpath != NULL) free(request->relpath);
  free(request);
}

struct clientnode *accept_connection(int serverfd) {
  struct clientnode *clinode = new_clientnode();

  clinode->data.fd = accept(serverfd, (struct sockaddr*) &clinode->data.sock, &clinode->data.socklen);
  clinode->data.lastseen = time(NULL);
  if (debug) printf("\n===================================\nconnection (%d)\n", clinode->data.fd);
  if (debug) printerror(__FILE__,__LINE__);
  if (clinode->data.fd == -1) {
    if (debug) fprintf(stderr, "Couldn't accept connection.  Check if firewall prevents it?\n");
    /* why would accept fail? */
    return NULL;
  } else {
    /* we accepted a connection, add it to the list */
    client_add(&clients, clinode);
    return clinode;
  }
}

struct clientnode *ssl_accept_connection(int serversslfd) {
  struct clientnode *clinode = accept_connection( serversslfd );
  int ssl_error_code;
  if (clinode == NULL) return NULL;

  /* Here we make a new buffered I/O object attached to the client's 
  socket which will be used for reading and writing to the client.  We
  also make a new SSL connection, and attach the BIO object to that 
  connection. */
  clinode->data.ssl_client_bio = BIO_new_socket( clinode->data.fd, BIO_NOCLOSE );
  clinode->data.ssl_connection = SSL_new( g_ssl_context );
  SSL_set_bio( clinode->data.ssl_connection, clinode->data.ssl_client_bio, clinode->data.ssl_client_bio );

  if ( (ssl_error_code = SSL_accept(clinode->data.ssl_connection)) <= 0) {
    if (debug) printf("couldn't complete SSL handshake (accept)\n");
    // leave no trace!!
    client_remove(&clients, clinode);
    delete_clientnode(clinode);
    return NULL;
  }

  clinode->data.isSSL = 1;

  if (fcntl(clinode->data.fd, F_SETFL, O_NONBLOCK) == -1) {
    if (debug) printf("couldn't set socket to be nonblocking.\n");
    if (debug) printerror(__FILE__,__LINE__);
    client_remove(&clients, clinode);
    delete_clientnode(clinode);
    return NULL;
  }

  if (debug) printf("SSL handshaking completed.\n");

  return clinode; 
}

void kill_client(struct clientnode *client);

void process_client(struct clientnode *client) {
  int retval;

  retval = serve_client(&client->data);
 
  if (retval == CLOSED || retval == CLOSE) {
    if (debug) printf("connection closed\n");
    /* it wasn't closed by the client, we'll do it */

    kill_client(client);
  } 
}

void timeout_client(struct clientnode *client) {
  if (debug) printf("Client %d idle %f seconds or reqlen too large (%d), terminating connection.\n", 
    client->data.fd, difftime(time(NULL),client->data.lastseen), client->data.reqlen);
  kill_client(client);
}

void kill_client(struct clientnode *client) {
  close_client(&client->data);
  if (debug) printerror(__FILE__,__LINE__);
  client_remove(&clients, client);
  delete_clientnode(client);
}

int serve_client(struct clientdata *client) {
  /* read one request from the socket.  if it's a Connection: close, 
   * deal with it accordingly */
  struct requestinfo *request = new_requestinfo();
  int keepalive = 1, connstatus = STILL_ALIVE;

  if (client->readpending) {
    if (getclientrequest(client) == CLOSED) return CLOSED;
    client->readpending = 0;
  }

  /* TODO: if the request keeps being unfinished, maybe it's malformed? */
  while ((request->requesttype = loadheader(client, request)) != REQUEST_UNFIN) {
    /* serve up the page */
    if (request->requesttype == REQUEST_UNSUP) {
      if (debug) printf("request type unsupported\n");

      request->responsetype = 501;
      connstatus = serve_page_error(client, request);

    } else if (request->requesttype == REQUEST_MALFORMED) {
      if (debug) printf("bad request\n");

      request->responsetype = 400;
      connstatus = serve_page_error(client, request);
      /* can't recover from a bad request; die */
      connstatus = CLOSE;

    } else if (!request->httpversion11) {
      if (debug) printf("versions other than HTTP/1.1 not supported\n");

      request->responsetype = 505;
      connstatus = serve_page_error(client, request);

    } else {
      if (debug) printf("serving page...\n");

      request->responsetype = 200;
      connstatus = serve_page(client, request);
    }

    keepalive=request->keepalive;
    if (keepalive == 0 || connstatus != STILL_ALIVE) break;
  }


  delete_requestinfo(request);

  /* if the request doesn't say keep-alive and the operation completed okay, set status to CLOSED */
  /* don't want to close the connection if it's already been closed on the other end */
  if (!keepalive && connstatus != CLOSED) {
    connstatus = CLOSE;
  }

  if (keepalive) {
    if (debug) printf("keep connection alive\n");
  } else {
    if (debug) printf("don't keep connection alive\n");
  }

  return connstatus;
}


int getreadfds(fd_set *readfds, int serverfd, int serversslfd, struct clientnode *root) {
  int maxfd;
  FD_ZERO(readfds);
  if (serverfd)    FD_SET(serverfd, readfds);
  if (serversslfd) FD_SET(serversslfd, readfds);
  maxfd = max(serverfd, serversslfd);
  while (root != NULL) {
    FD_SET(root->data.fd, readfds);
    maxfd = max(maxfd, root->data.fd);
    root = root->next;
  }
  return maxfd;
}

int getmaxfd(struct clientnode *root) {
  int maxfd = 0;
  while (root != NULL) {
    maxfd = max(maxfd, root->data.fd);
    root = root->next;
  }
  return maxfd;
}

void client_add(struct clientnode **root, struct clientnode *newnode) {
  struct clientnode *curr;
  if (*root == NULL) {
    /* list is empty */
    *root = newnode;
    newnode->next = NULL;
  } else {
    /* put it at the end */
    curr = *root;
    while (curr->next != NULL) {
      curr = curr->next;
    }
    curr->next = newnode;
    curr->next->next = NULL;
  }
}

int client_remove(struct clientnode **root, struct clientnode *node) {
  struct clientnode *curr;
  if (*root == NULL) {
    /* they're crazy, ignore them */
  } else {
    /* find it and remove it */
    if (*root == node) {
      /* it's the first thing in the list, so move root up one */
      *root = (*root)->next;
      return 1;
    } else {
      /* it's deeper in the list */
      curr = *root;
      while (curr->next != node && curr->next != NULL) {
        curr = curr->next;
      }
      if (curr->next != NULL) {
        /* found it! take it out of the list. */
        curr->next = curr->next->next;
        return 1;
      } else {
        /* they asked to remove something that's not in the 
         * list, let it fall through and return 0 */
      }
    }
  }
  return 0;
}
