

import java.io.File;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import javax.vecmath.Vector3d;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


/**
 * This reads an XML file to load a Simulation. It is based upon the XMLParser that
 * Peter Kim wrote. Minor changes were made to the format. The format is specified in
 * DessaXML.xsd. 
 * 
 * @author Peter Kim
 * @author Blake Sweeney
 * @version 1.4
 * 
 * TODO Implement Queue loading
 */
public class XMLReader {

	/** The Simulation constructed by this reader*/
	private Simulation simulation;

	/** A flag to detect weather or not this is a reloaded XML*/
	private boolean reloaded = false;

	/** Count for the Subunit ID */
	private int suId; 

	/** Counter for BindingSite numbers*/
	private int bsID;

	/** A HashMap of the names of BindingSiteTypes mapping to a Vector of their partners*/
	private HashMap<String, BindingSiteType> bstMap;

	/** A Vector of all SubunitTypes created by this reader*/
	private Vector<SubunitType> subts = new Vector<SubunitType>();

	/**
	 * A basic constructor
	 */
	public XMLReader() {
		simulation = null;
	}

	/**
	 * Reads file from given path and parses it to load simulation.
	 * 
	 * @param path - Path of user's input XML file
	 */
	public XMLReader(String path) throws Exception {

		File F = new File(path);
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		Document doc = null;

		try {
			DocumentBuilder db = dbf.newDocumentBuilder();
			doc = db.parse(F);
		} catch (Exception e) {
			throw new Exception(e.getMessage());
		}

		readSystem(doc.getElementsByTagName("System"));

		Element sysElem = (Element) doc.getElementsByTagName("System").item(0);

		//Solution soln = buildSolution(sysElem.getElementsByTagName("Solution"));
		bstMap = new HashMap<String, BindingSiteType>();
		
		buildBSTMap(sysElem.getElementsByTagName("BindingSiteTypes"));

		subts = buildSubunitTypes(sysElem.getElementsByTagName("SubunitTypes"));

		HashMap<String, HashMap<String, Double>> conformationalSwitchMap = 
			buildConformationalSwitchingTime(sysElem.getElementsByTagName("ConformationalSwitch"));

		Vector<Assembly> assemblies = buildAssemblies(sysElem.getElementsByTagName("Assemblies"));

		simulation = new Simulation(assemblies, 0.0, bstMap, conformationalSwitchMap/*, bstMap, soln*/);
	}

	/**
	 * Returns the Simulation generated by this XMLReader
	 * 
	 * @return Simulation - the Simulation generated by this XMLReader
	 */
	public Simulation getSim()
	{ return simulation; }

	/**
	 * Reads the System Element to insure the XML is of correct format and sets up
	 * for reading.
	 * 
	 * @param sys - The Node list that is all Elements with the tag System
	 */
	private void readSystem(NodeList sys) 
	{
		Element sysElem = (Element) sys.item(0);

		// Reloaded is being checked for saving and reloading feature for XML.
		reloaded = sysElem.getAttribute("reloaded").equals("yes");

		Test.simType = sysElem.getAttribute("type");
	}

	/**
	 * This method builds a Solution based upon the Element in the XML file
	 * 
	 * @param solutionElem - The solution Element to build from
	 * @return Solution - The Solution created.
	 */
	/*private Solution buildSolution(NodeList sol) {

		Element solutionElem = (Element) sol.item(0);

		Double temperature = new Double(solutionElem.getAttribute("temp"));
		Double pH = new Double(solutionElem.getAttribute("pH"));
		Double volume = new Double(solutionElem.getAttribute("volume"));

		return new Solution(temperature.doubleValue(), pH.doubleValue(), volume.doubleValue());
	}*/ 



	/**
	 * This method builds a Vector of all BindingSiteTypes based upon the xml
	 * 
	 * @param bsts - The Nodelist from the <BindingSiteTypes> tag
	 * @return Vector<BindingSite> - A Vector of all BindingSiteTypes this xml specifies
	 */
	private void buildBSTMap(NodeList bsts) {

		Element bstsElem = (Element) bsts.item(0);
		NodeList bstList = bstsElem.getElementsByTagName("BindingSiteType");
		int bstLen = bstList.getLength();
		int prtLen = 0;
		for (int i = 0; i < bstLen; ++i) {

			Element bstElem = (Element) bstList.item(i);
			String name = bstElem.getAttribute("name");

			//Getting tolerance values
			Element toleranceElem = (Element) bstElem.getElementsByTagName("Tolerance").item(0);

			Double trans = new Double(toleranceElem.getAttribute("translation"));				
			double translation = trans.doubleValue();

			Double rot = new Double(toleranceElem.getAttribute("rotation"));
			double rotation = rot.doubleValue();

			Double bend = new Double(toleranceElem.getAttribute("bending"));
			double bending = bend.doubleValue();

			double [] tolerance = {translation, rotation, bending};

			//Getting Position values
			Element posElem = (Element) bstElem.getElementsByTagName("Pos").item(0);
			Double pX = new Double(posElem.getAttribute("x"));
			Double pY = new Double(posElem.getAttribute("y"));
			Double pZ = new Double(posElem.getAttribute("z"));

			Vector3d pos = new Vector3d(pX.doubleValue(), pY.doubleValue(), pZ.doubleValue());

			//Getting partners
			NodeList partnerList = bstElem.getElementsByTagName("Partner");

			HashMap<String, PartnerProperties> partnerMap = new HashMap<String, PartnerProperties>();
			prtLen = partnerList.getLength();
			for(int j = 0; j < prtLen; ++j)
			{				
				Element partnerElem = (Element) partnerList.item(j);
				String partnerName = partnerElem.getAttribute("name");
				Double angle = new Double(partnerElem.getAttribute("angle"));
				Double bindTime = new Double(partnerElem.getAttribute("bindTime"));
				Double breakTime = new Double(partnerElem.getAttribute("breakTime"));
				Double fastBindTime = new Double(partnerElem.getAttribute("fastBindTime"));
				PartnerProperties pp = new PartnerProperties(angle, bindTime, breakTime, fastBindTime);

				partnerMap.put(partnerName, pp);
			}
			bstMap.put(name,new BindingSiteType(pos, tolerance, name, partnerMap));
		}
	}



	/**
	 * Builds the SubunitTypes based upon the xml specifications
	 * 
	 * @param subts - The NodeList specified by the <SubunitTypes> tag
	 * @return Vector<SubunitType> - A vector of all specified SubunitTypes
	 */
	private Vector<SubunitType> buildSubunitTypes(NodeList subts) {

		Vector<SubunitType> allTypes = new Vector<SubunitType>();

		Element subtsElem = (Element) subts.item(0);
		NodeList subtList = subtsElem.getElementsByTagName("SubunitType");
		int subtLen = subtList.getLength();
		int domLen = 0;
		for (int i = 0; i < subtLen; ++i)
		{
			Element subtElem = (Element) subtList.item(i);
			String name = subtElem.getAttribute("name");

			double subRadius = Test.bindingSiteHeight * Test.subunitToBS;

			NodeList domList = subtElem.getElementsByTagName("Domain");
			Vector<Domain> domains = new Vector<Domain>();

			domains = new Vector<Domain>();
			domLen = domList.getLength();
			for (int j = 0; j < domLen; ++j) {

				Element domElem = (Element) domList.item(j);
				Domain domain = buildDomain(domElem);
				domains.add(domain);
			}

			Element upElem = (Element) subtElem.getElementsByTagName("Up").item(0);
			double posX = Double.parseDouble(upElem.getAttribute("x"));
			double posY = Double.parseDouble(upElem.getAttribute("y"));
			double posZ = Double.parseDouble(upElem.getAttribute("z"));

			Vector3d upVec = new Vector3d(posX, posY, posZ);

			SubunitType subt = new SubunitType(domains, name, Test.subMass, subRadius, upVec);

			allTypes.add(subt);
		}

		return allTypes;
	}



	/**
	 * Builds a Domain based upon the Domain Element from xml
	 * 
	 * @param domElem - the Domain Element to build off of
	 * @return Domain - the Domain specified by the xml
	 */
	private Domain buildDomain(Element domElem) {

		String currentConfName = new String();
		int id = Integer.parseInt(domElem.getAttribute("name")); //domainID++;

		currentConfName = domElem.getAttribute("currentconf");

		NodeList confList = domElem.getElementsByTagName("Conformation");
		HashMap<String,Conformation> conformations = new HashMap<String,Conformation>();
		Conformation currentConf = null;
		int cnfLen = confList.getLength();
		for (int i = 0; i < cnfLen; ++i) 
		{			
			Element confElem = (Element) confList.item(i);
			Conformation conf = buildConformation(confElem);	

			if(conf.getName().equals(currentConfName))
				currentConf = conf;

			conformations.put(conf.getName(),conf);
		}

		NodeList posList = domElem.getElementsByTagName("Pos");

		Element posElem = (Element) posList.item(0);

		double posX = Double.parseDouble(posElem.getAttribute("x"));
		double posY = Double.parseDouble(posElem.getAttribute("y"));
		double posZ = Double.parseDouble(posElem.getAttribute("z"));

		Vector3d pos = new Vector3d(posX, posY, posZ);

		Domain domain = new Domain(conformations, pos, id);
		boolean change = domain.changeConf(currentConf.getName());
		if (change == false) {
			System.out.println("Could not change to the specified current conformation");
			System.exit(-1);
		}

		return domain;
	}



	/**
	 * This method builds a Conformation from the XML element specifying one
	 * 
	 * @param confElem - the Conformation Element in the XML
	 * @return Conformation - the Conformation specified by the XML
	 */
	private Conformation buildConformation(Element confElem) {

		String name = new String();
		double energy = Double.MAX_VALUE;

		name = confElem.getAttribute("name");

		energy = Double.parseDouble(confElem.getAttribute("energy"));

		NodeList bsList = confElem.getElementsByTagName("BindingSite");
		Vector<BindingSite> bss = new Vector<BindingSite>();
		int bsLen = bsList.getLength();
		for(int i = 0; i < bsLen; ++i) 
			bss.add(buildBindingSite((Element) bsList.item(i)));

		return new Conformation(bss, energy, name);
	}




	/**
	 * This method reads the BindingSite tag and then looks at all possible BindingSiteTypes
	 * and creates a new BindingSite as needed.
	 * 
	 * @param bsElem - the xml Element specifying a BindingSite
	 * @return BindingSite - the BindingSite specified by the xml
	 */
	private BindingSite buildBindingSite(Element bsElem) 
	{
		String type = bsElem.getAttribute("type");
		return new BindingSite(type, -1,bstMap.get(type).getBSTPosition());
	}

	/**
	 * This method  reads the conformational switching element and builds a HashMap based off of
	 * them
	 * 
	 * @param conformationsElem
	 * @return HashMap<String, HashMap<String, Double>> - A HashMap that maps from conformation name
	 * to a HashMap of names of its partners and the switching times.
	 */
	private HashMap<String, HashMap<String, Double>> buildConformationalSwitchingTime(NodeList conformationsList) {


		Element conformationsElem = (Element) conformationsList.item(0);

		HashMap<String, HashMap<String, Double>> confTimeMap = new HashMap<String, HashMap<String, Double>>();
		NodeList confTimeList = conformationsElem.getElementsByTagName("ConformationTime");
		int iLen = confTimeList.getLength();
		int jLen = 0;
		for (int i = 0; i < iLen; ++i) {

			Element confTimeElem = (Element) confTimeList.item(i);

			String confName = confTimeElem.getAttribute("name");

			NodeList switchList = confTimeElem.getElementsByTagName("List");
			HashMap<String, Double> map = new HashMap<String, Double>();
			jLen = switchList.getLength();
			for (int j = 0; j < jLen; ++j) {

				Element listElem = (Element) switchList.item(j);
				String listName = listElem.getAttribute("name");
				Double time = new Double(listElem.getAttribute("time"));
				map.put(listName, time);
			}
			confTimeMap.put(confName, map);
		}

		return confTimeMap;	
	}



	/**
	 * This method builds the Assemblies by reading through the Assemblies Element in the
	 * XML and building each Assembly.
	 * 
	 * @param asmsList - the List of all <Assemblies> tags. There should only be one.
	 * @return Vector<Assembly> - A vector of all Assemblies
	 */
	private Vector<Assembly> buildAssemblies(NodeList asmsList) {

		Element asmsElem = (Element) asmsList.item(0);

		NodeList asmList = asmsElem.getElementsByTagName("Assembly");
		Vector<Assembly> assemblies = new Vector<Assembly>();
		int asmNum = 0;
		int asmLen = asmList.getLength();
		int subLen = 0;
		for(int i = 0; i < asmLen; ++i) {

			Element asmElem = (Element) asmList.item(i);
			NodeList subList = asmElem.getElementsByTagName("Subunit");

			String name = asmElem.getAttribute("name");
			int amount = Integer.parseInt(asmElem.getAttribute("amount"));

			for(int j = 0; j < amount; ++j) 
			{				
				if (reloaded) {
					asmNum = Integer.parseInt(name);
				}else{
					++asmNum;
				}

				Vector<Subunit> subs = new Vector<Subunit>();
				subLen = subList.getLength();
				for(int k = 0; k < subLen; ++k) 
				{
					Element subElem = (Element) subList.item(k);
					Subunit sub = buildSubunit(subElem);
					subs.add(sub);
				}
				Assembly asm = new Assembly(subs,asmNum);
				assemblies.add(asm);
			}
		}
		return assemblies;
	}



	/**
	 * This method builds a Subunit for an Assembly based upon the XML element
	 * 
	 * @param subElem - The XML element to build off of
	 * @param asm - The Assembly this Subunit belongs to.
	 * @return Subunit - A subunit constructed off of the XML Element
	 */
	private Subunit buildSubunit(Element subElem) {

		String name = subElem.getAttribute("name");
		double posX = 0;
		double posY = 0;
		double posZ = 0;

		NodeList posList = subElem.getElementsByTagName("Pos");
		Element posElem = (Element) posList.item(0);

		posX = Double.parseDouble(posElem.getAttribute("x"));
		posY = Double.parseDouble(posElem.getAttribute("y"));
		posZ = Double.parseDouble(posElem.getAttribute("z"));

		Vector3d subPos = new Vector3d(posX, posY, posZ);

		String subtName = subElem.getAttribute("type");
		SubunitType subt = null;
		int size = subts.size();

		for (int i = 0; i < size && subt == null; ++i) {

			SubunitType tempSubt = subts.get(i);
			if (tempSubt.getName().equals(subtName)) 
				subt = new SubunitType(tempSubt);
		}

		if (reloaded) 
			suId = Integer.parseInt(name);
		else
			++suId;

		NodeList domainList = subElem.getElementsByTagName("Domain");
		size = domainList.getLength();
		for (int j = 0; j < size; ++j)
		{
			Element domElem = (Element) domainList.item(j);
			int dID = Integer.parseInt(domElem.getAttribute("id"));
			Iterator<Domain> domItr = subt.getDomains().iterator();

			Domain dom=null;
			while(domItr.hasNext())
			{
				Domain d = domItr.next();
				if(d.getDomainId()==dID){
					dom = d;
					break;
				}
			}

			boolean change = dom.changeConf(domElem.getAttribute("curconf"));

			if (change == false) {
				System.out.println("Could not change to the specified current conformation");
				System.exit(-1);
			}
		}

		Subunit sub = new Subunit(subt, suId, subPos);

		Vector<BindingSite> bs = sub.getAllBindingSites();
		size = bs.size();
		for (int i = 0; i < size; ++i)
			bs.get(i).setID(++bsID);

		return sub;
	}
}
