

/**
 * Event implements Comparable so that two events can be compared based on the
 * end time. The posting time of an Event can be checked against the valid time
 * of Assemblies involved to see if the event has been made invalid by other
 * events on the Assemblies involved.
 * 
 * @author Sue Yi Chew
 * @author Rori Rohlfs
 * @author Tiequan Zhang
 * @author Blake Sweeney
 * @version 1.4
 *  
 */


public class Event implements Comparable<Event> {

	/**
	 * The time this event was posted (when it was created) of this event. It
	 * doesn't matter whether the event will actually occur or not
	 */
	private double postTime;
	/**
	 * the time that this event will be complete (assuming it is not invalidated
	 * before then. Determines sorting order
	 */
	private double endTime; 
	/**
	 * Array of names of Assemblies involved. If this is a ConfChangeEvent or a
	 * BreakBondEvent, only first element is used, second is null. If this a
	 * FormBondEvent, the second element may or may not be used
	 */
	private Assembly[] assembliesInvolved;
	
	/**
	 * bs and partner are two binding sites involved in one binding event or breaking event.
	 */
	private BindingSite bs;

	private BindingSite partner;
	
	/** In Conformation changing event, the subunit involved  */
	private Subunit sub;
	
	/** ID of the domain related in the conformation changing event*/
	private int domainID;

	/** Conformation information for conformation changing event*/
	private Conformation conf;
	
	/** Event type for binding or breaking event*/
	private EventType evtType;
	
	@SuppressWarnings("unused")
	private Event(){}
	
	/**
	 * Constructor for binding and breaking event.
	 * 
	 * @param curTime - current simulation time
	 * @param eTime - event completed time
	 * @param assems - assemblies involved in this event
	 * @param bindSite - binding site 1 involved in this event
	 * @param partnerBindSite - binding site 2 involved in this event
	 * @param type - event type, should be binding or breaking
	 */
	public Event(double curTime, double eTime, Assembly[] assems, 
			BindingSite bindSite, BindingSite partnerBindSite, EventType type)
	{
		postTime = curTime;
		endTime = eTime;
		assembliesInvolved = assems;
		bs = bindSite;
		partner = partnerBindSite;
		sub = null;
		domainID = -1;
		conf = null;
		evtType = type;
	}
	
	/**
	 * Constructor for conformation changing event
	 * 
	 * @param curTime - current simulation time
	 * @param eTime - event completed time
	 * @param assems - assemblies involved in this event
	 * @param s - subunit that conformation changed
	 * @param domID - domain ID of the subunit conformation changed
	 * @param cnf - conformation information
	 */
	public Event(double curTime, double eTime, Assembly[] assems,
			Subunit s, int domID, Conformation cnf)
	{
		postTime = curTime;
		endTime = eTime;
		assembliesInvolved = assems;
		bs = null;
		partner = null;
		sub = s;
		domainID = domID;
		conf = cnf;
		evtType = EventType.cnfChngEvt;
	}

	/**
	 * Method returns the event type
	 * 
	 * @return EventType information
	 */
	public final EventType getEventType()
	{
		return evtType;
	}
	
	/**
	 * Method returns event posted time
	 * 
	 * @return double post time
	 */
	public final double getPostTime() {
		return postTime;
	}
	
	/**
	 * Method sets the event posted time
	 * 
	 * @param pTime
	 */
	public final void setPostTime(double pTime) {
		postTime = pTime;
	}

	/**
	 * Method returns the event completed time
	 * 
	 * @return double completed time
	 */
	public final double getEndTime() {
		return endTime;
	}

	/**
	 * Method sets the event completed time
	 * 
	 * @param eTime
	 */
	public final void setEndTime(double eTime) {
		endTime = eTime;
	}
	
	/**
	 * Method returns the assemblies involved in this event
	 * 
	 * @return an assemblies array
	 */
	public final Assembly[] getAssembliesInvolved() {
		return assembliesInvolved;
	}
	
	/**
	 * Method sets the assemblies should be involved in this event
	 * 
	 * @param assemInvolved - an assemblies array
	 */
	public final void setAssembliesInvolved(Assembly[] assemInvolved) {
		assembliesInvolved = assemInvolved;
	}

	/**
	 * Method returns binding site involved in the event
	 * 
	 * @return bs - the binding site involved
	 */
	public final BindingSite getBS() {
		return bs;
	}

	/**
	 * Method returns partner binding site of bs, that also involved in the event
	 * 
	 * @return partner - the partner binding site involved
	 */
	public final BindingSite getPartner() {
		return partner;
	}

	/**
	 * Method returns the subunit that conformation changing happened
	 * 
	 * @return sub - the subunit have conformation changing
	 */
	public final Subunit getSubunit() {
		return sub;
	}
	
	/**
	 * Method returns the domain ID in conformation changing
	 * 
	 * @return domainID - int ID of the domain
	 */
	public final int getDomainID() {
		return domainID;
	}

	/**
	 * Methods returns the conformation
	 * 
	 * @return conf - confmation 
	 */
	public final Conformation getNewConf() {
		return conf;
	}
	
	   /**
     * Compares Object e and this Event based on their end time.
     * 
     * @param e
     *            Object
     * 
     * @return 1 if this Event's end time is earlier than Object e, 0 if they
     *         are the same, -1 if this Event's end time is greater than Object
     *         e
     */
    public int compareTo(Event e) {
    	
    	if ((this == null) && (e == null)) 
    		return 0;
    	else if (this == null) 
    		return -1;
    	else if (e == null) 
    		return 1;
    	else 
    		return -(Double.compare(endTime, e.getEndTime()));
    }



    /**
     * Checks the validity of this Event. Event is valid if it was post after
     * the most recent events for the Assemblies involved. That is, if the
     * posting time of this event is greater than or equal to validTimes of each
     * Assembly involved. validTimes[1] may be null if only one assembly was
     * involved in this event.
     * 
     * @param validTimes
     *            Array of validTimes of all assemblies this event involves
     * @return boolean
     */
    public boolean isValid(double[] validTimes) {
        for (int i = 0; i < validTimes.length; ++i) {
            if (postTime < validTimes[i]) {
                return false;
            }
        }
        return true;
    }

}
