

import java.io.FileWriter;
import java.io.IOException;
import java.util.HashSet;
import java.util.Vector;

import java.util.HashMap;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Quat4d;
import javax.vecmath.Vector3d;


/**
 * Assembly holds an Assembly of Subunits and the AssemblyGraphic. It also
 * tracks the next Event for this Assembly and its validity. Assemblies can be
 * bonded, split, or relaxed.
 * 
 * @author Rori Rohlfs
 * @author Sue Yi Chew
 * @author Tiequan Zhang
 * @author Woo Tae Kim
 * @author Blake Sweeney
 * @version 1.4
 *  
 */


public class Assembly {
	/** A vector of the subunits */
	private Vector<Subunit> mysubs;

	/** The valid time for any events with this Assembly */
	private double validTime;

	/** The ID of this Assembly */
	private int myID;

	/**
	 * A constructor
	 * 
	 * @param s - A Vector of the Subunits for this Assembly
	 * @param name - The name of this Assembly
	 */
	public Assembly(Vector<Subunit> s, int id) {

		mysubs = new Vector<Subunit>();
		int size = s.size();
		for (int i = 0; i < size; ++i) {
			mysubs.add(s.get(i));
			mysubs.get(i).setAssembly(this);
		}
		myID = id;
		validTime = -1.0;


	}

	/**
	 * Method binds this assembly to another one using a specified event.
	 * Checks for stearic hindrance and fast binding
	 * 
	 * @param bindasm - Assembly to bind to
	 * @param e - Event to use to specify the binding
	 * @return Assembly - the new Assembly composed of the two previous ones
	 * 
	 * @Description The binding event considers two subunits, oldSub and bindSub, where we test if
	 * bindSub can be moved and rotated to effectively bind to oldSub, then determine if any other 
	 * subunits in the assemblies can bind together. 
	 */
	public Assembly bindAssembly(Assembly bindasm, Event e) {
	
		BindingSite oldbs = e.getBS();
		Subunit oldSub = oldbs.getSubunit();

		BindingSite bindBS = e.getPartner();
		Subunit bindSub = bindBS.getSubunit();

		Vector3d oldBSPosRelative = new Vector3d();

        /* To determine the relative locations and directionality of the two binding sites with
         * respect to their subunits, we subtract the 3d location of the subunits from the 3d
         * location of their respective binding sites. These relative locations of the binding
         * sites can be used to determine the angle between the two binding sites and thus
         * how much the new assembly must be rotated to line up the binding sites for proper
         * assembly bonding.
		 */
		
		oldBSPosRelative.sub(oldbs.getPosReal(), oldSub.getPositionReal());
		Vector3d bindBSPosRelative = new Vector3d();
		bindBSPosRelative.sub(bindBS.getPosReal(), bindSub.getPositionReal());
		
        /* The angle calculated is the angle between the two relative position vectors of the
         * binding sites. 
         * For proper bonding, the angle must be close to pi: ----> <-----
		 */
        double angle = oldBSPosRelative.angle(bindBSPosRelative);
		
        /* To make the two assembly properly binding together, rotation might need to be applied
         * to one assembly. The firstAxisAngle4d is a 4d vector. It's composed of firstAxis, a 3d
         * vector that is perpendicular to the relative position vector to be rotated. This 3d 
         * vector is the axis by which the position vector is rotated with a desired angle. The 
         * angle is stored as the fourth element in firstAxisAngle4d.
         */
		Quat4d firstRotation = new Quat4d();
		AxisAngle4d firstAxisAngle4d = null;
		
		if (angle < 0.000001) {
		/* If the angle is very close to 0, the two binding sites look like this:
         * ---->      ----->
         * Thus bindSub must be rotated 180 degrees (pi radians) for proper bonding.
         */
			Vector3d firstAxis = new Vector3d();
			double temAngle = bindBSPosRelative.angle(new Vector3d(1, 0, 0));
			if ((temAngle < 0.1) || ((Math.PI - temAngle) < 0.1)) {             
            /* If the binding site relative position vector is close to or opposite to (1,0,0), 
             * we cross the vector with (0,1,0) to form the axis of rotation, which would be
             * perpendicular to the position vector to be roated			
             */
				firstAxis.cross(bindBSPosRelative, new Vector3d(0, 1, 0));
			} else {
			/*
			 * If the binding site relative position vector is not close to or opposite to
			 * (1,0,0), we can cross the vector with (1,0,0) to form the axis of rotation.	
			 */
				firstAxis.cross(bindBSPosRelative, new Vector3d(1, 0, 0));
			}
			firstAxisAngle4d = new AxisAngle4d(firstAxis, Math.PI);
			/*In this case, the vector need to rotate 180 degrees.*/
		} else if ((Math.PI - angle) < 0.00001) {
		   	/* If the angle between the two relative position vectors is close to Pi, they
        	 * are already rotated properly and no rotation is necessary. Therefore,
        	 * firstAxisAngle4d is simply a rotation of 0 radians about the vector axis (1,0,0)	
        	 */
			firstAxisAngle4d = new AxisAngle4d(new Vector3d(1, 0, 0), 0);
		} else {
			/*
			 * In the non-trivial case where the angle between the two relative position
        	 * vectors is not close to 0 or Pi, we must first find an axis of rotation that 
        	 * is perpendicular to both by calculating their cross product. 
			 */
			Vector3d firstAxis = new Vector3d();
			firstAxis.cross(bindBSPosRelative, oldBSPosRelative);
			firstAxis.normalize();
			
            /* For the angle to rotate bindSub's relative position vector, subtract Pi from
             * the angle between the two position vectors
			 */
			firstAxisAngle4d = new AxisAngle4d(firstAxis, angle - Math.PI);
		}

		firstRotation.set(firstAxisAngle4d);
        /* firstRotation is a quaternion (4 dimensional vector with special multiplicative
         * properties) that is calculated from the 4d vector in firstAxisAngle4d by:
        
         * Given firstAxisAngle4d = (x,y,z,a), 
         * firstRotation = (x*sin(a/2), y*sin(a/2), z*sin(a/2), cos(a/2))
        
         * quaternions are used to rotate vectors...by converting bindBSPosRelative into a 
         * 4d vector (use 0 for the fourth dimension), then, using quaternion multiplication,
         * firstRotation*bindBSPosRelative*(firstRotation^(-1)) will rotate bindBSPosRelative
         * to the desired angle maintaining vector length. This is done similarly in this java
         * code by using a matrix transform constructed from the quaternion to produce the 
         * desired effect.
        
         * There are two types of rotation necessary in prepping the new subunit for proper
         * bonding. The first is described above with firstRotation in aligning the two binding
         * sites. The second is to align the two assemblies with the subunits' respective
         * up-vectors.
                      
         * First define oldProj as the component of the oldSub upVector that is perpendicular
         * to oldBSPosRelative.
		 */

		Vector3d oldProj = Test.makePerpendicular(oldBSPosRelative, oldSub
				.getUpRelative());
		
        /* Second, we rotate bindSub's upVector by firstAxisAngle4d to understand its
         * directionality after aligning the binding sites.
		 */
		Vector3d newUpV = Test.rotateByAxis(bindSub.getUpRelative(),
				firstAxisAngle4d);
		
		/* We then construct newProj as the component of the rotated newSub upVector that is
         * perpendicular to oldBSPosRelative. 
         */
		Vector3d newProj = Test.makePerpendicular(oldBSPosRelative, newUpV);

        /* We can then construct the angle between newProj and oldProj as a measure for
         * the angle between the two up vectors and how much the new assembly must be rotated
         * for the binding subunits to be aligned.    
		 */
		double currentAngleBetweenProjectedVectors = newProj.angle(oldProj);
		
        /* rotAngle is the desired angle between subunits that is given in the xml input file.
         * This is standardly set at 0.
		 */	
		double rotAngle = Simulation.getBSTBindAngle(oldbs.getBSTName(),bindBS.getBSTName());
		Vector3d testVector = new Vector3d();
		
        /* If the angle between the projected vectors is nonzero, construct testVector to be
         * perpendicular to both oldProj and newProj making sure to flip the angle if the
         * testVector is directed opposite to oldBSPosRelative. 
		 */
		if (currentAngleBetweenProjectedVectors>0.00000001) {
			testVector.cross(oldProj, newProj);
			testVector.normalize();
			if (Math.abs(oldBSPosRelative.angle(testVector) - Math.PI) <= 0.000001)
				currentAngleBetweenProjectedVectors = -currentAngleBetweenProjectedVectors;
		}
		
        /* Define the second 4d rotation vector using oldBSPosRelative as the axis of rotation
         * and the difference between the desired angle and the current angle between projected
         * vectors as the angle of rotation.
		 */
		AxisAngle4d rot = new AxisAngle4d(oldBSPosRelative, rotAngle
				- currentAngleBetweenProjectedVectors);
		
		/* Construct a new rotation quaternion called TotalRotation based on this 4d vector */
		Quat4d totalRotation = new Quat4d();
		totalRotation.set(rot);

        /* Multiply the two quaternions firstRotation and totalRotation to produce one
         * quaternion that can conduct both rotations of the new subunit: 
         * q2*q1*A*(q1^-1)*(q2^-1) = (q2*q1)*A*((q2*q1)^-1)
		 */
		totalRotation.mul(totalRotation, firstRotation);
		
        /* Now that we know how to rotate the assemblies, we must determine how far to move the
         * subunits so they are in the proper positions to bind.
        
         * First we calculate translation as the difference between real positions of the 
         * two binding sites.
         */
		Vector3d translation = new Vector3d();
		translation.sub(oldSub.getPositionReal(), bindSub.getPositionReal());
		
		/* We then construct the vector tem to include the distance from the old subunit to
         * the old binding site. We include the distance from the new subunit to the new
         * binding site by scaling tem by (1 + ratio of new-distance/old-distance)    
		 */
		Vector3d tem = new Vector3d(oldBSPosRelative);

        /* We now cycle through each subunit of bindasm (the new assembly) and move and rotate
         * each subunit in the desired manner.
         */
		double scale = 1 + bindBSPosRelative.length()
		/ oldBSPosRelative.length();
		tem.scale(scale);
		
		/* tem is added onto translation as the whole distance the new assembly must be moved */
		translation.add(tem);

        /* We now cycle through each subunit of bindasm (the new assembly) and move and rotate
         * each subunit in the desired manner.
		 */	
		Vector<Subunit> bindSubs = bindasm.getSubunits();

		Vector3d refPoint = new Vector3d(bindSub.getPositionReal());
		int size = bindSubs.size();
		for (int i = 0; i < size; ++i) {
			
			/*Use subMove to move each subunit*/
			Subunit cursub = bindSubs.get(i);

			cursub.subMove(refPoint, translation, totalRotation);
		}
		
		/* We check if there is steric hinderance between the new assembly and the original
         * assembly. If binding is hindered, we stop the binding process.
		 */
		if (stericHindrance(bindasm)) {
			return null;
		}

		else {
        	/* We now call processLoop which loops through all subunits in each assembly to
        	 * test if any other binding processes can also occur in joining these assemblies.
             */
			processLoop(this, bindasm, oldbs, bindBS);
			Vector<Subunit> bindsubs = bindasm.getSubunits();

			/* Adds all Subunit(s) in bindasm to this Assembly */
			mysubs.addAll(bindsubs);
			size = bindSubs.size();
			for (int i = 0; i < size; ++i) {
				Subunit cursub = bindsubs.get(i);
				cursub.setAssembly(this);
			}


			/*if(Test.springForce)
				updateSpringForce();*/
			
			return this;
		}
	}


	/**
	 * Method that do loops through all binding sites of all subunits inside the bind assembly
	 * from previous method and the candidate partner binding sites to check whether they
	 * are compatible or not.
	 *
	 * @param a0 Assembly that bind to another one
	 * @param a1 Assembly bind partner of a0
	 * @param bs0 Binding site from a0
	 * @param bs1 Binding site from a1  
	 * @return
	 */
	private void processLoop(Assembly a0, Assembly a1, BindingSite bs0, BindingSite bs1) {
		/*boolean operOK=false;

		operOK = */bs0.bindTo(bs1);
		/*if(!operOK)
			System.out.println("Bind operation failed in process loop");*/
		
		Subunit sub1 = bs1.getSubunit();
		Subunit sub0 = bs0.getSubunit();

		/*operOK = */sub1.addPartner(sub0);

		/*if(!operOK)
			System.out.println("addParter operation failed in process loop");*/

		/*operOK = */sub0.addPartner(sub1);

		/*if(!operOK)
			System.out.println("addParter operation failed in process loop");*/

		NeighborGroups ngloop = new NeighborGroups(a0, true);
		Vector<Subunit> subs = a1.getSubunits();
		int size = subs.size();
		int kSize = 0;
		int jSize = 0;
		for (int i = 0; i < size; ++i) 
		{
			Subunit sub = (Subunit) subs.get(i);
			Vector<BindingSite> bss = sub.getBindingSites();
			kSize = bss.size();
			for (int k = 0; k < kSize; ++k) 
			{
				BindingSite tembs1 = (BindingSite) bss.get(k);
				if (!tembs1.isBound()) {
					Vector3d bsTipTem1 = tembs1.getPosReal();

					Vector<BSA> bsNeighbors = ngloop.findLoopCandidates(bsTipTem1);

					if (bsNeighbors.size() == 0) 
						continue;
					jSize = bsNeighbors.size();
					for (int j = 0; j < jSize; ++j) {

						BSA temBSA = bsNeighbors.get(j);
						BindingSite tembs0 = temBSA.bs;

						if (Simulation.isCompatible(tembs1.getBSTName(),tembs0.getBSTName())) {

							Vector3d temv = new Vector3d(tembs0.getPosReal());

							Vector3d v1 = new Vector3d(temv);
							v1.sub(tembs0.getSubunit().getPositionReal());

							Vector3d v2 = new Vector3d(bsTipTem1);
							v2.sub(tembs1.getSubunit().getPositionReal());                        

							temv.sub(bsTipTem1);

							if (temv.length() <= Test.distanceTolerance ) {
								/*operOK = */tembs1.bindTo(tembs0);
								/*if(!operOK)
									System.out.println("bind operation failed in (for loop) of process loop");

								operOK = */(temBSA.sub).addPartner(sub);
								/*if(!operOK)
									System.out.println("addPartner operation failed in (for loop) of process loop");

								operOK = */sub.addPartner(temBSA.sub);

								/*if(!operOK)
									System.out.println("addPartner operation failed in (for loop) of process loop");*/
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Performs a FastBindEvent.
	 * 
	 * @param e - The FastBindEvent to perform
	 */
	public void fastBind(Event e) 
	{    	
		e.getBS().bindTo(e.getPartner());
		Subunit sub = (e.getBS()).getSubunit();
		Subunit subPartner = (e.getPartner()).getSubunit();
		sub.addPartner(subPartner);
		subPartner.addPartner(sub);
	}

	/**
	 * This gets the counts of unbound BindingSites in this Assembly. This is
	 * used for faster event sampling in the Simulation. 
	 * 
	 * @return HashMap<String, Integer> - A HashMap of the names of each BindingSiteType
	 * 	and the counts of how many are unbound.
	 */
	public HashMap<String, Integer> getBSCounts() {

		HashMap<String, Integer> tem = new HashMap<String, Integer>();
		int size = mysubs.size();
		int jSize = 0;
		for (int i = 0; i < size; ++i) {

			Subunit s = mysubs.get(i);
			Vector<BindingSite> bss = s.getBindingSites();
			jSize = bss.size();
			for (int j = 0; j < jSize; ++j)
			{            	
				BindingSite b = bss.get(j);

				if (!b.isBound()) {
					String bst = b.getBSTName();

					if (tem.containsKey(bst)) {
						Integer n = (Integer) tem.get(bst);
						tem.put(bst, new Integer(n.intValue() + 1));
					} else 
						tem.put(bst, new Integer(1));
				}
			}
		}
		return tem;
	}

	/**
	 * Returns the number of subunits in this Assembly
	 * 
	 * @return int - the number of Subunits in this Assembly
	 */
	public int numSubunits() 
	{ return mysubs.size(); }

	/**
	 * Given a <code>BreakBondEvent</code> this will split the <code>Assembly</code> 
	 * into two
	 * 
	 * @param e - A <code>BreakBondEvent</code> splitting this <code>Assembly</code>
	 * @param newasmnum - Number to use to create unique name for one of the
	 * new <code>Assembly</code>
	 * @return <code>Assembly</code> - The new split <code>Assembly</code>
	 */
	public Assembly splitAssembly(Event e, int newasmnum) 
	{
		/*boolean operOK=false;*/
		BindingSite bs1 = e.getBS();
		BindingSite bs2 = e.getPartner();
		Subunit bsSub1 = bs1.getSubunit();
		Subunit bsSub2 = bs2.getSubunit();
		bs1.breakBond();

		/*operOK = */bsSub1.removePartner(bsSub2);
		/*if(!operOK)
			System.out.println("RemovePartner failed in Split Assembly! -BUG EXISTS!");

		operOK = */bsSub2.removePartner(bsSub1);

		/*if(!operOK)
			System.out.println("RemovePartner failed in Split Assembly! -BUG EXISTS!");
*/

		if (!(isConnected(bsSub1, bsSub2))) {

			//second assembly
			Vector<Subunit> detachsubunits = getConnected(bsSub2);
			//detach each SubunitGraphic and update BindingSite Assembly names
			Assembly detachedAssembly = new Assembly(detachsubunits,newasmnum);
			int size = detachsubunits.size();
			for (int j = 0; j < size; ++j) {
				Subunit curSubunit = (Subunit) detachsubunits.get(j);
				mysubs.remove(curSubunit);
				curSubunit.setAssembly(detachedAssembly);
			}
			
			return detachedAssembly;
		}

		//if bond break didn't make 2 assemblies, return null
		return null;
	}
	
	/**
	 * A method to check whether two <Code>Subunit</Code> objects s1 and s2 are connected or not.
	 * If s1 and s2 in the same assembly, they should be connected, though may not directly bound.
	 * It will call another recursive search function <Code>getConnected</Code> to do the check.
	 * 
	 * @param s1 - <Code>Subunit</Code> object 1
	 * @param s2 - <Code>Subunit</Code> object 2   
	 * @return A boolean value, if connected, return true; otherwise return false.
	 */
	private boolean isConnected(Subunit s1, Subunit s2)
	{
		HashSet<Integer> seen = new HashSet<Integer>();
		seen.add(s1.getID());
		return getConnected(s1,seen,s2.getID(),false);    	
	}

	/**
	 * A recursive depth first search method to check whether two subunits are connected or not.
	 *
	 * @param s - <Code>Subunit</Code> object 1
	 * @param seen - A HashSet that store the subunits already checked 
	 * @param sId - The unique subunit id of <Code>Subunit</Code> object 2	
	 * @param found - a boolean variable store the connected state (true or false)
	 * @return A boolean value, if connected, return true; otherwise return false.
	 */
	private boolean getConnected(Subunit s, HashSet<Integer> seen, int sId, boolean found)
	{
		if(found) return found;

		Vector<Subunit> boundSubunits = s.getBoundSubunits();
		int size = boundSubunits.size();
		for(int i =0; i < size && (!found); ++i)
		{
			Subunit sub = boundSubunits.get(i);
			int id = sub.getID();
			if(id == sId)
			{
				found = true;
				break;
			}

			if(seen.add(id))
				found = getConnected(sub,seen,sId,found);    		
		}

		return found;
	}

	/**
	 * Returns Vector containing all the subunits are in the same Assembly with
	 * Subunit sub, including sub itself. Similar to Dijkstra algorithm
	 * 
	 * @param sub Subunit
	 * @return Vector of Subunits
	 */
	public Vector<Subunit> getConnected(Subunit sub) {
		Vector<Subunit> ret = new Vector<Subunit>();
		HashSet<Integer> seen = new HashSet<Integer>();
		seen.add(sub.getID());//mark subunit
		ret.add(sub);
		return getConnected(ret, sub,seen);
	}    

	/**
	 * Finds all Subunit objects that are in the same Assembly as Subunit sub
	 * marks them(including Subunit sub) and return a Vector of all Subunits
	 * 
	 * @param v  Vector
	 * @param sub Subunit
	 * @return Vector of all Subunit objects in the same Assembly
	 *  
	 */
	private Vector<Subunit> getConnected(Vector<Subunit> v, Subunit sub,HashSet<Integer> seen) {

		Vector<Subunit> boundSubunits = sub.getBoundSubunits();
		int size = boundSubunits.size();
		for(int i =0; i < size; ++i) {
			Subunit curSub = boundSubunits.get(i);
			if (seen.add(curSub.getID())) {
				v.add(curSub);
				getConnected(v, curSub,seen);
			}
		}
		return v;
	}

	/**
	 * Checks for stearic hindrance of this <code>Assembly</code> with a partner.
	 * 
	 * @param partner - The <code>Assembly</code> to look at for stearic hindrance
	 * @return boolean - True if hindrance is detected, false otherwise
	 */
	private boolean stericHindrance(Assembly partner) {

		NeighborGroups ng = null;
		Vector<Subunit> v = null;

		if (this.numSubunits() > partner.numSubunits()) {
			ng = new NeighborGroups(this);
			v = partner.getSubunits();
		} else {
			ng = new NeighborGroups(partner);
			v = this.getSubunits();
		}
		int size = v.size();
		for (int i = 0; i < size; ++i) {

			if (ng.findHindrance(v.get(i))) 
				return true;
		}

		return false;
	}

	/**
	 * A method that apply break event in a loop assembly. In currently simulation,
	 * this situation is forbidden.
	 *
	 * @param bs1 - <Code>BindingSite</Code> object 1
	 * @param bs2 - <Code>BindingSite</Code> object 2
	 * @return A boolean value shows whether this loop break or not.
	 */
	public boolean splitAssemblyInLoop(BindingSite bs1, BindingSite bs2) 
	{
		/*boolean operOK=false;//flags added by RP*/
		Subunit bsSub1 = bs1.getSubunit();
		Subunit bsSub2 = bs2.getSubunit();
		boolean breakLoop = false;
		bs1.breakBond();

		/*operOK = */bsSub1.removePartner(bsSub2);
		/*if(!operOK)
			System.out.println("RemovePartner failed in SplitAssemblyInLoop! -BUG EXISTS!");

		operOK = */bsSub2.removePartner(bsSub1);

		/*if(!operOK)
			System.out.println("RemovePartner failed in SplitAssemblyInLoop! -BUG EXISTS!");
*/

		if ((isConnected(bsSub1, bsSub2))) 
			breakLoop = true;


		/*operOK=false;
		operOK=*/bs1.bindTo(bs2);

		/*if(!operOK)
			System.out.println("bindTo failed in SplitAssemblyInLoop! -BUG EXISTS!");

		operOK = */bsSub1.addPartner(bsSub2);

		/*if(!operOK)
			System.out.println("addPartner failed in SplitAssemblyInLoop! -BUG EXISTS!");

		operOK = */bsSub2.addPartner(bsSub1);

		/*if(!operOK)
			System.out.println("addPartner failed in SplitAssemblyInLoop! -BUG EXISTS!");*/

		return breakLoop;
	}

	/**
	 * Adds a Subunit to this Assembly
	 * 
	 * @param newSubunit - The Subunit to add
	 */
	public void addSubunit(Subunit newSubunit) 
	{  mysubs.add(newSubunit); }



	/**
	 * Returns the subunits in this Assembly
	 * 
	 * @return Vector<Subunit> - A vector of all subunits in this Assembly
	 */
	public Vector<Subunit> getSubunits() 
	{ return mysubs; }



	/**
	 * Returns the valid time for this Assembly
	 * 
	 * @return double - the valid time
	 */
	public double validTime() 
	{  return validTime; }


	/**
	 * Sets the valid time for this Assembly
	 * 
	 * @param time - the valid time to set to
	 */
	public void setValidTime(double time) 
	{ validTime = time; }

	/**
	 * Returns the name of this Assembly
	 * 
	 * @return String - the Name of this Assembly
	 */
	public int getID() 
	{ return myID; }

	/**
	 * Returns a Vector of all free BindingSites in this Assembly
	 * 
	 * @return Vector<BindingSite> - All free BindingSites in this Assembly
	 */
	public Vector<BindingSite> getFreeBindingSites() {

		Vector<BindingSite> free = new Vector<BindingSite>(); 
		int size = mysubs.size();
		for (int i = 0; i < size; ++i) 
			free.addAll(mysubs.get(i).getFreeBindingSites());

		return free;
	}



	/**
	 * A toString method
	 */
	public String toString() {
		StringBuffer buff = new StringBuffer();

		buff.append("Name: ");
		buff.append(myID);
		buff.append(" ValidTime: ");
		buff.append(validTime);
		buff.append("\n");

		for (int i = 0; i < mysubs.size(); i++)
			buff.append(mysubs.get(i) + "\n");

		buff.append("\n");

		return buff.toString();
	}

	/**
	 * Updates the values the spring force model of this Assembly
	 *//*
	public void updateSpringForce() {

		boolean moreBind = true;

		while(moreBind) {

			double currentEnergy = getAssemblyEnergy();
			updateDeriv(currentEnergy);
			double optimalEnergy = findOptimalState(currentEnergy);
			boolean inside = false;

			while ((currentEnergy - optimalEnergy) > 0.000001) {

				if(currentEnergy < optimalEnergy) {
					System.out.println("Something is wrong for getting currentEnergy, optimal");
					System.exit(-1);
				}

				currentEnergy = optimalEnergy;
				updateDeriv(currentEnergy);
				optimalEnergy = findOptimalState(currentEnergy);
			}

			if(inside)
				System.out.println("end of optimal energy = " + optimalEnergy);

			moreBind = findOtherBond();
		}
	}



	private void updateDeriv(double currentEnergy) {

		double derivStep = 0.000000001;

		for(int i = 0; i < mysubs.size(); i++) {

			double energy;

			Subunit sub = mysubs.get(i);

			sub.saveCurrentPos();

			sub.moveSub(new Vector3d(derivStep, 0, 0));
			energy = getAssemblyEnergy();

			sub.setXDeriv((currentEnergy - energy) / derivStep);
			sub.restorePos();

			sub.saveCurrentPos();    		
			sub.moveSub(new Vector3d(0, derivStep, 0));
			energy = getAssemblyEnergy();
			sub.setYDeriv((currentEnergy - energy) / derivStep);
			sub.restorePos();

			sub.saveCurrentPos();    		
			sub.moveSub(new Vector3d(0, 0, derivStep));

			energy = getAssemblyEnergy();
			sub.setZDeriv((currentEnergy - energy) / derivStep);
			sub.restorePos();

			sub.saveCurrentPos();
			sub.rotateSub(new AxisAngle4d(1,0,0,derivStep));
			energy = getAssemblyEnergy();
			sub.setXRotDeriv((currentEnergy - energy)/derivStep);
			sub.restorePos();

			sub.saveCurrentPos();
			sub.rotateSub(new AxisAngle4d(0,1,0,derivStep));
			energy = getAssemblyEnergy();
			sub.setXRotDeriv((currentEnergy - energy)/derivStep);
			sub.restorePos();

			sub.saveCurrentPos();
			sub.rotateSub(new AxisAngle4d(0,0,1,derivStep));
			energy = getAssemblyEnergy();
			sub.setXRotDeriv((currentEnergy - energy)/derivStep);
			sub.restorePos();
		}
	}



	*//**
	 * Given the current energy this will find the optimal energy for the
	 * Assembly
	 * 
	 * @param currentEnergy - the current energy
	 * @return double - the optimal energy
	 *//*
	public double findOptimalState(double currentEnergy) {

		double optimalEnergy = currentEnergy;
		double energy;
		double upperBound = 100000.0;
		double lowerBound = 0.0;
		double kVal = (upperBound + lowerBound) / 2.0;

		while(true) {

			for(int i = 0; i < mysubs.size(); i++) {

				Subunit sub = mysubs.get(i);
				sub.saveCurrentPos();
				sub.applySpringForce(kVal);
			}

			energy = getAssemblyEnergy();

			if(energy < optimalEnergy) {
				optimalEnergy = energy;
				lowerBound = kVal;
			} else 
				upperBound = kVal;

			kVal = (upperBound + lowerBound) / 2.0;

			if((upperBound - lowerBound) < 0.00000001) {
				if(currentEnergy - optimalEnergy < 0.00000001) {

					for(int i = 0; i < mysubs.size(); i++) 
						mysubs.get(i).restorePos();
				}
				return optimalEnergy;
			}

			for(int i = 0; i < mysubs.size(); i++) 
				mysubs.get(i).restorePos();
		}
	}



	*//**
	 * Gets the total energy for this assembly, but summing the energy
	 * of each subunit
	 * 
	 * @return double - the energy of this Assembly
	 *//*
	private double getAssemblyEnergy() {

		double energy = 0.0;

		for(int i = 0; i < mysubs.size(); i++) 
			energy+= mysubs.get(i).getEnergy();

		return energy;
	}



	*//**
	 * 
	 * @return
	 *//*
	private boolean findOtherBond() {

		boolean found = false;

		for(int i = 0; i < mysubs.size()-1; i++) {

			Subunit subOne = mysubs.get(i);

			for(int j = i+1; j < mysubs.size(); j++) {

				Subunit subTwo = mysubs.get(j);

				Vector<BindingSite> bss1 = subOne.getBindingSites();
				Vector<BindingSite> bss2 = subTwo.getBindingSites();

				for(int k = 0; k < bss1.size(); k++) {

					BindingSite bs1 = bss1.get(k);

					if(!bs1.isBound()) {

						for(int l = 0; l < bss2.size(); l++) {

							BindingSite bs2 = bss2.get(l);

							if(!bs2.isBound() && Simulation.isCompatible(bs1.getBSTName(), bs2.getBSTName())) {

								Vector3d tipPos = new Vector3d(bs1.getPosReal());
								tipPos.sub(bs2.getPosReal());

								if(tipPos.length() <= Test.distanceTolerance) {

									subOne.addPartner(subTwo);
									subTwo.addPartner(subOne);
									bs1.bindTo(bs2);
									found = true;
									System.out.println("found other");
								}
							}
						}
					}
				}
			}
		}
		return found;
	}*/


}