

import java.io.*;

import org.w3c.dom.*;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.sax.*;

import java.lang.Exception;
import javax.vecmath.*;
import javax.media.j3d.*;

import java.util.*;

/**
 * XMLParser class handles all the methods that are related to XML parsing. 
 * XMLParser is capable of parsing XML template that is provided by user, and 
 * load the simulation based on that XML.
 * XMLParser also handles saving the current state of simulation into XML.
 * 
 * @author Kim
 *
 * @version 1.3
 */
public class XMLParser {
	
	
	private Document doc; // Document that has xml template
	private HashMap partnerMap; // Hashmap for storing subunits within polymer
	private HashMap bstMap; // Hashmap for binding site types
	private HashMap bindingAngles = new HashMap(); // Hashmap for up vector angle difference
	public Simulation sim; // Simulator that will be made by XML parser
	
	private int asmId; // Assembly id - for tracking purpose
	private int suId;  // Subunit id
	private int bsId;  // bindingsite id
	private int domainId; // domain id
	private Vector bsts; // vector of binding site types
	private Vector subts; // vector of subunit types
	private boolean reloaded; // To specify wheather the simulation is reloaded
	private int asmNum; // Assembly number for tracking in reloading.

	/**
	 * Reads file from given path and parses it to load simulation.
	 * 
	 * @param path, Path of user's imput XML file
	 * 
	 */
	public XMLParser(String path)
	{
		File F = new File(path);
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		try{
		DocumentBuilder db = dbf.newDocumentBuilder();
		doc = db.parse(F);
		} catch (Exception e){
			System.out.println("Unable to open document");
			System.exit(-1);
		}
		Element type = (Element) doc.getElementsByTagName("System").item(0);
		Integer assemNum = null;
		Test.simChoice = type.getAttribute("type");
		try{
			// reloaded is being checked for saving and reloading feature for XML.
			if(type.getAttribute("reloaded").equals("yes"))
			{
				reloaded = true;
				assemNum = new Integer(type.getAttribute("AssemblyNumber"));
				asmNum = assemNum.intValue();

			}
		}catch(Exception e)
		{
			reloaded = false;
		}
		
		
		asmId = -1;
		bsId = 0;
		suId = 0;
		bsts = new Vector();
		subts = new Vector();
		return; 
	}
	
	/**
	 * 
	 * It sets new XML file path if it is necessary to change.
	 * 
	 * @param path, path of XML file
	 */
	public void setDocument(String path)
	{
		File F = new File(path);
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		try{
		DocumentBuilder db = dbf.newDocumentBuilder();
		doc = db.parse(F);
		} catch (Exception e){
			System.out.println("Unable to open document");
			System.exit(-1);
		}
	}

	/**
	 * It sets the Solution class for simulation. Values are read from XML document
	 * that is set when construcing XMLParser.
	 * 
	 * @return Solution
	 */
	public Solution getNewSolution()
	{
		Element soln = (Element) doc.getElementsByTagName("Solution").item(0);

		Double temperature = new Double(soln.getAttribute("Temp"));
		Double pH = new Double(soln.getAttribute("pH"));
		Double volume = new Double(soln.getAttribute("Volume"));
		
		Solution sol = new Solution(temperature.doubleValue(), pH.doubleValue(), volume.doubleValue());
		return sol;
	}

	/**
	 * This method sets the BindingSiteType class for simulation.
	 * Values are read from XML.
	 *
	 */
	
	public void setBindingSiteTypes()
	{
		partnerMap = new HashMap();
		bstMap = new HashMap();
		//bindingAngles = new HashMap();
		
		
		Element temp = (Element) doc.getElementsByTagName("BindingSiteTypes").item(0);		
		NodeList bsTypeList = temp.getElementsByTagName("BindingSiteType");
		
		//Going through list of binding site types
		
		for(int i = 0; i < bsTypeList.getLength(); i++)
		{
			Element bsType = (Element) bsTypeList.item(i);		
			String bsTypeName = bsType.getAttribute("name");
			
			Element tolerance = (Element) bsType.getElementsByTagName("tolerance").item(0);			
			Double transition = new Double(tolerance.getAttribute("transition"));
			Double torsion = new Double(tolerance.getAttribute("torsion"));
			Double bending = new Double(tolerance.getAttribute("bending"));						
			
			double [] tolerances = {transition.doubleValue(), torsion.doubleValue(), bending.doubleValue()};
			
			Vector partner = new Vector();
			HashMap partnerAngle = new HashMap();
			NodeList partnerList = bsType.getElementsByTagName("partner");
			
			String partnerName;
			BindingSiteType bst = new BindingSiteType(tolerances, bsTypeName, partner);

			// Sets the binding site types that are possible to bind to current binding site type
			
			for(int j = 0; j < partnerList.getLength(); j++)
			{
				Element partnerElem = (Element) partnerList.item(j);
				partnerName = partnerElem.getAttribute("name");
				partner.add(partnerName);
				
			// Angle is the angle between two up vectors for binding site type pair	
				Double angle = new Double(partnerElem.getAttribute("angle"));
				bst.setBindingAngle(partnerName, angle.doubleValue());
				partnerAngle.put(partnerName, angle);								
			}
			
			bsts.add(bst);
			partnerMap.put(bsTypeName, partner);
			bstMap.put(bsTypeName, bst);
			bindingAngles.put(bsTypeName, partnerAngle);
		}

	}
	
	/**
	 * This method acutally builds assemblies and as it builds assemblies, 
	 * it also constructs subunits and bindingsites which are part of assembly.
	 * 
	 * 
	 * @param assemblies, Vector where assemblies will be stored for simulation
	 * @param soln, Solution class that is built from XMLParser.getNewSolution()
	 */
	public void setAssemblies(Vector assemblies, Solution soln)
	{			   	    
		NodeList assemList = doc.getElementsByTagName("Assembly");

		for(int i =0; i < assemList.getLength(); i++)
		{
			Element assemElem = (Element) assemList.item(i);
			Integer numAssemblies = new Integer(assemElem.getAttribute("amount"));
			String tempAsmName = assemElem.getAttribute("name");
			for(int j = 0; j < numAssemblies.intValue(); j++)
			{	
				asmId++;
				String asmName;
				if(reloaded)
					asmName = tempAsmName;
				else
					asmName = tempAsmName + asmId;
				
				NodeList subunitList = assemElem.getElementsByTagName("Subunit");
				Vector asmsubs = new Vector();
				HashMap subMap = new HashMap();
				
				for(int k =0; k < subunitList.getLength(); k++)
				{
					Element subunitElem = (Element) subunitList.item(k);

					// setting values for position of subunit within assembly
					NodeList check = subunitElem.getElementsByTagName("Position");
					Vector3d sPos = null;
					if(check != null)
					{
						Element pos = (Element) check.item(0);
						Double x = new Double(pos.getAttribute("x"));
						Double y = new Double(pos.getAttribute("y"));
						Double z = new Double(pos.getAttribute("z"));
						
						sPos = new Vector3d(x.doubleValue(), y.doubleValue(), z.doubleValue());
						
					}
					else
					{
						System.out.println("Pos is needs to specified");
						System.exit(-1);
					}
										
					//setting values for rotation of subunint within assembly
					check = subunitElem.getElementsByTagName("Rotation");
					Quat4d sRot = null;
					if(check != null)
					{
						Element rot = (Element) check.item(0);
						Double w = new Double(rot.getAttribute("w"));
						Double x = new Double(rot.getAttribute("x"));
						Double y = new Double(rot.getAttribute("y"));
						Double z = new Double(rot.getAttribute("z"));
						sRot = new Quat4d(x.doubleValue(), y.doubleValue(), z.doubleValue(), w.doubleValue());
					}
					else
					{
						System.out.println("Rot is needs to specified");
						System.exit(-1);
					}
					
					// setting values for velocity of subunit within assembly
					check = subunitElem.getElementsByTagName("Velocity");
					Vector3d sVel = null;
					if(check != null)
					{
						Element rot = (Element) check.item(0);
						Double x = new Double(rot.getAttribute("x"));
						Double y = new Double(rot.getAttribute("y"));
						Double z = new Double(rot.getAttribute("z"));
						
						sVel = new Vector3d(x.doubleValue(),y.doubleValue(),z.doubleValue());
					}
					else
					{
						System.out.println("Vel is needs to specified");
						System.exit(-1);
					}
					
					// setting values for rotational velocity of subunit within assembly
					check = subunitElem.getElementsByTagName("Rotational_Velocity");
					Quat4d srVel = null;
					if(check != null)
					{
						Element rot = (Element) check.item(0);
						
						/*
						if(reloaded)
							r = new Double(rot.getAttribute("w"));
						else
							r = new Double(rot.getAttribute("r"));\
							*/
						Double w = new Double(rot.getAttribute("w"));
						Double x = new Double(rot.getAttribute("x"));
						Double y = new Double(rot.getAttribute("y"));
						Double z = new Double(rot.getAttribute("z"));
						
						srVel = new Quat4d(w.doubleValue(),x.doubleValue(),y.doubleValue(),z.doubleValue());
						/*
						if(reloaded)
							srVel = new Quat4d(x.doubleValue(),y.doubleValue(),z.doubleValue(), r.doubleValue());
						else	
							srVel = new Quat4d(r.doubleValue(),x.doubleValue(),y.doubleValue(),z.doubleValue());
							*/
					}
					else
					{
						System.out.println("Pos is needs to specified");
						System.exit(-1);
					}
					
					String subType = subunitElem.getAttribute("type");
					
					if(reloaded)
					{
						Integer subID = new Integer(subunitElem.getAttribute("name"));
						suId = subID.intValue();
						
					}
					// Subunit type for different subunit is built by calling setSubunitType method
					SubunitType subt = setSubunitType(soln, subType, asmName);
					
					Subunit sub;
					
					if(reloaded)
					{
						Integer subID = new Integer(subunitElem.getAttribute("name"));
						suId = subID.intValue();
						sub = new Subunit(subt, subt.getDomains(), soln, subID.intValue(), sPos, sRot, sVel, srVel);
					}
					else
						sub = new Subunit(subt, subt.getDomains(), soln, suId++, sPos, sRot, sVel, srVel);
					
					//Subunit is colored differently based on their type name to distinguish one from the other
					if(subt.getName().equals("only1"))
						sub.colorBlue();
					else if(subt.getName().equals("only3"))
						sub.colorYellow();
					if(Test.simChoice.equals("tsevensimple"))
					{
						if(subt.getName().equals("only"))
							sub.colorBlue();
						else
							sub.colorYellow();
					}
					asmsubs.add(sub);

					
					String subName = subunitElem.getAttribute("name");
					subMap.put(subName, sub);
					
				}
				Assembly asm = new Assembly(asmsubs, asmName);
				assemblies.add(asm);
				
				//start with oligomers
				if(asmsubs.size() > 1)
					setPolymers(asm, assemElem, asmsubs, subMap);
			}
		}
	}


	/**
	 * 
	 * setPolymers checks if user provied assembly with multiple subunits or not.
	 * If multiple subunits are provided, it binds them togather.
	 * 
	 * @param asm, Assembly that is being checked for multiple subunits.
	 * 
	 * @param assemElem, Node from XML file - <Assembly> - which stores the information
	 * 				     of particular assembly.
	 * @param asmsubs,   Vector of subunits that is in Assembly.
	 * @param subMap,    Hashmap which stores the different subunits
	 * 
	 */
	public void setPolymers(Assembly asm, Element assemElem, Vector asmsubs, HashMap subMap)
	{
		NodeList subList = assemElem.getElementsByTagName("Subunit");
		
		
		for(int k = 0; k < subList.getLength(); k++)
		{		   
			
			Element subElem = (Element) subList.item(k);
			String sub1Name = subElem.getAttribute("name");
			Subunit sub1 = (Subunit) subMap.get(sub1Name);
			NodeList bindList = subElem.getElementsByTagName("bindTo");

			for(int l = 0; l < bindList.getLength(); l++)
			{
				//checks the name of subunits that are binding, and binds them
				// using binding sites that are given
				
				
				Element bindElem = (Element) bindList.item(l);
				String otherSubName = bindElem.getAttribute("subunit");
				
				// binding site name for primary subunit
				String bs1Name = bindElem.getAttribute("bindingSite");
				
				//binding site name for binding subunit
				String bs2Name = bindElem.getAttribute("bindTo");
				
				Vector bss = sub1.getBindSites();
				BindingSite bs1 = null;
				for(int m = 0; m < bss.size(); m++)
				{
					BindingSite tempBs = (BindingSite) bss.get(m);
					if(tempBs.getBSName().equals(bs1Name))
					{
						bs1 = tempBs;
						break;
					}
				}
				
				//binding subunit
				Subunit sub2 = (Subunit) subMap.get(otherSubName);
				bss = sub2.getBindSites();
				BindingSite bs2 = null;
				for(int m = 0; m < bss.size(); m++)
				{
					BindingSite tempBs = (BindingSite) bss.get(m);
					if(tempBs.getBSName().equals(bs2Name))
					{
						bs2 = tempBs;
						break;
					}
				}
				
				
				bindSubunits(asm, sub1, sub2, bs1, bs2);
			}
		}
	}

	/**
	 * This method rotates v2 to make perpendicular to v1
	 * 
	 * @param v1 3D vector that is 
	 * @param v2 3D vector that is being rotated
	 * @return new 3D vector which is perpendicular to v1
	 */
	
	public static Vector3d makePerpendicular(Vector3d v1, Vector3d v2)
	{

		double angle = v1.angle(v2);
		AxisAngle4d rot;
		if(angle == Math.PI/2)
		{
			return new Vector3d(v2);
		}
		else if(angle == 0)
		{
			Vector3d tempAxis = new Vector3d();
			tempAxis.cross(v2, new Vector3d(1,0,0));
			rot = new AxisAngle4d(tempAxis, Math.PI/2);

		}
		else if(angle == Math.PI)
		{
			Vector3d tempAxis = new Vector3d();
			tempAxis.cross(v2, new Vector3d(1,0,0));
			rot = new AxisAngle4d(tempAxis, -Math.PI/2);						
		}
		else
		{
			Vector3d tempAxis = new Vector3d();
			tempAxis.cross(v1, v2);
			rot = new AxisAngle4d(tempAxis, Math.PI/2 - angle);
		}
		Quat4d rotation = new Quat4d();
		rotation.set(rot);
		
		Vector3d temp = rotate(v2, rotation);
		//System.out.println("angle after rotate = " + v1.angle(temp));
		return temp;
		
	}	
	
	/**
	 * This method rotate 3D vector v by given Quaternion q 
	 * @param v Vector3d, 3D vector that is being rotated 
	 * @param q Quat4d, Quaternion which specifies the amount of rotation
	 * 
	 * @return Vector3d 
	 */
    public static Vector3d rotate(Vector3d v, Quat4d q) {
        double[] vd = new double[3];
        v.get(vd);
        Quat4d vq = new Quat4d(vd[0], vd[1], vd[2], 0.0);

        Quat4d rotvq = new Quat4d();
        rotvq.mul(q, vq);
        rotvq.mulInverse(q);

        double[] rotvd = new double[4];
        rotvq.get(rotvd);
        Vector3d rotv = new Vector3d(rotvd[0], rotvd[1], rotvd[2]);
        return rotv;

    }
    /**
     * 
     * @param asm
     * @param sub
     * @param transform
     * @return
     */
	public static Vector3d transformObjectToWorld(Assembly asm, Subunit sub, Vector3d transform)
	{

		Quat4d subRot = sub.getGraphic().getRot();
		Quat4d asmRot = asm.mygraphic.getRot();
		
		
		asmRot.mul(subRot);
		//asmRot.inverse();
		subRot.mul(asmRot);
        double[] vd = new double[3];
        transform.get(vd);
        
        Quat4d vq = new Quat4d(vd[0], vd[1], vd[2], 0.0);
        Quat4d rotvq = new Quat4d();
        
        rotvq.mul(asmRot, vq);
        rotvq.mulInverse(asmRot);

        
        double[] rotvd = new double[4];
        rotvq.get(rotvd);
        Vector3d rotv = new Vector3d(rotvd[0], rotvd[1], rotvd[2]);
        return rotv;
        
	}
	/**
	 * 
	 * This method checks if given binding sites are compatiable and if so, it binds subunits
	 * and move them into correct position and orientation
	 * 
	 * @param asm
	 * @param sub1
	 * @param sub2
	 * @param bs1
	 * @param bs2
	 */
	
	public void bindSubunits(Assembly asm, Subunit sub1, Subunit sub2, BindingSite bs1, BindingSite bs2)
	{
		if(bs1.bindTo(bs2))
		{
			sub1.addPartner(sub2);
			sub2.addPartner(sub1);
			
			if(!reloaded)
			{
				Vector3d oldBsV = bs1.getPos(); 
				asm.centerAssembly(sub2);
			
				Vector3d newBsV = bs2.getPos();
			
				Vector3d vcenter1 = new Vector3d(bs1.getPos());
			
				double angle = oldBsV.angle(newBsV);

				Vector3d axis = new Vector3d();
				axis.cross(newBsV, oldBsV);

				if( !((axis.x == 0) && (axis.y == 0) &&(axis.z == 0)))
				{
					AxisAngle4d rotAngle = new AxisAngle4d(axis, angle - Math.PI);
					sub2.subRotate(rotAngle);
				}
				else if(Math.abs(angle) <= 0.0000001)
				{
					Vector3d tempAxis = new Vector3d();
					tempAxis.cross(oldBsV, new Vector3d(1,0,0));
					AxisAngle4d rotAngle = new AxisAngle4d(tempAxis, angle - Math.PI);

					sub2.subRotate(rotAngle);
				}

		//		if(Test.simChoice.equals("tsevensimple") || Test.simChoice.equals("virus") || Test.simChoice.equals("tthree"))
					vcenter1.scale(2);
		//		else
		//			vcenter1.scale(2 * Test.tipScale);
				sub2.getSubunitGraphic().setPos(vcenter1);				
				Transform3D subtr1 = sub1.getTransform();
				Transform3D subtr2 = sub2.getTransform();
				Transform3D product = new Transform3D();
				product.mul(subtr1, subtr2);
				sub2.setTrans(product);

			
				Vector3d old_upVec = sub1.getST().getUpVec();
				Vector3d new_upVec = sub2.getST().getUpVec();
 	
         	 
				Vector3d oldProj = XMLParser.makePerpendicular(bs1.getPos(), old_upVec);
				Vector3d newProj = XMLParser.makePerpendicular(bs2.getPos(), new_upVec);
         	 
				oldProj = transformObjectToWorld(asm, sub1, oldProj);
				newProj = transformObjectToWorld(asm, sub2, newProj);    	

				Vector3d test = transformObjectToWorld(asm, sub1, bs1.getPos());
				Vector3d test2 = new Vector3d();
				test2.cross(oldProj, newProj);
         	
         			
         		
				double rotAngle = newProj.angle(oldProj);
				
				if( Math.abs(test.angle(test2) - Math.PI) <= 0.0000001)
				{
					rotAngle = -rotAngle;
				}
         	
				String primaryName = bs1.getBST().getName();
				String partnerName = bs2.getBST().getName();
	    		    	
				HashMap partnerAngles = (HashMap) bindingAngles.get(primaryName);
				Double angles = (Double) partnerAngles.get(partnerName);

				AxisAngle4d rotationAxis = new AxisAngle4d(bs1.getPos(), angles.doubleValue() - rotAngle);
				sub2.subRotate(rotationAxis);
			}
		}
	}
					
	/**
	 * 
	 * This method builds subunits based on XML file given by user. User has to provide 
	 * the name of subunit, type of subunit, position and rotation relative to assembly.
	 * @param soln
	 * @param subType
	 * @param asmName
	 * @return SubunitType
	 */

		
	public SubunitType setSubunitType(Solution soln, String subType, String asmName)
	{
		NodeList subTypeList = doc.getElementsByTagName("SubunitType");
		SubunitType subt;
		for(int i = 0; i < subTypeList.getLength(); i++)
		{
			Element subtElem = (Element) subTypeList.item(i);
			
			String name = subtElem.getAttribute("name");
			if(name.equals(subType))
			{
				Vector doms = setDomains(subtElem, asmName);
				
				subt = new SubunitType(doms, name, soln, 3.4, Test.bindingSiteHeight*Test.subunitToBS);
				if(subts.size() == 0)
					subts.add(subt);
				else
				{
					for(int j = 0; j < subts.size(); j++)
					{
						SubunitType tempSubt = (SubunitType) subts.get(j);
						if(tempSubt.getName().equals(subt.getName()))
							break;
						if(j == (subts.size()-1))
							subts.add(subt);
					}
				}
				Element upElem = (Element) subtElem.getElementsByTagName("Up").item(0);
				Double ux = new Double(upElem.getAttribute("x"));
				Double uy = new Double(upElem.getAttribute("y"));
				Double uz = new Double(upElem.getAttribute("z"));
					
				Vector3d upVec = new Vector3d(ux.doubleValue(), uy.doubleValue(), uz.doubleValue());
				subt.storeUpVec(upVec);

				return subt;
			}
		}
		return null;
	}
	/**
	 * 
	 * This method parses XML template, and buids vector of domains.
	 * 
	 * @param subtElem 
	 * 			Element, XML element which stores subunit type information
	 * @param asmName 
	 * 			String, name of assembly that this domain will belong
	 * 
	 * @return Vector
	 */
	public Vector setDomains(Element subtElem, String asmName)
	{
		Vector doms = new Vector();
		
		NodeList domainList = subtElem.getElementsByTagName("Domain");
		//System.out.println("lenth =" + domainList.getLength());
		for(int i = 0; i < domainList.getLength(); i++)
		{
			Element domElem = (Element) domainList.item(i);
			Integer id = new Integer(domElem.getAttribute("id"));
			
			domainId = id.intValue();
			Vector confs = setConformations(domElem, asmName);
			
			String currentConfName = domElem.getAttribute("currentConf");
			Conformation currentConf = null;
			if(currentConfName == null || currentConfName.equals(""))
			{
				currentConf = (Conformation) confs.get(0);
			}
			else
			{
				for(int j =0; j < confs.size(); j++)
				{
					Conformation tempConf = (Conformation) confs.get(j);
					if(tempConf.getName().equals(currentConfName))
					{
						currentConf = tempConf;
						break;
				}
				}
			}
			Domain dom = new Domain("only", confs, new Vector3d(), id.intValue(), currentConf);
			doms.add((dom));
		}
		return doms;
	}
	/**
	 * 
	 * This method sets conformations
	 * 
	 * @param domElem
	 * 			Element, parent element of conformations
	 * 
	 * @param asmName
	 * 			String, name of assembly
	 * 
	 * @return Vector
	 */
	public Vector setConformations(Element domElem, String asmName)
	{
		Vector confs = new Vector();
		NodeList confsList = domElem.getElementsByTagName("Conformation");
		
		for(int i =0; i < confsList.getLength(); i++)
		{
			Element confElem = (Element) confsList.item(i);
			String name = confElem.getAttribute("name");
			Double eValue = new Double(confElem.getAttribute("Evalue"));
			Vector bss = setBindingSites(confElem, asmName);
			
			Conformation conf = new Conformation(bss, eValue.doubleValue(), name);
			//System.out.println("bss length = " + bss.size());
			confs.add(conf);
		}
		return confs;
	}
	/**
	 * 
	 * This method sets binding site for for subunit
	 * 
	 * @param confElem
	 * 		Element, parent element of binding sites
	 * @param asmName
	 * 		String, Assembly name that this binding site is belonged
	 * @return Vector
	 */
	public Vector setBindingSites(Element confElem, String asmName)
	{
		Vector bss = new Vector();

		NodeList bsList = confElem.getElementsByTagName("BindingSite");
		if(bsList == null)
		{
			bss = new Vector();
		}
		else
		{
			for(int i = 0; i < bsList.getLength(); i++)
			{
				Element bsElem = (Element) bsList.item(i);
				String bstName = bsElem.getAttribute("type");
				String bsName = bsElem.getAttribute("name");
				
				BindingSiteType bst = (BindingSiteType) bstMap.get(bstName);
				if(bst == null)
				{
					System.out.println("No such binding site type was declared");
					System.exit(-1);
				}
				
				NodeList check = bsElem.getElementsByTagName("Bpos");
				Vector3d posD = null;
				if(check != null)
				{
					Element pos = (Element) check.item(0);
					Double bx = new Double(pos.getAttribute("x"));
					Double by = new Double(pos.getAttribute("y"));
					Double bz = new Double(pos.getAttribute("z"));
											
					posD = new Vector3d(bx.doubleValue(), by.doubleValue(), bz.doubleValue());
				}
				else
				{
					System.out.println("Bpos is not specified");
					System.exit(-1);
				}
				
				check = bsElem.getElementsByTagName("Orientation");
				Quat4d orD = null;
				if(check != null)
				{
					Element orElem = (Element) check.item(0);
					
					Double or = new Double(orElem.getAttribute("w"));					
					Double ox = new Double(orElem.getAttribute("x"));
					Double oy = new Double(orElem.getAttribute("y"));
					Double oz = new Double(orElem.getAttribute("z"));

					if(reloaded)
						orD = new Quat4d(ox.doubleValue(), oy.doubleValue(), oz.doubleValue(), or.doubleValue());
					
					else if(Test.simChoice.equals("actin") || Test.simChoice.equals("tubulin") || Test.simChoice.equals("tetrahedron") || Test.simChoice.equals("tsevensimple"))
					{
							orD = new Quat4d(ox.doubleValue(), oy.doubleValue(), oz.doubleValue(), or.doubleValue());
					}
					else
					{
						AxisAngle4d tempOrD;
							tempOrD = new AxisAngle4d(ox.doubleValue(), oy.doubleValue(), oz.doubleValue(), or.doubleValue());					
						orD = new Quat4d();
					
						orD.set(tempOrD);
					}

				}
				else
				{
					System.out.println("Borientation is not specified");
					System.exit(-1);
				}
				BindingSite bs = new BindingSite(bst, bsId++, domainId, suId, asmName, posD, orD, bsName);
			
				if(bstName.equals("bsta"))
				{
					bs.colorYellow();
				}
				else if(bstName.equals("bstb"))
				{
					bs.colorGreen();
				}
				else if(bstName.equals("bstc"))
				{
					bs.colorBlue();
				}

				bss.add(bs);
			}
		}
		return bss;
	}
	/**
	 * 
	 * This method sets different transition times. Binding time, breaking time, fast binding,
	 * and conformational switching time are loaded at here.
	 * 
	 * @param soln
	 * @param assemblies
	 */
	public void setTransitionTime(Solution soln, Vector assemblies)
	{
		Element bindingInteract = (Element) doc.getElementsByTagName("BindingInteract").item(0);
		NodeList breakList = bindingInteract.getElementsByTagName("BindBreak");
		HashMap bondTimes = new HashMap();
		
		for(int i = 0; i < breakList.getLength(); i++)
		{
		    
			Element breakElem = (Element) breakList.item(i);
			NodeList timeList = breakElem.getElementsByTagName("Time");
			HashMap newMap = new HashMap();
			
			for(int j = 0; j < timeList.getLength(); j++)
			{
				Element timeElem = (Element) timeList.item(j);
				String name = new String(timeElem.getAttribute("name"));
				Double bindTime = new Double(timeElem.getAttribute("bindTime"));
				Double breakTime = new Double(timeElem.getAttribute("breakTime"));
				Double fastbindTime = new Double(timeElem.getAttribute("fastbindTime"));
				
				double [] bindBreak = new double[3];
				bindBreak[0] = bindTime.doubleValue();
				bindBreak[1] = breakTime.doubleValue();
				bindBreak[2] = fastbindTime.doubleValue();
				
				newMap.put(name, bindBreak);
			}
			String name = breakElem.getAttribute("name");
			bondTimes.put(name, newMap);
		}
		
		NodeList conf_List = bindingInteract.getElementsByTagName("ConformationTime");
		HashMap conftimes = new HashMap();
		for(int i = 0; i < conf_List.getLength(); i++ )
		{
			Element conf = (Element) conf_List.item(i);
			String name = conf.getAttribute("name");
			NodeList list = conf.getElementsByTagName("List");
			HashMap map = new HashMap();
			
			for(int j = 0; j < list.getLength(); j++)
			{
				Element item = (Element) list.item(j);
				map.put(item.getAttribute("name"), new Double(item.getAttribute("time")));
			}
			
			conftimes.put(name, map);
		}
		//System.out.println("sdfafs: "+bondTimes);
		Element sysElem;
		Double currentTime = new Double(0.0);
		Integer currentStep = new Integer(0);
		
		if(reloaded)
		{
			
			sysElem = (Element) doc.getElementsByTagName("System").item(0);
			currentTime = new Double(sysElem.getAttribute("currentTime"));
			Integer prevStep = new Integer(sysElem.getAttribute("currentStep"));			
			currentStep = new Integer(prevStep.intValue() + 1);		
		}
		
		sim = new Simulation(soln, assemblies, currentTime.doubleValue(), bondTimes, conftimes, partnerMap);
		sim.storeBindingSiteTypes(bsts);
		sim.storeSubunitTypes(subts);
		if(reloaded)
		{
			sim.asnum = asmNum + 1;
			sim.currentStep = currentStep.intValue();
		}
		
	}

	/**
	 * 
	 * This method is main method for writing current state of simulation in to XML
	 * 
	 * @param simulator
	 * 		Simulation, simulator that is currently runing
	 * @param fileName
	 * 		String, file path where XML will be stored
	 */
	public static void writeToXML(Simulation simulator, String fileName)
	{
		try{
			File f = new File(fileName);
			DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
			DocumentBuilder db = dbf.newDocumentBuilder();
			
			Document document = db.parse(f);
			Element root = document.getDocumentElement();
			
			root.setAttribute("type", Test.simChoice);
			Integer asmNum = new Integer(simulator.asnum);
			root.setAttribute("AssemblyNumber", asmNum.toString());
			root.setAttribute("currentTime", ""+simulator.curtime);
			root.setAttribute("currentStep", ""+(simulator.currentStep -1));
			
			writeSolution(document, simulator, root);

			writeAssembly(document, simulator, root);

			writeBStypes(document, simulator, root);

			writeBindingInteract(document, simulator, root);

			writeSubunitType(document, simulator, root);


			 TransformerFactory tFactory = TransformerFactory.newInstance();
             
             SAXTransformerFactory sft = (SAXTransformerFactory) tFactory;
             
             FileWriter writer = new FileWriter(fileName);
             
     StreamResult result = new StreamResult(writer);
     
     DOMSource source = new DOMSource(document);
     Transformer transformer = sft.newTransformer();
     transformer.transform(source, result);

     writer.close();

     
     BufferedReader bfr = new BufferedReader(new FileReader(fileName));
     StringBuffer readable = new StringBuffer(fileName);

     int insertIndex = -1;
     for(int i = 0; i < readable.length(); i++)
     {
             char ch = readable.charAt(i);
             if(ch == '.')
             {
                 insertIndex = i;
             }
     }
     
     if(insertIndex != -1)
     {
         readable.insert(insertIndex, "_readable");
     }
     
     BufferedWriter bfw = new BufferedWriter(new FileWriter(readable.toString()));
     char ch  = (char) bfr.read();
     
     int tagNumber = -1;
     while(((int) ch) != 65535)
     {
             if(ch == '<')
             {
                     char checkCh = (char) bfr.read();
                     if(checkCh == '/')
                     {
                             tagNumber--;
                     }
                     for(int i = 0; i < tagNumber; i++)
                     {
                             bfw.write('\t');
                     }
                     
                     if(checkCh != '/')
                             tagNumber++;
                     
             bfw.write(ch);
             bfw.write(checkCh);
             }
             else if(ch =='/')
             {
                     char checkCh = (char) bfr.read();
                     bfw.write(ch);
                     bfw.write(checkCh);
                     if(checkCh == '>')
                     {
                             bfw.newLine();                          
                             tagNumber--;
                     }
             }
             else if(ch == '>')
             {
                     bfw.write(ch);
                     bfw.newLine();
             }
             else
             {
                     bfw.write(ch);
             }
             bfw.flush();
             ch = (char) bfr.read();
     }
     
     bfw.flush();
     bfw.close();

			} catch (Exception e){
					System.out.println(e);
					System.exit(-1);
				}		
		
	}
	/**
	 * 
	 * This method writes solution in to XML
	 * 
	 * @param doc
	 * 		Document, document where XML will be saved
	 * @param sim
	 * 		Simulation, simulator that is currently running
	 * @param root
	 * 		Element, parent of solution element
	 */
	public static void writeSolution(Document doc, Simulation sim, Element root)
	{
		Solution soln = sim.mysoln;
		Element solnElem = doc.createElement("Solution");
		root.appendChild(solnElem);
		Double temp = new Double(soln.getProperty("temp"));
		Double pH = new Double(soln.getProperty("pH"));
		Double volume = new Double(soln.getProperty("volume"));
		
		solnElem.setAttribute("temp", temp.toString());
		solnElem.setAttribute("pH", pH.toString());
		solnElem.setAttribute("volume", volume.toString());
		
	}
	/**
	 * 
	 * This method goes through all the assemblies, and write them in to XML
	 * 
	 * @param document
	 * 		Document, document where XML will be saved
	 * @param sim
	 * 		Simulation, simulator that is currently running
	 * @param root
	 * 		Element, parent of assembly element
	 */
	public static void writeAssembly(Document document, Simulation simulator, Element root)
	{
		Vector assemblies = simulator.getAssemblies();
		for(int i=0; i < assemblies.size(); i++)
		{
			Assembly asm = (Assembly) assemblies.get(i);
			Element asmElem = document.createElement("Assembly");
			asmElem.setAttribute("name", asm.getName());

			asmElem.setAttribute("amount", "1");
			root.appendChild(asmElem);
			writeSubunit(document, asm, asmElem);
		}
	}
	
	/**
	 * 
	 * This method goes through all the subunits with in assembly, and write tem into
	 * XML
	 * 
	 * @param doc
	 * 		Document, document where XML will be saved
	 * @param asm
	 * 		Assembly, assemly that hold subunits
	 * @param asmElem
	 * 		Element, parent of subunit element
	 */
	public static void writeSubunit(Document document, Assembly asm, Element asmElem)
	{
		Vector subs = asm.getSubunits();
		for(int i =0; i < subs.size(); i++)
		{
			Subunit sub = (Subunit) subs.get(i);
			Element subElem = document.createElement("Subunit");
			asmElem.appendChild(subElem);
			Integer suID = new Integer(sub.getid());
			
			subElem.setAttribute("name", suID.toString());
			subElem.setAttribute("type", sub.getSubunitTypeName());
			
			Element posElem = document.createElement("Position");
			Vector3d pos = sub.getPos();
			Double xpos = new Double(pos.x);
			Double ypos = new Double(pos.y);
			Double zpos = new Double(pos.z);
			
			posElem.setAttribute("x", xpos.toString());
			posElem.setAttribute("y", ypos.toString());
			posElem.setAttribute("z", zpos.toString());
			
			subElem.appendChild(posElem);
			
			Element rotElem = document.createElement("Rotation");
			Quat4d rot = sub.getRot();
			Double wrot = new Double(rot.w);
			Double xrot = new Double(rot.x);
			Double yrot = new Double(rot.y);
			Double zrot = new Double(rot.z);
			
			rotElem.setAttribute("x", xrot.toString());
			rotElem.setAttribute("y", yrot.toString());
			rotElem.setAttribute("z", zrot.toString());
			rotElem.setAttribute("w", wrot.toString());
			
			subElem.appendChild(rotElem);
			
			Element velElem = document.createElement("Velocity");
			Vector3d vel = sub.getV();
			Double vx = new Double(vel.x);
			Double vy = new Double(vel.y);
			Double vz = new Double(vel.z);
			
			velElem.setAttribute("x", vx.toString());
			velElem.setAttribute("y", vy.toString());
			velElem.setAttribute("z", vz.toString());
			
			subElem.appendChild(velElem);
			
			Element rotVelElem = document.createElement("Rotational_Velocity");
			Quat4d vrot = sub.getRotV();
			Double vwrot = new Double(vrot.w);
			Double vxrot = new Double(vrot.x);
			Double vyrot = new Double(vrot.y);
			Double vzrot = new Double(vrot.z);
			
			rotVelElem.setAttribute("x", vxrot.toString());
			rotVelElem.setAttribute("y", vyrot.toString());
			rotVelElem.setAttribute("z", vzrot.toString());
			rotVelElem.setAttribute("w", vwrot.toString());
			
			subElem.appendChild(rotVelElem);
			
			Vector bss = sub.getBindSites();
			
			for(int j =0; j < bss.size(); j++)
			{
				BindingSite bs = (BindingSite) bss.get(j);
				if(bs.isBound())
				{
					Element bindElem = document.createElement("bindTo");
					BindingSite bsPartner= bs.getPartner();
					Integer partnerSubID = new Integer(bsPartner.getSubunitID());
					bindElem.setAttribute("subunit", partnerSubID.toString());
					bindElem.setAttribute("bindTo", bsPartner.getBST().getName());
					bindElem.setAttribute("bindingSite", bs.getBST().getName());

					subElem.appendChild(bindElem);
				}
			}
		}
	}
	
	/**
	 * 
	 * This method search thrrough all the binding site types, their partners, and
	 * their binding angles
	 * 
	 * @param doc
	 * 		Document, document where XML will be saved
	 * @param sim
	 * 		Simulation, simulator that is currently running
	 * @param root
	 * 		Element, parent of bindingSite type element
	 */
	public static void writeBStypes(Document doc, Simulation sim, Element root)
	{
		Element bstElem = doc.createElement("BindingSiteTypes");
		root.appendChild(bstElem);
		Vector bsts = sim.getBindingSiteTypes();
		HashMap bstPartner = sim.getBSTPartner();
		
		for(int i = 0; i < bsts.size(); i++)
		{	
			Element elem = doc.createElement("BindingSiteType");
			
			bstElem.appendChild(elem);
			
			BindingSiteType bst = (BindingSiteType) bsts.get(i);
			String bstName = bst.getName();
			elem.setAttribute("name", bstName);
						
			Element tolElem = doc.createElement("tolerance");
			double[] tolerances = bst.getTolerance();
			
			Double trans = new Double(tolerances[0]);
			Double tors = new Double(tolerances[1]);
			Double bend = new Double(tolerances[2]);
			
			tolElem.setAttribute("transition", trans.toString());
			tolElem.setAttribute("torsion", tors.toString());
			tolElem.setAttribute("bending", bend.toString());
			
			elem.appendChild(tolElem);
			

			Vector partners = (Vector) bstPartner.get(bstName);
			for(int j = 0; j < partners.size(); j++)
			{
				String partnerName = (String) partners.get(j);
				Element partElem = doc.createElement("partner");
				partElem.setAttribute("name", partnerName);
				elem.appendChild(partElem);
				Double angle = new Double(bst.getBindingAngle(partnerName));
				
				partElem.setAttribute("angle", angle.toString());

			}
		}
	}
	
	/**
	 * 
	 * This method goes through possible binding interactions - binding time, breaking time
	 * and fast binding time - and write them into XML.
	 * 
	 * Also, this method records the conformational switching time.
	 * 
	 * @param doc
	 * 		Document, document where XML will be saved
	 * @param sim
	 * 		Simulation, simulator that is currently running
	 * @param root
	 * 		Element, parent of BindingInteract element
	 */
	public static void writeBindingInteract(Document doc, Simulation sim, Element root)
	{
		Element bindingElem = doc.createElement("BindingInteract");
		root.appendChild(bindingElem);
		
		HashMap bondTimes = sim.getBondTime();
		Iterator keys = bondTimes.keySet().iterator();

		while(keys.hasNext())
		{
			String name = (String) keys.next();
			Element bindbreakElem = doc.createElement("BindBreak");
			bindingElem.appendChild(bindbreakElem);
			bindbreakElem.setAttribute("name", name);
			
			HashMap partners = (HashMap) bondTimes.get(name);
			Iterator partnerKeys = partners.keySet().iterator();
			
			while(partnerKeys.hasNext())
			{
				String partnerName = (String) partnerKeys.next();
				Element partnerElem = doc.createElement("Time");
				bindbreakElem.appendChild(partnerElem);
				
				partnerElem.setAttribute("name", partnerName);
				
				double[] times = (double[]) partners.get(partnerName);
				Double bindTime = new Double(times[0]);
				Double breakTime = new Double(times[1]);
				Double fastbindTime = new Double(times[2]);
				
				partnerElem.setAttribute("bindTime", bindTime.toString());
				partnerElem.setAttribute("breakTime", breakTime.toString());
				partnerElem.setAttribute("fastbindTime", fastbindTime.toString());
			}
		}
		HashMap confTimes = sim.getConfTime();
		
		Iterator confTime = confTimes.keySet().iterator();
		while(confTime.hasNext())
		{
			Element confTimeElem = doc.createElement("ConformationTime");
			bindingElem.appendChild(confTimeElem);
			String confName = (String) confTime.next();
			
			confTimeElem.setAttribute("name", confName);
			
			HashMap listMap = (HashMap) confTimes.get(confName);			
			Iterator lists = listMap.keySet().iterator();
			
			while(lists.hasNext())
			{
				Element listElem = doc.createElement("List");
				confTimeElem.appendChild(listElem);
				String listName = (String) lists.next();
				Double time = (Double) listMap.get(listName);
				
				listElem.setAttribute("name", listName);
				listElem.setAttribute("time", time.toString());
			}
		}
	}
	
	/**
	 * 
	 * This method writes all the subunit types in the simulation into XML
	 * 
	 * @param doc
	 * 		Document, document where XML will be saved
	 * @param sim
	 * 		Simulation, simulator that is currently running
	 * @param root
	 * 		Element, parent of subunitType element
	 */
	public static void writeSubunitType(Document doc, Simulation sim, Element root)
	{
		Vector subts = sim.getSubTypes();

		for(int i = 0; i < subts.size(); i++)
		{
			SubunitType subt = (SubunitType) subts.get(i);
			Element subtElem = doc.createElement("SubunitType");
			root.appendChild(subtElem);
			String subtName = subt.getName();
			subtElem.setAttribute("name", subtName);
			Element upVecElem = doc.createElement("Up");
			Vector3d upVec = subt.getUpVec();
			upVecElem.setAttribute("x", ""+upVec.x);
			upVecElem.setAttribute("y", ""+upVec.y);
			upVecElem.setAttribute("z", ""+upVec.z);
			
			subtElem.appendChild(upVecElem);
			writeDomains(doc, subt, subtElem);
		}		
	}
	
	/**
	 * 
	 * This method writes different domains in the subunit type in to XML
	 * 
	 * @param doc
	 * 		Document, document that XML will be saved
	 * @param subt
	 * 		SubunitType, SubunitType which holds domains
	 * @param subtElem
	 * 		Element, parent element of domains
	 */
	public static void writeDomains(Document doc, SubunitType subt, Element subtElem)
	{
		Vector domains = subt.getDomains();
		for(int i = 0; i < domains.size(); i++)
		{
			Domain dom = (Domain) domains.get(i);
			Element domElem = doc.createElement("Domain");
			subtElem.appendChild(domElem);
			
			Integer domainID = new Integer(dom.dmid);
			String curConfName = dom.curconf.getName();

			domElem.setAttribute("id", domainID.toString());
			domElem.setAttribute("currentConf", curConfName);
			writeConformation(doc, dom, domElem);
		}
	}
	
	/**
	 * 
	 * This method checks possible conformational switch for subunit type, and 
	 * write them into XML.
	 * 
	 * @param doc
	 * @param dom
	 * @param domElem
	 */
	public static void writeConformation(Document doc, Domain dom, Element domElem)
	{
		Vector confs = dom.getConfs();
		for(int i =0; i < confs.size(); i++)
		{
			Conformation conf = (Conformation) confs.get(i);
			Element confElem = doc.createElement("Conformation");
			domElem.appendChild(confElem);
			
			String name = conf.getName();
			Double eValue = new Double(conf.getE());
			
			confElem.setAttribute("name", name);
			confElem.setAttribute("Evalue", eValue.toString());
			writeBindingSite(doc, conf, confElem);
		}
	}
	/**
	 * 
	 * @param doc
	 * @param conf
	 * @param confElem
	 */
	public static void writeBindingSite(Document doc, Conformation conf, Element confElem)
	{
		Vector bss = conf.getBindSites();
		if(bss != null)
		{
			for(int i = 0; i < bss.size(); i++)
			{
				BindingSite bs = (BindingSite) bss.get(i);
				Element bsElem = doc.createElement("BindingSite");
				confElem.appendChild(bsElem);
				String bstName = bs.getBST().getName();
			
				bsElem.setAttribute("name", bstName);
				bsElem.setAttribute("type", bstName);
			
				Element bposElem = doc.createElement("Bpos");
				bsElem.appendChild(bposElem);

				Vector3d pos = bs.getPos();
				Double xpos = new Double(pos.x);
				Double ypos = new Double(pos.y);
				Double zpos = new Double(pos.z);
			
				bposElem.setAttribute("x", xpos.toString());
				bposElem.setAttribute("y", ypos.toString());
				bposElem.setAttribute("z", zpos.toString());
			

				Element orElem = doc.createElement("Orientation");
				bsElem.appendChild(orElem);
			
				Quat4d rot = bs.getRot();
				
				Double wrot = new Double(rot.w);
				Double xrot = new Double(rot.x);
				Double yrot = new Double(rot.y);
				Double zrot = new Double(rot.z);
			
				orElem.setAttribute("w", wrot.toString());
				orElem.setAttribute("x", xrot.toString());
				orElem.setAttribute("y", yrot.toString());
				orElem.setAttribute("z", zrot.toString());
			}
		}
	}
	
	
}
