
import java.util.Vector;
import java.util.HashMap;
import javax.vecmath.*;

/**
 * Tetrahedron sets up a tetrahedron-like Simulation, where there is one type of
 * Subunit with a single Conformation and Domain and three binding sites. The
 * initial number of Subunits, and the binding and breaking times control the
 * assembly of tetrahedron
 * 
 * @author Tiequan Zhang
 * @version 1.3
 */

public class Tetrahedron {
//  Initial number of Subunits are set here.
    private static int numSubunits = Test.numSubunits;
    private static int bsid = 0; //BindingSite ID

    private static int domainID = 0; //Domain ID

    private static int stid = 0; //SubunitType ID

    private static int suid = 0; //Subunit ID

   // private static double radius = Test.subunitRadius; //radius of subunit

    //binding and breaking rates set here:
    //average time for a binding event between two compatible BindingSites
    private double bindTime = Test.bindTime;

    //average time for a breaking event between two compatible BindingSites
    private double breakTime = Test.breakTime;

    //average time for a bonding event between two compatible BindingSites,
    //positioned to bind, in the same Assembly
    private double fastbindTime = Test.fastbindtime;

    //  average time for enabling binding ability
    private double bindToNoneTime = Test.bindToNoneTime;

    //  average time for disabling binding ability
    private double noneToBindTime = Test.noneToBindTime;

    

    private static HashMap bindingPartner = new HashMap();
    private static Simulation sim;

    private static Solution soln;

    private static Vector assemblies;


   private static Vector bindingSiteTypes = new Vector();
   private static Vector subunitTypes = new Vector();
    
    public Tetrahedron() {
        soln = new Solution();
        initAssembly();

        double[] bindBreak = new double[3];
        bindBreak[0] = bindTime;
        bindBreak[1] = breakTime;
        bindBreak[2] = fastbindTime;
        double[] bindBreak2 = new double[3];
        bindBreak2[0] = bindTime;
        bindBreak2[1] = breakTime;
        bindBreak2[2] = fastbindTime;
        
        double[] noBindBreak = new double[3];
        noBindBreak[0] = 99999999999999.0;
        noBindBreak[1] = 0.00000000000001;
        noBindBreak[2] = 99999999999999.0;

        HashMap bstaHash = new HashMap();
        bstaHash.put(new String("bsta"), bindBreak);
        bstaHash.put(new String("bstb"), noBindBreak);
        bstaHash.put(new String("bstc"), noBindBreak);

        HashMap bstbHash = new HashMap();
        bstbHash.put(new String("bstb"), (double[]) bindBreak);
        bstbHash.put(new String("bsta"), (double[]) noBindBreak);
        bstbHash.put(new String("bstc"), (double[]) noBindBreak);

        HashMap bstcHash = new HashMap();
        bstcHash.put(new String("bsta"), (double[]) noBindBreak);
        bstcHash.put(new String("bstb"), (double[]) noBindBreak);
        bstcHash.put(new String("bstc"), (double[]) bindBreak);

        HashMap bondtimes = new HashMap();
        bondtimes.put(new String("bsta"), bstaHash);
        bondtimes.put(new String("bstb"), bstbHash);
        bondtimes.put(new String("bstc"), bstcHash);

        HashMap bsMap = new HashMap(), noneMap = new HashMap();
        bsMap.put("none", new Double(bindToNoneTime));
        noneMap.put("bs", new Double(noneToBindTime));

        HashMap conftimes = new HashMap();
        conftimes.put("bs", bsMap);
        conftimes.put("none", noneMap);

        sim = new Simulation(soln, assemblies, 0.0, bondtimes, conftimes,
                bindingPartner);
        
        sim.storeBindingSiteTypes(bindingSiteTypes);
        sim.storeSubunitTypes(subunitTypes);
    }
    public Tetrahedron(String xml) {
        
        soln = new Solution();
        XMLParser test = new XMLParser(Test.mode);
        assemblies = new Vector();
        test.setBindingSiteTypes();
        test.setAssemblies(assemblies, soln);
        test.setTransitionTime(soln, assemblies);
        sim = test.sim;
        
    }
    /**
     * Returns Simulation of Tetrahedron
     * 
     * @return Simulation
     *  
     */
    public static Simulation getSim() {
        return sim;
    }

    /**
     * Creates all Subunits with appropriate bsID, domainID, ConformationID and
     * stID. Adds all Assemblies to Vector assemblies (which is later used in
     * the Simulation construction)
     */
    private static void initAssembly() {
        assemblies = new Vector();

        //bindingsitetype definitions
        double[] tolerances = { 0.2, 0.1, 0.1 };
        Vector partnersa = new Vector();
        Vector partnersb = new Vector();
        Vector partnersc = new Vector();

        partnersa.add("bsta");
        BindingSiteType bsta = new BindingSiteType(tolerances, "bsta",
                partnersa);
        bsta.setBindingAngle("bsta", 2.6721173923948456);
        
        partnersb.add("bstb");
        BindingSiteType bstb = new BindingSiteType(tolerances, "bstb",
                partnersb);
        bstb.setBindingAngle("bstb", -0.42591627350063505);
        
        partnersc.add("bstc");
        BindingSiteType bstc = new BindingSiteType(tolerances, "bstc",
                partnersc);
        bstc.setBindingAngle("bstc", 2.739364542972453);

        bindingSiteTypes.add(bsta);
        bindingSiteTypes.add(bstb);
        bindingSiteTypes.add(bstc);
        
        
        bindingPartner.put("bsta", partnersa);
        bindingPartner.put("bstb", partnersb);
        bindingPartner.put("bstc", partnersc);

        AxisAngle4d axisAnglea = new AxisAngle4d(1,0,0, Math.acos(Math
                .sqrt(1.0 / 3)));
        Quat4d rotationa = new Quat4d();
        rotationa.set(axisAnglea);
        Vector3d positiona = new Vector3d(Test.bindingSiteHeight *0, Test.bindingSiteHeight * Math.sqrt(1.0 / 3), Test.bindingSiteHeight *
                 Math.sqrt(2.0 / 3));

        
        
        
        AxisAngle4d axisAngleb1 = new AxisAngle4d(0, 0, 1, 2.0 / 3 * Math.PI);
        Quat4d qb = new Quat4d();
        qb.set(axisAngleb1);
        
        AxisAngle4d axisAngleb2 = new AxisAngle4d(-1 / 6.0, Math.sqrt(3) / 6, 0, Math.acos(Math
                .sqrt(1.0 / 3)));
        Quat4d rotationb = new Quat4d();
        rotationb.set(axisAngleb2);
        rotationb.mul(qb);
        Vector3d positionb = new Vector3d(-Test.bindingSiteHeight  / 2, -Test.bindingSiteHeight * Math.sqrt(3)
                / 6.0, Test.bindingSiteHeight  * Math.sqrt(2.0 / 3));
        

        AxisAngle4d  axisAnglec1 = new AxisAngle4d(0, 0, 1, -2.0 / 3 * Math.PI);
        Quat4d qc = new Quat4d();
        qc.set(axisAnglec1);

        AxisAngle4d axisAnglec2 = new AxisAngle4d(-1 / 6.0, -Math.sqrt(3) / 6, 0, Math.acos(Math
                .sqrt(1.0 / 3)));
        Quat4d rotationc = new Quat4d();
        rotationc.set(axisAnglec2);
        rotationc.mul(qc);

        Vector3d positionc = new Vector3d(Test.bindingSiteHeight / 2,
                -Test.bindingSiteHeight * Math.sqrt(3) / 6.0, Test.bindingSiteHeight * Math.sqrt(2.0 / 3));

        
        double subunitRadius=Test.subunitToBS*Test.bindingSiteHeight;


        boolean subunitTypeNotSeen = true;
        //loop for making subunittypes
        for (int i = 0; i < numSubunits; i++) {
            String asmname = "assem" + i;
            BindingSite bsa = new BindingSite(bsta, bsid++, domainID, suid,
                    asmname, positiona, rotationa, "bsa");
            BindingSite bsb = new BindingSite(bstb, bsid++, domainID, suid,
                    asmname, positionb, rotationb,  "bsb");
            BindingSite bsc = new BindingSite(bstc, bsid++, domainID, suid,
                    asmname, positionc, rotationc,  "bsc");
            
            
            bsa.colorYellow();
            bsb.colorGreen();
            bsc.colorBlue();
            /**
             * Adds BindingSites of one Subunit to Vector bss to create
             * Conformation
             */
            Vector bss = new Vector();
            bss.add(bsa);
            bss.add(bsb);
            bss.add(bsc);

            /**
             * Adds one Conformation of a Subunit to Vector confs to create
             * Domain
             */
            Vector confs = new Vector();
            Conformation bsConf = new Conformation(bss, 6.6, "bs");
            Conformation nonBsConf = new Conformation(new Vector(), 6.6, "none");
            confs.add(bsConf);
            confs.add(nonBsConf);
            /**
             * Adds one Domain of a Subunit to Vector Confsets to create
             * SubunitType
             */
            Domain dom = new Domain("only", confs, new Vector3d(), domainID,
                    bsConf);
            if (Test.csAllowed) {
                dom = new Domain("only", confs, new Vector3d(), domainID,
                        nonBsConf);
            }
            Vector doms = new Vector();
            doms.add(dom);

            SubunitType subType = new SubunitType(doms, "only", soln, 3.4, subunitRadius);
            subType.storeUpVec(new Vector3d(1,1,1));
            
            if (subunitTypeNotSeen) {
                subunitTypes.add(subType);
                subunitTypeNotSeen = false;
            }
            
            /**
             * set initial position, rotation, velocity, and rotational velocity
             */
            Vector3d sPos = new Vector3d();
            Quat4d sRot = new Quat4d(0.0, 0.0, 0.0, 1.0);
            Vector3d sVel = new Vector3d();
            Quat4d srVel = new Quat4d(0.0, 0.0, 0.0, 1.0);

            Subunit subunit = new Subunit(subType, doms, soln, suid++, sPos, sRot,
                    sVel, srVel);

            Vector assemblySubunits = new Vector();
            assemblySubunits.add(subunit);
            Assembly asm = new Assembly(assemblySubunits, asmname);

            assemblies.add(asm);
        }

    }

}