
import java.util.Vector;
import java.lang.Math;
import javax.vecmath.*;
import javax.media.j3d.*;

/**
 * Subunit holds information about a specific Subunit including its SubunitType,
 * current Conformations, ID, Solution, location/velocity mass, radius, bound
 * Subunits, and graphic.
 * 
 * @author Rori Rohlfs
 * @version 1.3
 * @Todo addpartner:how to make unique partner added to the vector
 *  
 */

public class Subunit {
    private SubunitType mySubunitType;

   //not in use
    private Vector myConfs; //Vector of current Conformations (for each Subunit

    // Domain)

    private int myID;

    private Solution mySolution;

    public Vector3d myPosition; //relative to the assembly

    public Quat4d myRotation; //relative to the assembly

    private Vector3d myv; //relative to the assembly

    private Quat4d myrotv; //relative to the assembly

    private Vector myPartners; //Vector of bound Subunits

    private Vector myDomains; //Vector of Domains

    private SubunitGraphic myGraphic;

   

    /**
     * Constructs a Subunit with the specified SubunitType, current
     * Conformations, Solution, ID, position, rotation, velocity, angular
     * velocity, mass, and radius. A graphic is construced according to the
     * parameters given
     * 
     * @param st
     *            SubunitType
     * @param doms
     *            Vector of Domains in this Subunit
     * @param soln
     *            Solution this Simulation takes place in
     * @param id
     *            int, this Subunit's ID
     * @param p
     *            Vector3d, this Subunit's position relative to its Assembly
     * @param r
     *            Quat4d, this Subunit's rotation relative to its Assembly
     * @param v
     *            Vector3d, this Subunit's velocity relative to its Assembly
     * @param rv
     *            Quat4d, this Subunit's rotational velocity relative to its
     *            Assembly
     */
    public Subunit(SubunitType st, Vector doms, Solution soln, int id,
            Vector3d p, Quat4d r, Vector3d v, Quat4d rv) {
        mySubunitType = st;
        myID = id;
        mySolution = soln;

        myPosition = p;
        myRotation = r;
        myv = v;
        myrotv = rv;

        myPartners = new Vector();
        myDomains = doms;

        myGraphic = new SubunitGraphic(this);
    }
  
    /**
     * Returns the change in Energy between the Conformations specified in
     * Vector confsa and those specified in confsb
     * 
     * @param confsa
     *            Vector of Conformations (one for each Domain in this Subunit)
     * @param confsb
     *            Vector of Conformations (one for each Domain in this Subunit)
     * @return double
     */
    private double deltaE(Vector confsa, Vector confsb) {
        double aesum = 0.0;
        double besum = 0.0;
        for (int i = 0; i < confsa.size(); i++) {
            aesum += ((Conformation) confsa.elementAt(i)).getE();
            besum += ((Conformation) confsb.elementAt(i)).getE();
        }
        return (besum - aesum);
    }

    /**
     * Finds the next set of Conformations this Assembly will take and returns
     * it along with the time until the conformation change. This function is
     * not presently used and is somwhat dated
     * 
     * @return Vector [Vector of Conformations, transition time Double]
     */
    private Vector NextConf() {
        Vector otherConfs = (Vector) mySubunitType.getConfSets();
        Vector eVals = new Vector();
        double sum = 0.0;
        double kb = 1.38 * (Math.pow(10.0, -23.0));
        double t = mySolution.getProperty("temp");
        double de = 0.0;
        Double newguy;

        Vector myConfs = new Vector();
        for (int i = 0; i < myDomains.size(); i++)
            myConfs.add(((Domain) myDomains.get(i)).getCurConf());

        //find total Ks's and make a vector of them
        for (int i = 0; i < otherConfs.size(); i++) {
            de = deltaE(myConfs, (Vector) (otherConfs.elementAt(i)));
            double tmp = Math.pow(Math.E, (-(de / (kb * t))));
            sum += tmp;
            newguy = new Double(tmp);
            eVals.add(newguy);
        }

        //get a random double in [0, Ks sum]
        double rand = 0.0;//s.getRand();MUST BE CHANGED WHEN CONFORMATINOS
        // START
        rand *= sum;

        //find corresponding conformation
        double sum2 = 0.0;
        int index = 0;
        while (sum2 < rand) {
            sum2 += ((Double) eVals.elementAt(index)).doubleValue();
            index++;
        }

        return (Vector) eVals.elementAt(index);
    }

    /**
     * Changes the Conformation of Domain number domain to newconf. If the
     * conformation change failed, returns false, returns true otherwise.
     * 
     * @param dmid
     *            int, ID of the Domain in question
     * @param newconf
     *            Conformation to be transformed into
     * @return boolean
     */
    public boolean changeConf(int dmid, Conformation newConf) {
        if (mySubunitType.isConf(newConf, dmid)
                && !(((Domain) myDomains.get(dmid)).getCurConf()).bound()) {
            Domain dom = (Domain) myDomains.get(dmid);
            Conformation curConf = dom.getCurConf();
            myGraphic.changeConformation(curConf, newConf);
            dom.changeConf(newConf);

            return true;
        }
        return false;
    }
    
    public void colorBlue() {
        myGraphic.colorBlue();
    }
    public void colorYellow() {
        myGraphic.colorYellow();
    }
    /**
     * Returns the position of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite off of this Subunit
     * @return Vector3d
     */
    public Vector3d getBSPos(BindingSite bs) {
        Vector3d bspos = new Vector3d();
        bspos = bs.getPos();
        boolean found = false;

        Vector myConfs = new Vector();
        for (int i = 0; i < myDomains.size(); i++)
            myConfs.add(((Domain) myDomains.get(i)).getCurConf());

        for (int i = 0; i < myConfs.size(); i++) {
            Conformation curconf = (Conformation) myConfs.elementAt(i);
            //if curconf has bs, get bs pos
            if (curconf.hasBS(bs)) {
                bspos = bs.getPos();
                found = true;
            }
        }
        return bspos;
    }

    /**
     * Returns the rotation of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite off of this Subunit
     * @return Quat4d
     */
    public Quat4d getBSRot(BindingSite bs) {
        Quat4d bsor = new Quat4d(0.0, 0.0, 0.0, 1.0);
        boolean found = false;
        Vector myConfs = new Vector();
        for (int i = 0; i < myDomains.size(); i++)
            myConfs.add(((Domain) myDomains.get(i)).getCurConf());

        for (int i = 0; i < myConfs.size(); i++) {
            Conformation curconf = (Conformation) myConfs.elementAt(i);
            if (curconf.hasBS(bs)) {
                bsor = bs.getRot();
                found = true;
            }
        }
        Debug.checkFalse(found, "getBSRot in Subunit");
        return bsor;
    }

    /**
     * Detaches this Subunit's graphic from its parent Assembly graphic
     */
    public void detachGraphic() {
        myGraphic.detach();
    }

    /**
     * Returns this Subunit's SubunitGraphic
     * 
     * @return SubunitGraphic
     */
    public SubunitGraphic getSubunitGraphic() {
        return myGraphic;
    }

    /**
     * Returns this Subunit's position in relation to its Assembly
     * 
     * @return Vector3d
     */
    public Vector3d getPos() {
        return myGraphic.getPos();
    }

    /**
     * Returns this Subunit's rotation in relation to its Assembly
     * 
     * @return Quat4d
     */
    public Quat4d getRot() {
        return this.myGraphic.getRot();
    }

    /**
     * Returns this Subunit's rotational velocity in relation to its Assembly
     * 
     * @return Quat4d
     */
    public Quat4d getRotV() {
        return myrotv;
    }

    /**
     * Returns this Subunit's velocity in relation to its Assembly
     * 
     * @return Vector3d
     */
    public Vector3d getV() {
        return myv;
    }

    /**
     * Returns this Subunit's radius
     * 
     * @return double
     */
    public double getRadius() {
        return mySubunitType.getRadius();
    }

    /**
     * Returns this Subunit's mass
     * 
     * @return double
     */
    public double getMass() {
        return mySubunitType.getMass();
    }

    /**
     * Returns this Subunit's Solution
     * 
     * @return Solution
     */
    public Solution getSoln() {
        return mySolution;
    }

    /**
     * Returns this Subunit's SubunitType
     * 
     * @return SubunitType
     */
    public SubunitType getST() {
        return mySubunitType;
    }

    /**
     * Returns a Vector of Subunits bound to this Subunit
     * 
     * @return Vector of Subunits
     */
    public Vector getBoundSubunits() {
        return myPartners;
    }

    /**
     * Sets this Subunit's transformation in relation to its Assembly to t
     * 
     * @param t
     *            Transform3D
     */
    public void setTrans(Transform3D t) {
        myGraphic.updateLocation(t);
    }

    /**
     * Returns true if this Subunit is bound to s, false otherwise
     * 
     * @param s
     *            Subunit
     * @return boolean
     */
    public boolean isBoundSubunit(Subunit s) {
        for (int i = 0; i < myPartners.size(); i++) {
            if (s.equals((Subunit) myPartners.elementAt(i)))
                return true;
        }
        return false;
    }

    /**
     * Returns a Vector of all the BindingSites that bind this Subunit to s. The
     * Vector is arranged as: [this Subunit's bs, s's bound bs, this Subunit's
     * bs2, s's bound bs2, ...]
     * 
     * @param s
     *            Subunit
     * @return Vector
     */
    public Vector getConnectingBSs(Subunit s) {
        Vector bss = new Vector();

        //search s's bindsites for sites bound to this
        for (int i = 0; i < myDomains.size(); i++) {
            Conformation curconf = ((Domain) myDomains.get(i)).getCurConf();
            Vector curbss = curconf.getBindSites();
            for (int j = 0; j < curbss.size(); j++) {
                BindingSite mybs = (BindingSite) curbss.get(j);
                if (mybs.isBound()) {
                    BindingSite mybspartner = mybs.getPartner();
                    //check to see if these bs's are on the right subunits
                    if (mybspartner.getSubunitID() == s.getid()) {
                        bss.add(mybs);
                        bss.add(mybspartner);
                    }
                }
            }
        }
        return bss;
    }

    /**
     * Returns true if this Subunit is not bound to any other Subunit, false
     * otherwise
     * 
     * @return boolean
     */
    public boolean isUnbound() {
        return (myPartners.size() == 0);
    }

    /**
     * Add Subunit s to this Subunit's list of bound partner Subunits
     * 
     * @param s
     *            Subunit to add
     */
    public void addPartner(Subunit s) {
        myPartners.add(s);
    }

    /**
     * Removes the specified Subunit from the vector of bound partner Subunits.
     * Returns true if the removal was sucessful, false otherwise
     * 
     * @param s
     *            Subunit to be removed
     * @return boolean
     */

    public boolean removePartner(Subunit s) {
        int id = s.getid();
        for (int i = 0; i < myPartners.size(); ++i) {
            if (id == ((Subunit) (myPartners.get(i))).getid()) {
                myPartners.remove(i);
                return true;
            }
        }
        return false;
    }

    /**
     * Returns this Subunit's ID
     * 
     * @return int
     */
    public int getid() {
        return myID;
    }

    /**
     * Sets this Subunit's ID to i
     * 
     * @param i
     *            int
     */
    public void setid(int i) {
        myID = i;
    }

    /**
     * Returns true if this Subunit is Subunit s
     * 
     * @param s
     *            Subunit
     * @return boolean
     */
    public boolean equals(Subunit s) {
        return (myID == s.getid());
    }

    /**
     * Returns a Vector of current Conformations (one for each Domain)
     * 
     * @return Vector
     */
    public Vector getConfs() {
        Vector myConfs = new Vector();
        for (int i = 0; i < myDomains.size(); i++)
            myConfs.add(((Domain) myDomains.get(i)).getCurConf());

        return myConfs;
    }

    public Vector getDomains() {
        return myDomains;
    }

    /**
     * Returns a Vector of the Subunit's current BindingSites and resets the
     * BindingSites' Domain ID's
     * 
     * @return Vector
     */
    public Vector getBindSites() {
        Vector bsites = new Vector();
     
        for (int i = 0; i < myDomains.size(); i++) {
            Vector cursites = new Vector();
            cursites = (((Domain) myDomains.get(i)).getCurConf())
                    .getBindSites();
            
            int cursites_size = cursites.size();
            
            for (int j = 0; j < cursites_size; j++) {
                BindingSite curbs = (BindingSite) cursites.elementAt(j);
                curbs.setDomainID(j);
                bsites.add(curbs);
            }
        }
        return bsites;
    }

    /**
     * Returns a Vector of the positions of this Subunit's current BindingSites
     * 
     * @return Vector
     */
    public Vector getBindSitePoses() {
        Vector bsposes = new Vector();
        for (int i = 0; i < myDomains.size(); i++) {
            Vector cursites = new Vector();
            Conformation curconf = ((Domain) myDomains.get(i)).getCurConf();
            cursites = (curconf).getBindSites();
            for (int j = 0; j < cursites.size(); j++) {
                BindingSite curbs = (BindingSite) cursites.elementAt(j);
                Vector3d curpos = curbs.getPos();
                bsposes.add(curpos);
            }
        }
        return bsposes;
    }

    /**
     * Returns a Vector of the rotations of this Subunit's current BindingSites
     * 
     * @return Vector
     */
    public Vector getBindSiteOrs() {
        Vector bsors = new Vector();
        for (int i = 0; i < myDomains.size(); i++) {
            Vector cursites = new Vector();
            Conformation curconf = ((Domain) myDomains.get(i)).getCurConf();
            cursites = (curconf).getBindSites();
            for (int j = 0; j < cursites.size(); j++) {
                BindingSite curbs = (BindingSite) cursites.elementAt(j);
                Quat4d curor = curbs.getRot();
                bsors.add(curor);
            }
        }
        return bsors;
    }

    /**
     * Returns the transform of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite
     * @return Transform3D
     */
    public Transform3D getBStoSubTransform(BindingSite bs) {
        return bs.getTransform();
    }

    /**
     * Returns the up vector of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     */
    public Vector3d getBSUptoSub(BindingSite bs) {
        return bs.getUp();
    }

    public Transform3D getTransform() {
        return myGraphic.getTransform();
    }

    /**
     * Returns the position of bs in relation to the parent Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     */
    public Vector3d getBindingSitePos(BindingSite bs) {
        return myGraphic.getBindingSitePos(bs);
    }

    /**
     * Returns the rotation of bs in relation to the parent Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Quat4d
     */
    public Quat4d getBindingSiteRot(BindingSite bs) {
        return myGraphic.getBindingSiteRot(bs);
    }

    /**
     * Returns a String version of this Subunit
     * 
     * @return String
     */
    public String toString() {
        String mystring = new String("Subunit ID: " + myID
                + "\nSubunitType Name: " + mySubunitType.getName() + "\nDomains:");
        for (int i = 0; i < myDomains.size(); i++) {
            mystring = mystring
                    + "\nDomain "
                    + i
                    + ":\n\tCurrent Conformation: "
                    + (((Domain) myDomains.elementAt(i)).getCurConf())
                            .toString() + "\n";
        }
        return mystring;
    }
    public String getSubunitTypeName()
    {
    	return mySubunitType.getName();
    }
    
    public void subRotate(AxisAngle4d rotation)
    {
    	Quat4d q1 = new Quat4d();
    	q1.set(rotation);
    	myRotation = myGraphic.getRot();
   // 	myrot.mul(q1);
    	myGraphic.rotate(q1);
    }
    
    public SubunitGraphic getGraphic()
    {
    	return myGraphic;
    
    }

}