/**
 * Event implments Comparable so that two events can be compared based on the
 * end time. The posting time of an Event can be checked against the valid time
 * of Assemblies involved to see if the event has been made invalid by other
 * events on the Assemblies involed.
 * 
 * @author Sue Yi Chew
 * @author Rori Rohlfs
 * @author Tiequan Zhang
 * @version 1.3
 *  
 */
public class Event implements Comparable {
    /**
     * The time this event was posted (when it was created) of this event. It
     * doesn't matter whether the event will actually occur or not
     */
    private double postTime;

    /**
     * the time that this event will be complete (assuming it is not invalidated
     * before then
     */
    private double endTime; // Determines priority. PQ is sorted using this

    // order.

    /**
     * Array of names of Assemblies involved. If this is a ConfChangeEvent or a
     * BreakBondEvent, only first element is used, second is null. If this a
     * FormBondEvent, the second element may or may not be used
     */
    private String[] assemNamesInvolved;

    /**
     * Constructs a default Event
     */
    public Event() {
        postTime = -1.0;
        endTime = -1.0;
    }

    /**
     * Constructs an Event made at curTime, with the given endTime and names of
     * Assemblies involved
     * 
     * @param curTime
     *            double, current time when event is found
     * @param endTime
     *            double, time when the event will be complete (used for
     *            priority)
     * @param assemNamesInvolved
     *            String[]
     */
    public Event(double curTime, double endTime, String[] assemNamesInvolved) {
        this.postTime = curTime;
        this.endTime = endTime;
        this.assemNamesInvolved = assemNamesInvolved;
    }

    /**
     * Gets the post time of this Event
     * 
     * @return double
     *  
     */
    public double getPostTime() {
        return postTime;
    }

    /**
     * Sets the post time of this Event to a new value
     * 
     * @param newPostTime
     *            double, newPostTime is the new value for post time of this
     *            event
     *  
     */
    public void setPostTime(double newPostTime) {
        postTime = newPostTime;
    }

    /**
     * Sets the end time of this Event to a new value
     * 
     * @param newEndTime
     *            double, newEndTime is the new vlaue for end time of this event
     *  
     */
    public void setEndTime(double newEndTime) {
        endTime = newEndTime;
    }

    /**
     * Gets the end time of this Event
     * 
     * @return double
     *  
     */
    public double getEndTime() {
        return endTime;
    }

    /**
     * Gets the names of all Assemblies involved in this Event
     * 
     * @return String[]
     *  
     */
    public String[] getAssemNamesInvolved() {
        return assemNamesInvolved;
    }

    /**
     * Sets the names of all Assemblies involved in this Event
     * 
     * @param assemNamesInvolved
     *            void
     *  
     */
    public void setAssemNamesInvolved(String[] assemNamesInvolved) {
        this.assemNamesInvolved = assemNamesInvolved;
    }

    /**
     * Compares Object e and this Event based on their end time.
     * 
     * @param e
     *            Object
     * 
     * @return 1 if this Event's end time is earlier than Object e, 0 if they
     *         are the same, -1 if this Event's end time is greater than Object
     *         e
     */
    public int compareTo(Object e) {
        if ((this == null) && (e == null)) {
            return 0;
        } else if (this == null) {
            return -1;
        } else if (e == null) {
            return 1;
        } else {

            return -(Double.compare(endTime, ((Event) e).getEndTime()));
        }
    }

    /**
     * public boolean equals(FormBondEvent e) { if () return true; }
     */

    /**
     * Checks the validity of this Event. Event is valid if it was post after
     * the most recent events for the Assemblies involved. That is, if the
     * posting time of this event is greater than or equal to validTimes of each
     * Assembly involved. validTimes[1] may be null if only one assembly was
     * involved in this event.
     * 
     * @param validTimes
     *            Array of validTimes of all assemblies this event involves
     * @return boolean
     */
    public boolean isValid(double[] validTimes) {
        for (int i = 0; i < validTimes.length; ++i) {
            if (postTime < validTimes[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * Returns true if there is only 1 assembly involved in this Event
     * 
     * @return boolean
     *  
     */

    public boolean sameAssembly() {
        return (assemNamesInvolved[1] == null);
    }

}