
import java.util.Vector;
import javax.vecmath.*;

/**
 * Conformation holds information about a particular conformation Domain of a
 * Subunit (energy, SubunitType, Domain, BindingSites present in this
 * Conformation)
 * 
 * @author Rori Rohlfs
 * @version 1.3
 *  
 */
public class Conformation {
    //BindingSites present in this Conformation of a domain in a Subunit
    private Vector mySites;

    //conformation name
    private String myName;

    //energy in this conformation (not used presently)
    private double myEnergy;

    //SubunitType's name, which should be assigned before put in simulator
    private String mySubunitTypeName;

    //ID of Domain in which this Conformation is
    private int myDomainID;

    /**
     * Constructs a default conformation
     */
    public Conformation() {
        mySites = new Vector();
        myName = new String();
        myEnergy = 0.0;
        mySubunitTypeName = new String();
        myDomainID = -1;
    }

    /**
     * Constructs a Conformation with specified BindingSite Vector bs, energy e,
     * and name. Sets the ID of SubunitTypeID and DomainID associated to -1.
     * 
     * @param bs
     *            Vector of BindingSite
     * @param e
     *            double energy
     * @param n
     *            String name of this Conformation
     */
    public Conformation(Vector bs, double e, String n) {
        myName = n;
        mySites = bs;
        myEnergy = e;
        mySubunitTypeName = new String();
        myDomainID = -1;
    }

    /**
     * Returns the bindingsites present in this conformation
     * 
     * @return Vector
     */
    public Vector getBindSites() {
        return mySites;
    }

    /**
     * Checks if any BindingSite in this Conformation is bound, if so, returns
     * true
     * 
     * @return boolean
     */
    public boolean bound() {
        for (int i = 0; i < mySites.size(); i++) {
            if (((BindingSite) mySites.get(i)).isBound())
                return true;
        }
        return false;
    }

    /**
     * Returns true if this Conformation contains the BindingSite bs
     * 
     * @param bs
     *            BindingSite to be checked
     * @return boolean
     */
    public boolean hasBS(BindingSite bs) {
        for (int i = 0; i < mySites.size(); i++) {
            BindingSite cursite = (BindingSite) mySites.elementAt(i);
            if (cursite.equals(bs))
                return true;
        }
        return false;
    }

    /**
     * Sets the name of SubunitType in this Conformation to stname, sets the ID
     * of Domain of this Conformation to did
     * 
     * @param stname
     *            String stname
     * @param did
     *            int Domain ID
     */
    public void setSubunitType(String stname, int did) {
        mySubunitTypeName = stname;
        myDomainID = did;
    }

    /**
     * Returns the name of this Conformation
     * 
     * @return String
     */
    public String getName() {
        return myName;
    }

    /**
     * Returns the ID of this Conformation's Domain
     * 
     * @return int
     */
    public int getDomainID() {
        return myDomainID;
    }

    /**
     * Returns of energy of this Conformation
     * 
     * @return double
     */
    public double getE() {
        return myEnergy;
    }

    /**
     * Returns Vector of BindingSites present in this Conformation
     * 
     * @return Vector
     */
    public Vector getSites() {
        return mySites;
    }

    /**
     * Returns true if c is equal to this
     * 
     * @return boolean
     */
    public boolean isEqual(Conformation c) {
        return myName.equals(c.getName());
    }

    /**
     * Returns a String version of this Conformation
     * 
     * @return String
     */
    public String toString() {
        String mystring = new String("Conformation name: " + myName
                + "\nConformation Energy: " + myEnergy
                + "\nProtein Conformation:");

        for (int i = 0; i < mySites.size(); i++) {
            BindingSite curbs = (BindingSite) mySites.elementAt(i);
            mystring += "\n\tBindingSite ID: " + curbs.getID();

            Vector3d pos = curbs.getPos();
            mystring += "\n\t\tPosition:\t" + pos.toString();

            Quat4d rot = curbs.getRot();
            mystring += "\n\t\tRotation:\t" + rot.toString();

            mystring = mystring + "\n\t\tBinding State(true if bound): "
                    + curbs.isBound();
            if (curbs.isBound()) {
                mystring = mystring + "\n\t\tPartner BindingSite Subunit ID: "
                        + (curbs.getPartner()).getSubunitID();
            }
        }
        return mystring;
    }
}