
import java.util.Vector;
import java.lang.Math;
import javax.vecmath.*;

/**
 * a SubunitType is a class of Subunits with common Conformations,
 * Domains and BindingSites
 * 
 * @author Rori Rohlfs
 * @version 1.0
 */

public class SubunitType {
    /**
     * Vecotor of Domains which include a vector of Conformation
     */
    private Vector confSets;
    
    private String myName;
    
    private int myID;

    private Solution mySolution;

    private double mymass;

    private double myradius;

    /**
     * Constructs SubunitType with default parameters
     */
    public SubunitType() {
        confSets = new Vector();
	myName = new String();
        mySolution = new Solution();
	mymass = 1.0;
	myradius = 0.08;
    }

    /**
     * Constructs SubunitType with specified parameters,
     * 
     * @param cs
     *            Vector which set Conformation and Domain information of this
     *            SubunitType
     * @param n
     *            String the name of this SubunitType
     * @param s
     *            Soltion
     * @param m
     *            double mass of this SubunitType
     * @param r
     *            double radius of this SubunitType
     */
    public SubunitType(Vector cs, String n, Solution s, double m, double r) {
        confSets = cs;
	myName = n;
        mySolution = s;
	mymass = m;
	myradius = r;
    }

    /**
     * 
     * Returns delta E which is the difference of all energies of confsA and
     * confsB where confsA and confsB are Vector of protein Conformation
     * 
     * @param confsA
     *            Vector of Conformation
     * @param confsB
     *            Vector of Conformation
     * @return double the difference of energy
     *  
     */
    private double deltaE(Vector confsA, Vector confsB) {
        double aesum = 0.0;
        double besum = 0.0;
        for (int i = 0; i < confsA.size(); i++) {
            aesum += ((Conformation) confsA.elementAt(i)).getE();
            besum += ((Conformation) confsB.elementAt(i)).getE();
        }

        return (besum - aesum);
    }

    /**
     * Returns Vector of Vectors of all Conformations for this SubunitType
     *
     * @return Vector
     */
    public Vector getConfSets() {
        Vector sets = new Vector();

        //Gets the maximum number of Conformations in all Domains
        int maxconfs = 0;
        for (int i = 0; i < confSets.size(); i++) {
            int temp = ((Vector) ((Domain) confSets.elementAt(i)).getConfs())
                    .size();
            if (temp > maxconfs) {
                maxconfs = temp;
            }
        }
      
        for (int i = 0; i < Math.pow(maxconfs, confSets.size()); i++) {
            Vector curconf = new Vector();
            boolean real = true;
            for (int j = 0; j < confSets.size(); j++) {
                Vector curdomconfs = (Vector) ((Domain) confSets.elementAt(j))
                        .getConfs();
                if (curdomconfs.size() > (((int) i / Math.pow(maxconfs, j)) % maxconfs)) {
                    Conformation cur = (Conformation) curdomconfs
                            .elementAt(((int) (i / Math.pow(maxconfs, j)))
                                    % maxconfs);
                    curconf.add(cur);
                } else {
                    real = false;
                }
            }
            if (real) {
                sets.add(curconf);
            }
        }

        return sets;
    }

    /**
     * Returns vector of possible conformations of Domain with id d
     * 
     * @param d
     *            int Domain id
     * @return Vector of Conformation
     */
    public Vector getDomainConfs(int d) {
        return ((Domain) confSets.elementAt(d)).getConfs();
    }

    /**
     * Returns this SubunitType's mass
     * 
     * @return double
     */
    public double getMass() {
	return mymass;
    }

    /**
     * Returns this SubunitType's radius
     * 
     * @return double
     */
    public double getRadius() {
        return myradius;
    }

    /**
     * Returns a String version of this SubunitType
     * 
     * @return String
     */
    public String toString() {
        String mystring = new String("SubunitType ID: " + myID);

        for (int i = 0; i < confSets.size(); i++) {
            Domain cur = ((Domain) confSets.elementAt(i));

            mystring = mystring + "\nDomain " + i + ":\n\tConformation Names:\t";

            Vector tconfs = cur.getConfs();
            for (int j = 0; j < tconfs.size(); j++) {
                mystring = mystring
                        + ((Conformation) tconfs.elementAt(j)).getName() + "\t";
            }

            Vector3d pos = cur.getPos();
            mystring += "\n\tPosition: " + pos.toString();
        }

        return mystring;
    }

    /**
     * Returns true if Conformation c is a possible conformation of Domain di,
     * false if not
     * 
     * @return boolean
     */
    public boolean isConf(Conformation c, int di) {
        Vector confs = ((Domain) confSets.elementAt(di)).getConfs();
        for (int i = 0; i < confs.size(); i++) {
            Conformation cur = (Conformation) confs.elementAt(i);
            if ((cur.getName()).equals(c.getName())) {
                return true;
            }
        }
        return false;
    }

    /**
     * Gets the name of this SubunitType
     * 
     * @return String
     *  
     */
    public String getName() {
        return myName;
    }

    /**
     * Gets the ID of this subunitType
     * 
     * @return int
     *  
     */
    /*
    public int getid() {
        return myID;
	}*/

    /**
     * Returns if s is the same as this SubunitType
     * 
     * @param s
     *            SubunitType
     * @return boolean
     */
    public boolean isEqual(SubunitType s) {
        return myName.equals(s.getName());

    }

}