
import javax.swing.*;
import java.awt.BorderLayout;
import java.awt.GraphicsConfiguration;
import com.sun.j3d.utils.universe.*;
import javax.media.j3d.*;
import javax.vecmath.*;

import java.util.Vector;
import com.sun.j3d.utils.behaviors.mouse.*;

/**
 * extends JPanel, displays the current simulation in a panel, allows user to
 * observe the simulation in 3-D space, allows user to use mouse to control the
 * view of simulation
 * 
 * @author Rori Rohlfs
 * @version 1.0
 * 
 *  
 */
public class SimulatorGraphic extends JPanel {

    private Canvas3D canvas3D = null;

    private Simulation sim = null;

    private SimpleUniverse simpleU = null;

    private BranchGroup scene = new BranchGroup();

    private double spacing = 1.5;

    private int squareSize = 1;

    /**
     * Constructs SimulatorGraphic based on sim
     * 
     * @param sim
     *            Simulation
     */
    public SimulatorGraphic(Simulation sim) {

        setLayout(new BorderLayout());
        setOpaque(true);
        /**
         * Finds the preferred GraphicsConfiguration for the system. This
         * object can then be used to create the Canvas3D objet for this system.
         */
        GraphicsConfiguration config = SimpleUniverse
                .getPreferredConfiguration();
        this.sim = sim;

        /**
         * Uses a valid GraphicConfiguration object(config) to create a new
         * Canvas3D object that Java 3D can render into
         */
        canvas3D = new Canvas3D(config);

        /**
         * Adds canvas3D to this container
         */
        add("Center", canvas3D);

        /**
         * Creates a locale, a single ViewingPlatform and a Viewer object
         * associates canvas3D to the Viewer object
         */
        simpleU = new SimpleUniverse(canvas3D);

        /**
         * Creates a Viewing object and sets the normal viewing distance in the
         * ViewPlatform transform based on the current field of view
         */
        simpleU.getViewingPlatform().setNominalViewingTransform();

        //adds every AssemblyGraphic to the scene graph
        makeSG(sim);
    }

    /**
     * Adds every AssemblyGraphic to the scene
     * 
     * @param sim
     *            Simulation is the parameter provided to the constructor
     */
    public void makeSG(Simulation sim) {
        BranchGroup newscene = new BranchGroup();
        newscene.setCapability(BranchGroup.ALLOW_DETACH);
        newscene.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
        newscene.setCapability(Group.ALLOW_CHILDREN_WRITE);

        /**
         * asmbs contains an array of Assembly objects
         */
        Vector asmbs = sim.getAssemblies();

	//grid spacing for initial monomers 
        double spacing = 1.5;
        int numberAssemblies = asmbs.size();
        squareSize = 1;
        while (numberAssemblies > Math.pow(squareSize, 2)) {
            squareSize++;
        }
        squareSize--;
        double x = -spacing * (squareSize / 2.0);
        double y = spacing * (squareSize / 2.0);
	//go through each assembly, adding their graphics
        for (int i = 0; i < asmbs.size(); i++) {
            Assembly curasm = (Assembly) asmbs.elementAt(i);
            AssemblyGraphic bgcurasm = curasm.getAssemblyGraphic();

            Quat4d rot = new Quat4d(sim.getRand(), sim.getRand(), sim.getRand(), sim.getRand());
            bgcurasm.updateLocation(rot, new Vector3d(x, y, 0.0));

            bgcurasm.detach();
            newscene.addChild(bgcurasm);

            //move x and y for next assembly
            if (x < spacing * (squareSize / 2.0)) {
                x += spacing;
            } else {
                x = -spacing * (squareSize / 2.0);
                y -= spacing;
            }
        }

        /**
         * add in mouse interaction.
         */
        /**
         * Constructs and initializes a BoundingSphere from a center and radius
         * 1000.0 TODO what is 1000.0
         */
        BoundingSphere mouseBounds = new BoundingSphere(new Point3d(), 1000.0);

        /*
         * Creates a reference to the "bottom most" transform in the
         * MultiTransformGroup that is above the ViewPlatform node
         */
        TransformGroup vpTrans = simpleU.getViewingPlatform()
                .getViewPlatformTransform();
        /**
         * creates rotate behavior, sets a new transformGroup replacing the old
         * one, sets the Behavior's scheduling region to the mouseBounds. adds
         * the child node (myMouseRotate) to newscene list of children
         */
        MouseRotate myMouseRotate = new MouseRotate(MouseBehavior.INVERT_INPUT);
        myMouseRotate.setTransformGroup(vpTrans);
        myMouseRotate.setSchedulingBounds(mouseBounds);
        newscene.addChild(myMouseRotate);

        /**
         * creates translate behavior, sets a new transformGroup replacing the
         * old one, sets the Behavior's scheduling region to the mouseBounds.
         * adds the child node (myMouseTranslate) to newscene list of children
         */
        MouseTranslate myMouseTranslate = new MouseTranslate(
                MouseBehavior.INVERT_INPUT);
        myMouseTranslate.setTransformGroup(vpTrans);
        myMouseTranslate.setSchedulingBounds(mouseBounds);
        newscene.addChild(myMouseTranslate);

        /**
         * creates zoom behavior, sets a new transformGroup replacing the old
         * one, sets the Behavior's scheduling region to the mouseBounds. adds
         * the child node (myMouseZoom) to newscene list of children
         */
        MouseZoom myMouseZoom = new MouseZoom(MouseBehavior.INVERT_INPUT);
        myMouseZoom.setTransformGroup(vpTrans);
        myMouseZoom.setSchedulingBounds(mouseBounds);
        newscene.addChild(myMouseZoom);

        //adds light ambient light source object
        AmbientLight lightA = new AmbientLight();
        lightA.setInfluencingBounds(new BoundingSphere(new Point3d(),
                Float.MAX_VALUE));
        lightA.setCapability(Light.ALLOW_STATE_WRITE);
        newscene.addChild(lightA);
        DirectionalLight lightdi = new DirectionalLight(new Color3f(1.f, 1.f,1.f), new Vector3f(-1.f, -0.5f, -1.f));
        lightdi.setInfluencingBounds(new BoundingSphere());
        lightdi.setCapability(Light.ALLOW_STATE_WRITE);
        newscene.addChild(lightdi);

        /**
         * Compiles the source BranchGroup associated with newscene and creates
         * and caches a compiled scene graph, detaches scene from its source
         */
        newscene.compile();
        if (scene != null) {
            scene.detach();
        }
        scene = newscene;
        simpleU.addBranchGraph(scene);
    }

    /**
     * adds a new Assembly's graphic to the scene
     * 
     * @param asm
     *            Assembly
     */
    public void addAssemblytoGraphic(Assembly asm) {
        int squareLength = ((int) spacing) * squareSize;
        Vector3d t = new Vector3d((sim.getRand() * 2.0 * squareLength)
                - squareLength, (sim.getRand() * 2.0 * squareLength)
                - squareLength, (sim.getRand() * 2.0 * squareLength)
                - squareLength);

        asm.setPos(t);
        AssemblyGraphic gasm = asm.getAssemblyGraphic();
        gasm.detach();
        scene.addChild(gasm);
    }
}

