
import java.util.Vector;
import java.util.HashMap;
import javax.vecmath.*;

/**
 * NeighborGroups stores assembly subunits in neighborgroups for
 * fast collision detection.That is, it divides the subunits 
 * in an Assembly into groups of physically close neighbors for 
 * that methods which need to compare physically close subunits.
 * 
 * @author Rori Rohlfs
 * @version 1.0
 */
public class NeighborGroups {
    // neighborgroups
    private HashMap groups; //keys = x-coords, 
    //                        values = hashmap
    //                                 |
    //                                 V
    //                               keys = y-coords, 
    //                               values = hashmaps
    //                                        |
    //                                        V
    //                                      keys = z-coords, 
    //                                      values = subunit ids in that region
    private double subdivision = 1.0;

    /**
     * Creates default NeighborGroups
     */
    public NeighborGroups() {
        groups = new HashMap();
    }

    /**
     * Creates NeighborGroups with specified Assembly asm.
     * actually sorts the Subunits in asm into neighborgroups 
     * according to a subdivision-spaced cubish grid
     * 
     * @param asm
     */
    public NeighborGroups(Assembly asm) {
        groups = new HashMap();

        Vector subs = asm.getSubunits();
        for (int i = 0; i < subs.size(); i++) {
            Subunit cursub = (Subunit) subs.get(i);
	    //System.out.println("NeighborGroups: cursub id " + cursub.getid());

            Vector3d cursubpos = cursub.getPos();
            double[] coords = new double[3];
            cursubpos.get(coords);

            Double x = round(coords[0]);
            Double y = round(coords[1]);
            Double z = round(coords[2]);

            if (groups.containsKey(x)) {
                HashMap yhash = (HashMap) groups.get(x);
                if (yhash.containsKey(y)) {
                    HashMap zhash = (HashMap) yhash.get(y);
                    if (zhash.containsKey(z)) {
                        Vector entry = (Vector) zhash.get(z);
			//System.out.println("NeighborGroups: put1 cursubid " + cursub.getid() + "at"+x+","+y+","+z);
                        entry.add(cursub);
                    } else {//zhash doesn't have this z-coord yet
                        Vector entry = new Vector();
                        entry.add(cursub);
                        zhash.put(z, entry);
			//System.out.println("NeighborGroups: put2 cursubid " + cursub.getid() + "at"+x+","+y+","+z);
                    }
                } else { //yhash doesn't have have this y-value yet
                    HashMap newzhash = new HashMap();
                    Vector entry = new Vector();
                    entry.add(cursub);
                    newzhash.put(z, entry);

                    yhash.put(y, newzhash);
		    //System.out.println("NeighborGroups: put3 cursubid " + cursub.getid() + "at"+x+","+y+","+z);
                }
            } else { //groups (xhash) doesn't have this x-value yet
                HashMap newzhash = new HashMap();
                Vector entry = new Vector();
                entry.add(cursub);
                newzhash.put(z, (Vector) entry);

                HashMap newyhash = new HashMap();
                newyhash.put(y, newzhash);

                groups.put(x, newyhash);
		//System.out.println("NeighborGroups: put4 cursubid " + cursub.getid() + "at"+x+","+y+","+z);
             
            }
        }
    }

    //private functions -----------------------------------------------

    /**
     * Copys every Subunit in newsubs to subunit Vector subs
     * @param subs
     * @param newsubs 
     */
    private void addEntries(Vector subs, Vector newsubs) {
        for (int i = 0; i < newsubs.size(); i++) {
            subs.add(newsubs.get(i));
        }
    }


    /**
     * Returns a Double of d rounded up to the next 0.5
     * @param d double
     * @return Double 
     */
    private Double round(double d) {
        int i = (int) d;
        double fraction = d - ((double) i);

        if (d > 0.0) {
            if (fraction < subdivision) {
                return new Double((double) i);
            }
            return new Double(((double) i) + subdivision);
        }

        if (fraction < -subdivision) {
            return new Double((double) i);
        }
        return new Double(((double) i) + subdivision);
    }

    //public functions------------------------------------------
    /**
     * Returns a Vector of all the Subunits within subs' neighborgroup cube
     * and the Subunits in all adjacent cubes
     * @param sub Subunit
     * @return Vector
     */
    public Vector getNeighborGroups(Subunit sub) {
        Vector subs = new Vector(); //keeps neighborsubs

        Vector3d subpos = sub.getPos();
        double[] coords = new double[3];
        //coords contains the position vector (x,y,z) of sub 
        subpos.get(coords);
     
        Double x = round(coords[0]);
        Double y = round(coords[1]);
        Double z = round(coords[2]);

        for (double i = -subdivision; i < subdivision * 2.0; i += subdivision) {
            for (double j = -subdivision; j < subdivision * 2.0; j += subdivision) {
                for (double k = -subdivision; k < subdivision * 2.0; k += subdivision) {
		    Double tx = new Double(x.doubleValue() + i);
		    Double ty = new Double(y.doubleValue() + j);
		    Double tz = new Double(z.doubleValue() + k);
		    
		    if (groups.containsKey(tx)) {
			HashMap yhash = (HashMap) groups.get(tx);
			if (yhash.containsKey(ty)) {
			    HashMap zhash = (HashMap) yhash.get(ty);
			    if (zhash.containsKey(tz)) {
				Vector entry = (Vector) zhash.get(tz);
				addEntries(subs, entry);
			    }
			} 
		    }
                }
            }
        }
        return subs;
    }

    /*
    public String toString() {
	String mystring = new String("");
	       

	Iterator xit = groups.keySet().iterator();
        while (xit.hasNext()) {
	    Double xkey = (Double) xit.next();
	    HashMap yhash = (HashMap) groups.get(xkey);
	    Iterator yit = yhash.keySet().iterator();
	    while(yit.hasNext()) {
		Double ykey = (Double) yit.next();
		HashMap zhash = (HashMap) yhash.get(ykey);
		Iterator zit = yhash.keySet().iterator();
		while(zit.hasNext()) {
		    Double zkey = (Double) zit.next();
		    Vector group = (Vector) zhash.get(zkey);
		  
		    if(group != null) {
			mystring += "<" + xkey + "," + ykey + "," + zkey + ">\t{";

			for(int i=0; i<group.size(); i++) {
			    mystring += ((Subunit) group.get(i)).getid();
			    if(i+1 < group.size())
				mystring += ",";
			    else
				mystring += "}\n";
			}
		    }
		      
		}
	    }
        }

	return mystring;
    }
    */
}
