
import java.util.Vector;
import java.util.HashMap;
import javax.vecmath.*;

/**
 * Tetrahedron sets up a tetrahedron-like Simulation, where there is one type of
 * Subunit with a single Conformation and Domain and three binding sites. The
 * initial number of Subunits, and the binding and breaking times control the
 * assembly of tetrahedron
 * 
 * @author Tiequan Zhang
 * @version 1.2
 */

public class Tetrahedron {
    private static Simulation sim;

    private static Solution soln;

    private static Vector assemblies;

    private static int bsid = 0; //BindingSite ID

    private static int domainID = 0; //Domain ID

    private static int stid = 0; //SubunitType ID

    private static int suid = 0; //Subunit ID

    private static double radius = Test.subunitRadius; //radius of subunit

    //binding and breaking rates set here:
    //average time for a binding event between two compatible BindingSites
    private double bindTime = Test.bindTime;

    //average time for a breaking event between two compatible BindingSites
    private double breakTime = Test.breakTime;

    //average time for a bonding event between two compatible BindingSites,
    //positioned to bind, in the same Assembly
    private double fastbindTime = Test.fastbindtime;

    //  average time for enabling binding ability
    private double bsToNoneTime = Test.bsToNoneTime;

    //  average time for disabling binding ability
    private double noneToBsTime = Test.noneToBsTime;

    //Initial number of Subunits are set here.
    private static int numSubunits = Test.numSubunits;

    private static HashMap bindingPartner = new HashMap();

    public Tetrahedron() {
        soln = new Solution();
        initAssembly();

        double[] bindbreak = new double[3];
        bindbreak[0] = bindTime;
        bindbreak[1] = breakTime;
        bindbreak[2] = fastbindTime;
        double[] bindbreak2 = new double[3];
        bindbreak2[0] = bindTime;
        bindbreak2[1] = breakTime;
        bindbreak2[2] = fastbindTime;
        double[] bindbreak3 = new double[3];
        bindbreak3[0] = bindTime;
        bindbreak3[1] = breakTime;
        bindbreak3[2] = fastbindTime;
        double[] nobindbreak = new double[3];
        nobindbreak[0] = 99999999999999.0;
        nobindbreak[1] = 0.00000000000001;
        nobindbreak[2] = 99999999999999.0;

        HashMap bst1hash = new HashMap();
        bst1hash.put(new String("bsta"), bindbreak);
        bst1hash.put(new String("bstb"), nobindbreak);
        bst1hash.put(new String("bstc"), nobindbreak);

        HashMap bst2hash = new HashMap();
        bst2hash.put(new String("bstb"), (double[]) bindbreak);
        bst2hash.put(new String("bsta"), (double[]) nobindbreak.clone());
        bst2hash.put(new String("bstc"), (double[]) nobindbreak.clone());

        HashMap bst3hash = new HashMap();
        bst3hash.put(new String("bsta"), (double[]) nobindbreak.clone());
        bst3hash.put(new String("bstb"), (double[]) nobindbreak.clone());
        bst3hash.put(new String("bstc"), (double[]) bindbreak3);

        HashMap bondtimes = new HashMap();
        bondtimes.put(new String("bsta"), bst1hash);
        bondtimes.put(new String("bstb"), bst2hash);
        bondtimes.put(new String("bstc"), bst3hash);

        HashMap bsMap = new HashMap(), noneMap = new HashMap();
        bsMap.put("none", new Double(bsToNoneTime));
        noneMap.put("bs", new Double(noneToBsTime));

        HashMap conftimes = new HashMap();
        conftimes.put("bs", bsMap);
        conftimes.put("none", noneMap);

        sim = new Simulation(soln, assemblies, 0.0, bondtimes, conftimes,
                bindingPartner);
    }

    /**
     * Returns Simulation of Tetrahedron
     * 
     * @return Simulation
     *  
     */
    public static Simulation getSim() {
        return sim;
    }

    /**
     * Creates all Subunits with appropriate bsID, domainID, ConformationID and
     * stID. Adds all Assemblies to Vector assemblies (which is later used in
     * the Simulation construction)
     */
    private static void initAssembly() {
        assemblies = new Vector();

        //bindingsitetype definitions
        double[] tolerances = { 0.2, 0.1, 0.1 };
        Vector partnersa = new Vector();
        Vector partnersb = new Vector();
        Vector partnersc = new Vector();
        partnersa.add("bsta");
        BindingSiteType bsta = new BindingSiteType(tolerances, "bsta",
                partnersa);
        partnersb.add("bstb");
        BindingSiteType bstb = new BindingSiteType(tolerances, "bstb",
                partnersb);
        partnersc.add("bstc");
        BindingSiteType bstc = new BindingSiteType(tolerances, "bstc",
                partnersc);

        bindingPartner.put("bsta", partnersa);
        bindingPartner.put("bstb", partnersb);
        bindingPartner.put("bstc", partnersc);

        Vector3d upbs1 = new Vector3d(1, 0, 0);
        AxisAngle4d orbs1a = new AxisAngle4d(upbs1, Math.acos(Math
                .sqrt(1.0 / 3)));
        Quat4d orbs1 = new Quat4d();
        orbs1.set(orbs1a);
        Vector3d posbs1 = new Vector3d(0, radius * Math.sqrt(1.0 / 3), radius
                * Math.sqrt(2.0 / 3));

        Vector3d upbs2 = new Vector3d(-1 / 6.0, Math.sqrt(3) / 6, 0);
        AxisAngle4d ax2 = new AxisAngle4d(0, 0, 1, 2.0 / 3 * Math.PI);
        Quat4d q2 = new Quat4d();
        q2.set(ax2);

        AxisAngle4d orbs2a = new AxisAngle4d(upbs2, Math.acos(Math
                .sqrt(1.0 / 3)));
        Quat4d orbs2 = new Quat4d();
        orbs2.set(orbs2a);
        orbs2.mul(q2);
        Vector3d posbs2 = new Vector3d(-radius / 2, -radius * Math.sqrt(3)
                / 6.0, radius * Math.sqrt(2.0 / 3));

        Vector3d upbs3 = new Vector3d(-1 / 6.0, -Math.sqrt(3) / 6, 0);
        AxisAngle4d ax3 = new AxisAngle4d(0, 0, 1, -2.0 / 3 * Math.PI);
        Quat4d q3 = new Quat4d();
        q3.set(ax3);

        AxisAngle4d orbs3a = new AxisAngle4d(upbs3, Math.acos(Math
                .sqrt(1.0 / 3)));
        Quat4d orbs3 = new Quat4d();
        orbs3.set(orbs3a);
        orbs3.mul(q3);

        Vector3d posbs3 = new Vector3d(radius / 2,
                -radius * Math.sqrt(3) / 6.0, radius * Math.sqrt(2.0 / 3));

        //loop for making subunittypes
        for (int i = 0; i < numSubunits; i++) {
            String asmname = "assem" + i;
            BindingSite bsa = new BindingSite(bsta, bsid++, domainID, suid,
                    asmname, posbs1, orbs1, upbs1);
            BindingSite bsb = new BindingSite(bstb, bsid++, domainID, suid,
                    asmname, posbs2, orbs2, upbs2);
            BindingSite bsc = new BindingSite(bstc, bsid++, domainID, suid,
                    asmname, posbs3, orbs3, upbs3);
            bsa.colorYellow();
            bsb.colorGreen();
            bsc.colorBlue();
            /**
             * Adds BindingSites of one Subunit to Vector bss to create
             * Conformation
             */
            Vector bss = new Vector();
            bss.add(bsa);
            bss.add(bsb);
            bss.add(bsc);

            /**
             * Adds one Conformation of a Subunit to Vector confs to create
             * Domain
             */
            Vector confs = new Vector();
            Conformation bsConf = new Conformation(bss, 6.6, "bs");
            Conformation nonBsConf = new Conformation(new Vector(), 6.6, "none");
            confs.add(bsConf);
            confs.add(nonBsConf);
            /**
             * Adds one Domain of a Subunit to Vector Confsets to create
             * SubunitType
             */
            Domain dom = new Domain("only", confs, new Vector3d(), domainID,
                    bsConf);
            if (Test.csAllowed) {
                dom = new Domain("only", confs, new Vector3d(), domainID,
                        nonBsConf);
            }
            Vector doms = new Vector();
            doms.add(dom);

            SubunitType subt = new SubunitType(doms, "only", soln, 3.4, 0.08);

            /**
             * set initial position, rotation, velocity, and rotational velocity
             */
            Vector3d sPos = new Vector3d();
            Quat4d sRot = new Quat4d(0.0, 0.0, 0.0, 1.0);
            Vector3d sVel = new Vector3d();
            Quat4d srVel = new Quat4d(0.0, 0.0, 0.0, 1.0);

            Subunit subunit = new Subunit(subt, doms, soln, suid++, sPos, sRot,
                    sVel, srVel);

            Vector asmsubs = new Vector();
            asmsubs.add(subunit);
            Assembly asm = new Assembly(asmsubs, asmname);

            assemblies.add(asm);
        }

    }

}