
import java.util.Vector;
//import java.util.Random;
import javax.vecmath.*;
import javax.media.j3d.*;

/**
 * Assembly holds an Assembly of Subunits and the AssemblyGraphic. It also
 * tracks the next Event for this Assembly and its validity. Assemblies can be
 * bonded, split, or relaxed.
 * 
 * @author Rori Rohlfs
 * @author Sue Yi Chew
 * @author Tiequan Zhang
 * @version 1.2
 *  
 */

public class Assembly {
    private String myname;

    private Vector mysubs; //Vector of Subunits

    private double k = 5.0; //arbitrary spring constant, replace this later

    private Event nextEvent; //minimum-time event seen so far

    //  the recent time that new events was sampled for this assembly

    private double validTime;

    AssemblyGraphic mygraphic; //holds a graphics tree to all the different

    /**
     * Constructs an default Assembly with no Subunits and no valid Events
     */
    public Assembly() {
        mysubs = new Vector();
        validTime = -1.0;
    }

    /**
     * Constructs an Assembly named name with Subunits from s
     * 
     * @param s
     *            Vector, all the Subunits to be in this Assembly
     * @param name
     *            String, the name of this Assembly
     */
    public Assembly(Vector s, String name) {
        /**
         * mysubs =s; should have the same effect as the following 4 lines.
         */
        mysubs = new Vector();
        for (int i = 0; i < s.size(); i++) {
            mysubs.add((Subunit) s.elementAt(i));
        }

        myname = name;
        validTime = -1.0;

        mygraphic = new AssemblyGraphic(this);
    }

    /**
     * translates and rotate the subunits in this assembly so that sub will be
     * the center of this Assembly and the relative postions and angles of other
     * Subunit(s) with Subunit will be maintained.
     * 
     * @param sub
     *            Subunit
     */
    private void centerAssembly(Subunit sub) {
        //get translation and rotation (relative to this Assembly) info of
        // Subunit sub
        Transform3D asmtosub = sub.getSubunitGraphic().getTransform();
        Quat4d asmtosubq = new Quat4d();
        Vector3d asmtosubv = new Vector3d();
        asmtosub.get(asmtosubv);
        asmtosub.get(asmtosubq);
        asmtosubq = Debug.checkQ4d(asmtosubq, asmtosub);
        asmtosubq.conjugate();
        asmtosubv.scale(-1.0);
        //translate and rotate every subunit
        for (int i = 0; i < mysubs.size(); i++) {
            Subunit cursub = (Subunit) mysubs.get(i);
            (cursub.getSubunitGraphic()).translate(asmtosubv);
            (cursub.getSubunitGraphic()).realRotate(asmtosubq);
        }
    }

    /**
     * Gets all the free BindingSite(s) in an assembly, and put them in an
     * vector
     * 
     * @return Vector
     *  
     */
    public Vector getFreebss() {
        Vector f = new Vector();
        Vector subs = this.getSubunits();
        for (int i = 0; i < subs.size(); i++) {
            Subunit s = (Subunit) subs.get(i);
            Vector bss = s.getBindSites();
            for (int j = 0; j < bss.size(); j++) {
                BindingSite bs = (BindingSite) bss.get(j);
                if (!bs.isBound()) {
                    f.add(bs);
                }
            }
        }
        return f;
    }

    //print out the binding site tip position within the assembly
    private void printAllTips() {
        Vector subs = mysubs;
        for (int i = 0; i < mysubs.size(); i++) {
            Subunit sub = (Subunit) subs.get(i);
            Vector bss = sub.getBindSites();
            for (int j = 0; j < bss.size(); j++) {
                BindingSite bs = (BindingSite) bss.get(j);
                System.out.println(getBindingSiteTipPos(sub, bs));
            }

        }

    }

    //After one binding is processed, find all other possible free binding
    // sites near in space that could form a bond
    private void processLoop(Assembly a0, Assembly a1, BindingSite bs0,
            BindingSite bs1) {

        Debug.checkFalse(bs0.bindTo(bs1),
                "addsubunit bind in Assembly.java didn't work");
        Subunit sub1 = a1.getSubunit(bs1);
        Subunit sub0 = getSubunit(bs0);
        sub1.addPartner(sub0);
        sub0.addPartner(sub1);

        NeighborGroups ngloop = new NeighborGroups(a0, true);
        Vector v = null;
        Vector subs = a1.getSubunits();
        for (int i = 0; i < subs.size(); i++) {
            Subunit sub = (Subunit) subs.get(i);
            Vector bss = sub.getBindSites();
            for (int k = 0; k < bss.size(); k++) {

                BindingSite tembs1 = (BindingSite) bss.get(k);
                if (!tembs1.isBound()) {
                    Vector3d bsTipTem1 = a1.getBindingSiteTipPos(sub, tembs1);
                    Vector bsNeighbors = ngloop.findLoopCandidates(bsTipTem1);

                    if (bsNeighbors.size() == 0) {
                        continue;
                    }

                    for (int j = 0; j < bsNeighbors.size(); j++) {
                        BSA temBSA = (BSA) bsNeighbors.get(j);
                        BindingSite tembs0 = temBSA.bs;
                        if (tembs1.getBST().isCompat(tembs0.getBST())) {
                            Vector3d temv = new Vector3d();
                            temv = a0.getBindingSiteTipPos(temBSA.sub, tembs0);
                            temv.sub(bsTipTem1);
                            if (temv.length() <= Test.distanceTolerance) {
                                if (Test.displayFastBinding) {
                                    tembs1.colorPink();
                                    tembs0.colorPink();
                                }
                                Debug
                                        .checkFalse(tembs1.bindTo(tembs0),
                                                "addsubunit2 bind in Assembly.java didn't work");
                                (temBSA.sub).addPartner(sub);
                                sub.addPartner(temBSA.sub);

                            }
                        }

                    }
                }
            }
        }

    }

    /**
     * returns true if there is steric hindrance in this assembly (ie: at least
     * two subunits are overlapping)
     * 
     * @return boolean
     */
    private boolean stericHindrance(Assembly partner) {
        NeighborGroups ng = null;
        Vector v = null;
        if (this.numSubunits() > partner.numSubunits()) {
            ng = new NeighborGroups(this);
            v = partner.getSubunits();
        } else {
            ng = new NeighborGroups(partner);
            v = this.getSubunits();
        }

        for (int i = 0; i < v.size(); i++) {
            Subunit sub1 = (Subunit) v.get(i);

            if (ng.findHindrance(sub1)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Returns the Subunit in this Assembly that holds bs, if that's not found
     * returns a new blank Subunit
     * 
     * @param bs
     *            BindingSite
     * 
     * @return Subunit
     */
    public Subunit getSubunit(BindingSite bs) {
        int subid = bs.getSubunitID();

        for (int i = 0; i < mysubs.size(); i++) {
            Subunit cursub = (Subunit) mysubs.get(i);

            Vector bss = cursub.getBindSites();
            for (int j = 0; j < bss.size(); j++) {
                if (((BindingSite) bss.get(j)).equals(bs))
                    return cursub;
            }
        }
        return new Subunit();
    }

    /**
     * bind an broken bond which breaks a loop in an assembly
     * 
     * @param e
     *            void
     *  
     */
    public void fastBind(FormBondEvent e) {
        Debug.checkFalse(e.getBS().bindTo(e.getPartner()),
                "error_in_fastBind_in_Assembly.");
        Subunit sub = getSubunit(e.getBS());
        Subunit subPartner = getSubunit(e.getPartner());
        sub.addPartner(subPartner);
        subPartner.addPartner(sub);
    }

    /**
     * get the amounts of different types of free BindingSite(s) in an assembly
     * 
     * @return int[]
     *  
     */
    public int[] getBSCounts() {
        int[] a = new int[3];
        for (int i = 0; i < mysubs.size(); i++) {
            Subunit s = (Subunit) mysubs.get(i);
            Vector bss = s.getBindSites();
            for (int j = 0; j < bss.size(); j++) {
                BindingSite b = (BindingSite) bss.get(j);
                if (!b.isBound()) {
                    if (b.getBST().getName().equals("bsta")) {
                        a[0] = a[0] + 1;
                    } else if (b.getBST().getName().equals("bstb")) {
                        a[1] = a[1] + 1;
                    } else if (b.getBST().getName().equals("bstc")) {
                        a[2] = a[2] + 1;
                    } else {
                        System.out.println("error_in_BSCount_in_Assembly");
                    }

                }
            }
        }
        return a;
    }

    private void addSubunit(Subunit newsub, BindingSite newbs,
            BindingSite mybs, boolean loop) {

        if (!loop) {
            Vector tmpbs = newsub.getBindSites();
            for (int i = 0; i < tmpbs.size(); i++) {
                ((BindingSite) tmpbs.get(i)).setAssemName(myname);
            }
            mysubs.add(newsub);
        }

        //add the subunit to the graphic (even if it's a loop, the graphic was
        // detached, so must be reattached)
        mygraphic.addSubunit(newsub);
    }

    /**
     * Used to split this Assembly when indicated by a BreakBondEvent. Returns
     * the newly split off Assembly, or null if the split didn't make a new
     * Assembly (rather, it broke a loop)
     * 
     * @param e
     *            BreakBondEvent, the event that causes the split
     * @param newasmnum
     *            int, the number of the Assembly to be split off
     * 
     * @return Assembly
     */
    public Assembly splitAssembly(BreakBondEvent e, int newasmnum) {
        BindingSite bs1 = e.getBS();
        BindingSite bs2 = e.getPartner();
        Subunit bsSub1 = getSubunit(bs1);
        Subunit bsSub2 = getSubunit(bs2);
        Debug
                .checkFalse(bs1.breakBond(),
                        "processEvent: breakBond() was unsuccessful in Assembly.java: splitAssembly");

        //break the bond (on Subunit level)
        Debug
                .checkFalse(
                        bsSub1.removePartner(bsSub2),
                        "tried to removePartner on Subunit that isn't a partner with bsSub1 in Assembly.java: splitAssembly");
        Debug
                .checkFalse(
                        bsSub2.removePartner(bsSub1),
                        "tried to removePartner on Subunit that isn't a partner with bsSub2 in Assembly.java: splitAssembly");

        CheckConnected cc = new CheckConnected();
        if (!(cc.isConnected(bsSub1, bsSub2))) {
            Vector keepsubunits = cc.getConnected(bsSub1);
            //second assembly
            CheckConnected cc2 = new CheckConnected();
            Vector detachsubunits = cc2.getConnected(bsSub2);
            //detach each SubunitGraphic and update BindingSite Assembly names
            Vector tmpbs;
            for (int j = 0; j < detachsubunits.size(); j++) {
                Subunit cursub = (Subunit) detachsubunits.get(j);
                mysubs.remove(cursub);
                cursub.detachGraphic();
                tmpbs = cursub.getBindSites();
                for (int i = 0; i < tmpbs.size(); i++) {
                    ((BindingSite) tmpbs.get(i)).setAssemName("Assembly#"
                            + (newasmnum));
                }
            }

            //make the new Assembly
            Assembly detachassembly = new Assembly(detachsubunits, "Assembly#"
                    + (newasmnum));

            return detachassembly;
        }

        //if bond break didn't make 2 assemblies, return null

        return null;
    }

    /**
     * Carries out the FormBondEvent e by binding bindasm to this through a bond
     * between bsSub1 in this Assembly and bsSub2 in bindasm.Both Assembly(s)
     * should be different. If the binding can not occur (because of steric
     * hindrance), returns a modified version of bindasm, otherwise returns
     * null.
     * 
     * @param bindasm
     *            Assembly, assembly to be bound
     * @param e
     *            FormBondEvent, the event that causes the binding
     * @param bsSub1
     *            Subunit, the Subunit on this Assembly that holds the
     *            BindingSite which forms the bond
     * @param bsSub2
     *            Subunit, the Subunit on bindasm that holds the BindingSite
     *            which forms the bond
     * 
     * @return Assembly
     *  
     */

    public Assembly bindAssemblies(Assembly bindasm, FormBondEvent e,
            Subunit bsSub1, Subunit bsSub2) {
        //oldasm = this; newasm = bindasm
        BindingSite oldbs = e.getBS();
        BindingSite newbs = e.getPartner();
        Debug.checkFalse(oldbs.getBST().isCompat(newbs.getBST()),
                "error1_in_bindAssemblies_in_Assembnly");
        Debug.checkFalse(!e.getBS().getAssemName().equals(
                e.getPartner().getAssemName()),
                "error2_in_bindAssembly_in_Assembly");

        //the translation and rotation info of old Assembly will be used to
        // relocate old Assembly or new produced Assembly
        Vector3d oldasmposv = getPos();
        Quat4d oldasmrotq = getRot();
        //in case of unsuccessful binding, send each Assembly back to the
        // original location and rotation
        Vector3d bindAsmposv = bindasm.getPos();
        Quat4d bindAsmrotq = bindasm.getRot();

        bindasm.centerAssembly(bsSub2);
        bindasm.updateLocation(new Transform3D());

        this.centerAssembly(bsSub1);
        this.updateLocation(new Transform3D());

        //find positions of oldbs and newbs
        Vector3d oldBsV = oldbs.getPos();
        Vector3d newBsV = newbs.getPos();
        Vector3d vcenter1 = new Vector3d(oldBsV);
        double angle = oldBsV.angle(newBsV);

        if (Test.simChoice == "virus") {
            //For Virus assembly
            //The method of rotation and translation are same for binding bsta
            // and bstb or bstc with bstc
            AxisAngle4d firstAngle = new AxisAngle4d(newbs.getUp(), Math.PI
                    - angle);
            bindasm.mygraphic.setRot(firstAngle);
            vcenter1.scale(2 * 0.13 / 0.08);
            bindasm.mygraphic.setPos(vcenter1);

        } else if (Test.simChoice == "actin") {
            //for Actin, the angle between bs and bsPartner is Math.PI
            vcenter1.scale(2 * 0.13 / 0.08);
            bindasm.mygraphic.setPos(vcenter1);
        } else if (Test.simChoice == "cube") {
            //for cube assembly

            if (angle > 0.1) {
                //the angle between bs and bsPartner is 0.5*Math.PI if binding
                // involved bsta and bstb
                AxisAngle4d firstAngle = new AxisAngle4d(newbs.getUp(), Math.PI
                        - angle);
                bindasm.mygraphic.setRot(firstAngle);
                vcenter1.scale(2 * 0.13 / 0.08);
                bindasm.mygraphic.setPos(vcenter1);
            } else {
                //the angle between bs and bsPartner is 0 if binding involved
                // bstc and bstc
                //first rotate Math.PI along (1,0,0), then rotate 0.5*Math.PI
                // along (0,0,1), then move bindasm

                AxisAngle4d firstAngle = new AxisAngle4d(newbs.getUp(), Math.PI
                        - angle);
                Quat4d q1 = new Quat4d();
                q1.set(firstAngle);
                AxisAngle4d secondAngle = new AxisAngle4d(new Vector3d(0, 0,
                        1.0), -0.5 * Math.PI);
                Quat4d q2 = new Quat4d();
                q2.set(secondAngle);
                q1.mul(q2);
                firstAngle.set(q1);

                bindasm.mygraphic.setRot(firstAngle);
                vcenter1.scale(2 * 0.13 / 0.08);
                bindasm.mygraphic.setPos(vcenter1);
            }
        } else if (Test.simChoice == "tubulin") {
            //for Tubulin
            if (angle > 0.1) {
                AxisAngle4d firstAngle = new AxisAngle4d(newbs.getUp(), Math.PI
                        - angle);
                bindasm.mygraphic.setRot(firstAngle);
                vcenter1.scale(2 * 0.13 / 0.08);
                bindasm.mygraphic.setPos(vcenter1);
                AxisAngle4d secondAngle = new AxisAngle4d(oldbs.getPos(),
                        0.0296);
                Quat4d q2 = new Quat4d();
                q2.set(secondAngle);
                bindasm.mygraphic.rotate(q2);

            } else {
                AxisAngle4d firstAngle = new AxisAngle4d(newbs.getUp(), Math.PI
                        - angle);
                bindasm.mygraphic.setRot(firstAngle);
                vcenter1.scale(2 * 0.13 / 0.08);
                bindasm.mygraphic.setPos(vcenter1);
            }
        } else if (Test.simChoice == "tetrahedron") {
            //for Tetrahedron, angle will be 0 for binding between bsta with
            // bsta,bstabwith bstb bstc with bstc
            AxisAngle4d firstAngle = new AxisAngle4d(0, 0, 1, Math.PI - angle);
            Quat4d q1 = new Quat4d();
            q1.set(firstAngle);

            AxisAngle4d secondAngle = new AxisAngle4d(oldbs.getUp(), Math.PI
                    - 2 * Math.asin(Math.sqrt(1.0 / 3)));
            Quat4d q2 = new Quat4d();
            q2.set(secondAngle);
            q2.mul(q1);
            bindasm.mygraphic.setRot(q2);
            vcenter1.scale(2 * 0.13 / 0.08);
            bindasm.mygraphic.setPos(vcenter1);
        }

        //compound bindasm so the subunits in bindasm hold the values for
        // pos/rot in relation to newasm or (0,0,0)
        bindasm.compoundTransforms();
        if (stericHindrance(bindasm)) {
            //unsuccessful binding due to setric hindrance, send two
            // Assembly(s) back to their original positions

            this.updateLocation(oldasmrotq, oldasmposv);
            bindasm.updateLocation(bindAsmrotq, bindAsmposv);
            return null;
        } else {

            processLoop(this, bindasm, oldbs, newbs);
            Vector bindsubs = bindasm.getSubunits();
            //Adds all Subunit(s) in bindasm to this Assembly
            for (int i = 0; i < bindsubs.size(); i++) {
                Subunit cursub = (Subunit) bindsubs.get(i);
                this.addSubunit(cursub, newbs, oldbs, false);
            }

            //return this new produced assembly to original position of oldasm
            (bindasm.getAssemblyGraphic()).detach();
            this.updateLocation(new Quat4d(), oldasmposv);
            return this;
        }

    }

    /**
     * Composes Assembly and Subunit transforms so that the Subunits hold all
     * the significant location information and the Assembly location
     * information is all the identity function
     */
    public void compoundTransforms() {
        //get assembly transform
        Transform3D asmtr = mygraphic.getTransform();

        //go through all subunits, composing transform
        for (int i = 0; i < mysubs.size(); i++) {

            Subunit cursub = (Subunit) mysubs.get(i);
            Transform3D subtr = cursub.getTransform();
            Transform3D product = new Transform3D();
            product.mul(asmtr, subtr);
            cursub.setTrans(product);
        }
        //set assembly transform to identity
        updateLocation(new Quat4d(0.0, 0.0, 0.0, 1.0), new Vector3d(0.0, 0.0,
                0.0));
    }

    /**
     * Returns the Vector3d v rotated by Quat4d q
     * 
     * @param v
     *            Vector3d
     * @param q
     *            Quat4d
     * 
     * @return Vector3d
     */
    private Vector3d rotate(Vector3d v, Quat4d q) {
        double[] vd = new double[3];
        v.get(vd);
        Quat4d vq = new Quat4d(vd[0], vd[1], vd[2], 0.0);

        Quat4d rotvq = new Quat4d();
        rotvq.mul(q, vq);
        rotvq.mulInverse(q);

        double[] rotvd = new double[4];
        rotvq.get(rotvd);
        Vector3d rotv = new Vector3d(rotvd[0], rotvd[1], rotvd[2]);
        return rotv;
    }

    //rotates this assembly by rot
    public void rotate(Quat4d rot) {
        mygraphic.rotate(rot);
    }

    /**
     * Returns the position of the binding site relative to the real axis system
     * 
     * @param sub
     *            Subunit, holds bs
     * @param bs
     *            BindingSite
     * 
     * @return Vector3d
     */
    public Vector3d getBindingSitePos(Subunit sub, BindingSite bs) {
        return mygraphic.getBindingSitePos(sub, bs);
    }

    public Vector3d getBindingSiteTipPos(Subunit sub, BindingSite bs) {

        Transform3D subTF = sub.getTransform();

        Point3d bsTip = new Point3d(bs.getTipPos());

        subTF.transform(bsTip);

        return new Vector3d(bsTip);

    }

    /**
     * Returns the rotation of the binding site relative to the real axis system
     * 
     * @param sub
     *            Subunit, holds bs
     * @param bs
     *            BindingSite
     * 
     * @return Quat4d
     */
    public Quat4d getBindingSiteRot(Subunit sub, BindingSite bs) {
        return mygraphic.getBindingSiteRot(sub, bs);
    }

    /**
     * Returns this Assembly's AssemblyGraphic
     * 
     * @return AssemblyGraphic
     */
    public AssemblyGraphic getAssemblyGraphic() {
        return mygraphic;
    }

    /**
     * Returns this Assembly's transform
     * 
     * @return Transform3D
     */
    public Transform3D getTransform() {
        return mygraphic.getTransform();
    }

    /**
     * Returns this Assembly's rotation
     * 
     * @return Quat4d
     */
    public Quat4d getRot() {

        return mygraphic.getRot();
    }

    /**
     * Returns this Assembly's position
     * 
     * @return Vector3d
     */
    public Vector3d getPos() {
        return mygraphic.getPos();
    }

    /**
     * Returns this Assembly's name
     * 
     * @return String
     */
    public String getName() {
        return myname;
    }

    /**
     * Returns a Vector of this Assembly's Subunits
     * 
     * @return Vector
     */
    public Vector getSubunits() {
        return mysubs;
    }

    /**
     * Sets this Assembly's position to v
     * 
     * @param v
     *            Vector3d
     */
    public void setPos(Vector3d v) {
        mygraphic.setPos(v);
    }

    /**
     * Sets this Assembly's position to v and rotation to q
     * 
     * @param q
     *            Quat4d
     * @param v
     *            Vector3d
     */
    public void updateLocation(Quat4d q, Vector3d v) {
        mygraphic.updateLocation(q, v);
    }

    /**
     * Sets this Assembly's transform to tr
     * 
     * @param tr
     *            Transform3D
     */
    public void updateLocation(Transform3D tr) {
        mygraphic.updateLocation(tr);
    }

    /**
     * Returns true if s is in this Assembly
     * 
     * @param s
     *            Subunit
     * 
     * @return boolean
     */
    public boolean containsSubunit(Subunit s) {
        for (int i = 0; i < mysubs.size(); i++) {
            if (mysubs.elementAt(i).equals(s))
                return true;
        }
        return false;
    }

    /**
     * public boolean containsSubunit(Subunit s) { return mysubs.contains(s); }
     * 
     * /** Returns number of subunits in this Assembly
     * 
     * @return int
     */
    public int numSubunits() {
        return mysubs.size();
    }

    /**
     * Sets the next event for this Assembly to ev
     * 
     * @param ev
     *            Event
     */
    public void setNextEvent(Event ev) {
        nextEvent = ev;
    }

    /**
     * Returns this Assembly's next Event
     * 
     * @return Event
     */
    public Event nextEvent() {
        return nextEvent;
    }

    /**
     * Returns this Assembly's next Event's end time
     * 
     * @return double
     */
    //note: throws pointer exception if event isn't set
    public double nextEventEndTime() {
        if (nextEvent != null)
            return nextEvent.getEndTime();
        return Double.MAX_VALUE;
    }

    /**
     * Sets the valid time for this Assembly to time
     * 
     * @param time
     *            double
     */
    public void setValidTime(double time) {
        validTime = time;
    }

    /**
     * Returns this Assembly's valid time
     * 
     * @return double
     */
    public double validTime() {
        return validTime;
    }

    /**
     * Returns this true if this Assembly is the same object as Assembly a
     * 
     * @param a
     *            Assembly
     * 
     * @return boolean
     */
    public boolean equals(Assembly a) {
        return myname.equals(a.getName());
    }

    /**
     * Returns a String version of this Assembly
     * 
     * @return String
     */
    public String toString() {
        String mystring = new String("Assembly name: " + myname
                + "\nSubunit IDs:\n");
        for (int i = 0; i < mysubs.size(); i++) {
            mystring += "\t" + (Subunit) mysubs.get(i);
        }
        return mystring;
    }

}