
import com.sun.j3d.utils.geometry.Sphere;
import javax.media.j3d.*;
import javax.vecmath.*;
import java.util.Vector;

/**
 * extends branchGroup to make a complete Subunit graphic using BindingSiteGraphic components:

                SubunitGraphic
		     /     | 		
                Lights     TRansformGroup     
		           |
			  protein
	        	  /    \        ...
		     sphere    TransformGroup  ...
		                |      ...
		                cone  ...

 * @author Rori Rohlfs
 * @version 1.0
 *
 *
 */
public class SubunitGraphic extends BranchGroup {

    //main TransformGroup
    private TransformGroup tg;

    /**
     * Constructs a SubunitGraphic with a default BranchGroup leading to 
     * the main TransformGroup and a sphere for the basic Subunit
     */
    public SubunitGraphic() {
        super();
        this.setCapability(BranchGroup.ALLOW_DETACH);
        this.setCapability(Group.ALLOW_CHILDREN_WRITE);

        /*
         * prepare the Subunit's appearance
         */
        ColoringAttributes ca = new ColoringAttributes(1.0f, 0.5f, 0.0f, 1);
        Appearance app = new Appearance();
        Material mat = new Material();
        mat.setAmbientColor(1.0f, 0.0f, 0.0f);
        app.setMaterial(mat);
        app.setColoringAttributes(ca);

        /**
         * Creats a sphere to represent a Subunit with specified radius,
         * specifies that normals are generated along with the positions. sets
         * its apperance
         */
        Sphere test_prot = new Sphere(0.5f, Sphere.GENERATE_NORMALS, app);
	
	//set up the main TransformGroup
        tg = new TransformGroup();
        BranchGroup protein = new BranchGroup();
        protein.addChild(test_prot);
        tg.addChild(protein);
        tg.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        tg.setCapability(Group.ALLOW_CHILDREN_WRITE);
        this.addChild(tg);

        //Sets light property of this SubunitGraphic
        AmbientLight lightA = new AmbientLight();
        lightA.setInfluencingBounds(new BoundingSphere());
        this.addChild(lightA);
        DirectionalLight lightdi = new DirectionalLight();
        lightdi.setInfluencingBounds(new BoundingSphere());
        this.addChild(lightdi);
    }

    /**
     * Constructs a SubunitGraphic based on sub.  the main TransformGroup is parent
     * to the Subunit sphere and graphics for all the BindingSites found in sub.  
     * 
     * @param sub
     *            Subunit
     *  
     */
    public SubunitGraphic(Subunit sub) {
        super();
        this.setCapability(BranchGroup.ALLOW_DETACH);
        this.setCapability(Group.ALLOW_CHILDREN_WRITE);
        this.setCapability(Group.ALLOW_CHILDREN_EXTEND);

        //protien holds sphere and cones for the subunit,
        BranchGroup protein = new BranchGroup();

        //Makes a "sphere",Sets its Apperance including color, material and
        //light property , then adds Sphere to BranchGroup protein
        ColoringAttributes ca = new ColoringAttributes(1.0f, 0.5f, 0.0f, 1);
        Appearance app = new Appearance();
        Material mat = new Material();
        mat.setDiffuseColor(0.6f, 0.0f, 0.0f);
        mat.setAmbientColor(0.3f, 0.0f, 0.0f);
        app.setMaterial(mat);
        app.setColoringAttributes(ca);
        Sphere test_prot = new Sphere((float) sub.getRadius(),
                Sphere.GENERATE_NORMALS, app);
        protein.addChild(test_prot);

        /*
         * Makes a cone for each BindingSite of Subunit sub and adds them to
         * protein BranchGroup
         */
        Vector bss = sub.getBindSites();
        for (int i = 0; i < bss.size(); i++) {
            BindingSite curbs = (BindingSite) bss.get(i);
            BindingSiteGraphic curbsg = curbs.getGraphic();

            protein.addChild(curbsg);
        }

        //sets up appropriate main TransformGroup
        Transform3D trans = new Transform3D(sub.getRot(), sub.getPos(), 1.0);
        tg = new TransformGroup(trans);
        tg.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        tg.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        tg.setCapability(Group.ALLOW_CHILDREN_WRITE);

        tg.addChild(protein);
        this.addChild(tg);

        //Sets light property of this SubunitGraphic
        AmbientLight lightA = new AmbientLight();
        lightA.setInfluencingBounds(new BoundingSphere());
        this.addChild(lightA);
        DirectionalLight lightdi = new DirectionalLight();
        lightdi.setInfluencingBounds(new BoundingSphere());
        this.addChild(lightdi);
    }

    /**
     * Sets the location of this SubunitGraphic to specified
     * rotation and position
     * 
     * @param rot
     *            Quat4d
     * @param pos
     *            Vector3d
     */
    public void updateLocation(Quat4d rot, Vector3d pos) {
        Transform3D trans = new Transform3D(rot, pos, 1.0);
        tg.setTransform(trans);
    }

    /**
     * Sets location of this SubunitGraphic to Transform3D t
     * 
     * @param t
     */
    public void updateLocation(Transform3D t) {
        tg.setTransform(t);
    }

    /**
     * Rotates this SubunitGraphic by q
     * 
     * @param q
     *            Quat4d
     *  
     */
    public void rotate(Quat4d q) {
        Transform3D trans = new Transform3D();
        tg.getTransform(trans);

        Quat4d rot = new Quat4d();
        trans.get(rot);
        Quat4d prod = new Quat4d();
        prod.mul(q, rot);

        trans.setRotation(prod);
        tg.setTransform(trans);
    }

    /**
     * Translates this SubunitGraphic by v
     * 
     * @param v
     *            Vector3d
     */
    public void translate(Vector3d v) {
        Transform3D trans = new Transform3D();
        tg.getTransform(trans);

        Vector3d pos = new Vector3d();
        trans.get(pos);
        pos.add(v);

        trans.setTranslation(pos);
        tg.setTransform(trans);
    }

    /**
     * Sets position of this SubunitGraphic to Vector3d v
     * 
     * @param v
     *            Vector3d
     *  
     */
    public void setPos(Vector3d v) {
        Transform3D trans = new Transform3D();
        tg.getTransform(trans);
        trans.setTranslation(v);
        tg.setTransform(trans);
    }

    /**
     * Sets rotation of this SubunitGraphic to Quat4d q
     * 
     * @param q
     *            Quat4d
     */
    public void setRot(Quat4d q) {
        Transform3D trans = new Transform3D();
        tg.getTransform(trans);
        trans.setRotation(q);
        tg.setTransform(trans);
    }

    /**
     * Return SubunitGraphic as a BranchGroup
     * 
     * @return BranchGroup
     */
    public BranchGroup branchGroup() {
        return (BranchGroup) this;
    }

    /**
     * Gets the Transform3D of this SubunitGraphic
     * 
     * @param Transform3D
     *  
     */
    public Transform3D getTransform() {
        Transform3D t = new Transform3D();
        tg.getTransform(t);
        return t;
    }

    /**
     * Returns the position of the BindingSite bs in relation to the Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     *  
     */
    public Vector3d getBindingSitePos(BindingSite bs) {
        Transform3D subtranst = new Transform3D();
        tg.getTransform(subtranst);
        Vector3d subtrans = new Vector3d();
        subtranst.get(subtrans);

        Vector3d bstrans = bs.getPos();

        Vector3d sum = new Vector3d();
        sum.add(subtrans, bstrans);

        return sum;
    }

    /**
     * Returns the actual rotation of the binding site bs in relation to the Assembly
     * 
     * @param bs
     * @return Quat4d
     *  
     */
    public Quat4d getBindingSiteRot(BindingSite bs) {
        Transform3D subtranst = new Transform3D();
        tg.getTransform(subtranst);
        Quat4d subrot = new Quat4d();
        subtranst.get(subrot);

        Quat4d bsrot = bs.getRot();

        Quat4d prod = new Quat4d();
        prod.mul(bsrot, subrot);

        return prod;
    }
}