import java.util.Vector;
import java.lang.Math;
import java.util.Random;
import javax.vecmath.*;
import javax.media.j3d.*;
import com.sun.j3d.utils.*;

/**
 * Subunit holds information about a specific Subunit including its
 * SubunitType, current Conformations, ID, Solution, location/velocity
 * mass, radius, bound Subunits, and graphic.
 * 
 * @author Rori Rohlfs
 * @version 1.0
 *  
 */

public class Subunit {
    //private variables ---------------------------------------
    private SubunitType mySType;
    private Vector myConfs;  //Vector of current Conformations (for each Subunit Domain)
    private int myid;
    private Solution mysoln;
    private Vector3d mypos;    //relative to the assembly
    private Quat4d myrot;      //relative to the assembly
    private Vector3d myv;      //relative to the assembly 
    private Quat4d myrotv;     //relative to the assembly  
    private double mymass;
    private double myradius;
    private Vector mypartners;    //Vector of bound Subunits
    private Vector mybindsites;   //Vector of current binding sites
    private SubunitGraphic mygraphic;

    //constructors ---------------------------------------------
    /**
     * Constructs a Subunit with default SubunitType, and other default
     * parameters
     */
    public Subunit() {
	mySType = new SubunitType();
	myConfs = new Vector();
	myid = -1;
	mysoln = new Solution();

	mypos = new Vector3d();
	myrot = new Quat4d(0.0, 0.0, 0.0, 1.0); 
	myv = new Vector3d(); 
	myrotv = new Quat4d(0.0, 0.0, 0.0, 1.0);
	
	mymass = 1.0;
        myradius = 0.8;	
	mypartners = new Vector();
	mybindsites = new Vector();
	mygraphic = new SubunitGraphic();
    }

    /**
     * Constructs a Subunit with the specified SubunitType, current Conformations, 
     * Solution, ID, position, rotation, velocity, angular velocity, mass, and radius.    
     * A graphic is construced according to the parameters given
     *
     * @param st
     *            SubunitType
     * @param confs
     *            Vector of current Conformations (one for each Domain in this Subunit)
     * @param soln
     *            Solution this Simulation takes place in
     * @param id
     *            int, this Subunit's ID
     * @param p
     *            Vector3d, this Subunit's position relative to its Assembly
     * @param r
     *            Quat4d, this Subunit's rotation relative to its Assembly
     * @param v
     *            Vector3d, this Subunit's velocity relative to its Assembly
     * @param rv
     *            Quat4d, this Subunit's rotational velocity relative to its Assembly
     * @param m
     *            double, this Subunit's mass
     * @param rad
     *            double, this Subunit's radius
     */
    public Subunit(SubunitType st, Vector confs, Solution soln, int id, 
                   Vector3d p, Quat4d r, Vector3d v, Quat4d rv, double m, double rad) {
	mySType = st;
	myConfs = confs;
	myid = id;
	mysoln = soln;

	mypos = p;
	myrot = r;
	myv = v;
	myrotv = rv;

	mymass = m;
	myradius = rad;
	mypartners = new Vector();
	mybindsites = new Vector();
	for(int i=0; i<confs.size(); i++) {
	    Conformation curconf = (Conformation) confs.get(i);
	    Vector bss = curconf.getBindSites();
	    for(int j=0; j<bss.size(); j++) {
		BindingSite curbs = (BindingSite) bss.get(j);
		mybindsites.add(curbs);
	    }
	}

	mygraphic = new SubunitGraphic(this);
    }


    //private methods -------------------------------------------

    /**
     * Returns the change in Energy between the Conformations specified 
     * in Vector confsa and those specified in confsb
     * 
     * @param confsa
     *            Vector of Conformations (one for each Domain in this
     *            Subunit)
     * @param confsb
     *            Vector of Conformations (one for each Domain in this
     *            Subunit)
     * @return double
     */
    private double deltaE(Vector confsa, Vector confsb) {
	double aesum = 0.0;
	double besum = 0.0;	
	for(int i=0; i<confsa.size(); i++) {
	    aesum += ((Conformation) confsa.elementAt(i)).getE();
	    besum += ((Conformation) confsb.elementAt(i)).getE();
	}
	return (besum-aesum);
    }

    /**
     * Finds the next set of Conformations this Assembly will take and
     * returns it along with the time until the conformation change.
     * This function is not presently used and is somwhat dated
     * 
     * @return Vector [Vector of Conformations, transition time Double]
     */
    private Vector NextConf() { 	
	Vector otherConfs = (Vector) mySType.getConfSets();
	Vector eVals = new Vector();
	double sum = 0.0;
	double kb = 1.38*(Math.pow(10.0, -23.0));
	double t = mysoln.getProperty("temp");
	double de = 0.0;
	Double newguy;

	//find total Ks's and make a vector of them
	for(int i=0; i<otherConfs.size(); i++) {
	    de = deltaE(myConfs, (Vector) (otherConfs.elementAt(i)));
	    double tmp = Math.pow(Math.E, (-(de/(kb*t))));
	    sum += tmp;
	    newguy = new Double(tmp);
	    eVals.add(newguy); 
	}
	
	//get a random double in [0, Ks sum]
	double rand = 0.0;//s.getRand();MUST BE CHANGED WHEN CONFORMATINOS START
	rand *= sum;
	
	//find corresponding conformation
	double sum2 = 0.0;
	int index = 0;
	while(sum2 < rand) {
	    sum2 += ((Double) eVals.elementAt(index)).doubleValue();
	    index++;
	}
	
	return (Vector) eVals.elementAt(index);
    }

    
    //public methods --------------------------------------------------

    /**
     * Changes the Conformation of Domain number domain to newconf.  
     * If the conformation change failed, returns false, returns true 
     * otherwise.
     * 
     * @param domain
     *            int, ID of the Domain in question
     * @param newconf
     *            Conformation to be transformed into
     * @return boolean
     */
    public boolean changeConf(int domain, Conformation newconf) { 
	if(mySType.isConf(newconf, domain) && !((Conformation) myConfs.elementAt(domain)).bound()) {
	    myConfs.setElementAt(newconf, domain);
	    
	    mybindsites.removeAllElements();
	    for(int i=0; i<myConfs.size(); i++) {
		Conformation curconf = (Conformation) myConfs.get(i);
		Vector bss = curconf.getBindSites();
		for(int j=0; j<bss.size(); j++) {
		    BindingSite curbs = (BindingSite) bss.get(j);
		    mybindsites.add(curbs);
		}
	    }
	    return true;
	}
	return false;
    }

    /**
     * Returns the position of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite off of this Subunit
     * @return Vector3d
     */
    public Vector3d getBSPos(BindingSite bs) {
	Vector3d bspos = new Vector3d();
 
	for(int i=0; i<myConfs.size(); i++) {
	    Conformation curconf = (Conformation) myConfs.elementAt(i);
	    //if curconf has bs, get bs pos
	    if(curconf.hasBS(bs)) 
		bspos = bs.getPos();
	}
	return bspos;
    }

    /**
     * Returns the rotation of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite off of this Subunit
     * @return Quat4d
     */
    public Quat4d getBSRot(BindingSite bs) {
	Quat4d bsor = new Quat4d(0.0,0.0,0.0,1.0);
 
	for(int i=0; i<myConfs.size(); i++) {
	    Conformation curconf = (Conformation) myConfs.elementAt(i);
	    if(curconf.hasBS(bs)) 
		bsor = bs.getRot();
	    }
	return bsor;
    }

    /**
     * Detaches this Subunit's graphic from its parent Assembly graphic
     */
    public void detachGraphic() {
	mygraphic.detach();
    }

    /**
     * Returns this Subunit's SubunitGraphic
     * 
     * @return SubunitGraphic
     */
    public SubunitGraphic getSubunitGraphic() {
	return mygraphic;
    }

    /**
     * Returns this Subunit's position in relation to its Assembly
     * 
     * @return Vector3d
     */
    public Vector3d getPos() {
	return mypos;
    }

    /**
     * Sets this Subunit's position in relation to its Assembly to p
     * 
     * @param p
     *            Vector3d
     */
    public void setPos(Vector3d p) {
	mypos = p;
	mygraphic.updateLocation(myrot, mypos);
    }

    /**
     * Returns this Subunit's rotation in relation to its Assembly
     * 
     * @return Quat4d
     */
    public Quat4d getRot() {
	return myrot;
    }
    
    /**
     * Sets this Subunit's rotation in relation to its Assembly to r
     * 
     * @param p
     *            Quat4d
     */
    public void setRot(Quat4d r) {
	myrot = r;
	mygraphic.updateLocation(myrot, mypos);
    }

    /**
     * Returns this Subunit's rotational velocity in relation to its Assembly
     * 
     * @return Quat4d
     */
    public Quat4d getRotV() {
	return myrotv;
    }

    /**
     * Sets this Subunit's rotational velocity in relation to its Assembly to r
     * 
     * @param p
     *            Quat4d
     */
    public void setRotV(Quat4d r) {
	myrotv = r;
    }
    
    /**
     * Returns this Subunit's velocity in relation to its Assembly
     * 
     * @return Vector3d
     */
    public Vector3d getV() {
	return myv;
    }

    /**
     * Sets this Subunit's velocity in relation to its Assembly to v
     * 
     * @param p
     *            Vector3d
     */
    public void setV(Vector3d v) {
	myv = v;
    }
    
    /**
     * Returns this Subunit's radius
     * 
     * @return double
     */
    public double getRadius() {
        return myradius;
    }

    /**
     * Returns this Subunit's mass
     * 
     * @return double
     */
    public double getMass() {
	return mymass;
    }

    /**
     * Returns this Subunit's Solution
     * 
     * @return Solution
     */
    public Solution getSoln() {
	return mysoln;
    }

   /**
     * Returns this Subunit's SubunitType
     * 
     * @return SubunitType
     */
    public SubunitType getST() {
	return mySType;
    }

   /**
     * Returns a Vector of Subunits bound to this Subunit
     * 
     * @return Vector of Subunits
     */
    public Vector getBoundSubunits() {
	return mypartners;
    }

    /**
     * Sets this Subunit's transformation in relation to its 
     * Assembly to t
     * 
     * @param p
     *            Transform3D
     */
    public void setTrans(Transform3D t) {
	t.get(myrot, mypos);
	mygraphic.updateLocation(t);
    }

    /**
     * Returns true if this Subunit is bound to s, false otherwise
     * 
     * @param s
     *            Subunit
     * @return boolean
     */
    public boolean isBoundSubunit(Subunit s) {
	for(int i=0; i<mypartners.size(); i++) {
	    if(s.equals((Subunit) mypartners.elementAt(i)))
		return true;
	}
	return false;
    }

    /**
     * Returns the orientation of the BindingSite on this Subunit with 
     * which s is bound to this Subunit (in relation to the parent Assembly)
     * 
     * @param s
     *            Subunit
     * @return Vector3d
     */
    public Vector3d getBSOrientation(Subunit s) {
	BindingSite bs = new BindingSite();
	
	//search s's bindsites for one bound to this
	for(int i=0; i<mybindsites.size(); i++) {
	    BindingSite mybs = (BindingSite) mybindsites.get(i);
	    if(mybs.isBound()) {
		BindingSite mybspartner = mybs.getPartner();
		//check to see if these bs's are on the right subunits 
		if(mybspartner.getSubunitID() == s.getid()) { 
		    bs = mybs;
		}
	    }
	}
	Vector3d or = getBSOrientation(bs);

	return or;
    }

    /**
     * Returns a Vector of all the BindingSites that bind this Subunit
     * to s.  The Vector is arranged as:
     * [this Subunit's bs, s's bound bs, this Subunit's bs2, s's bound bs2, ...]
     * 
     * @param s
     *            Subunit
     * @return Vector
     */
    public Vector getConnectingBSs(Subunit s) {
	Vector bss = new Vector();
       
	//search s's bindsites for sites bound to this
	for(int i=0; i<mybindsites.size(); i++) {
	    BindingSite mybs = (BindingSite) mybindsites.get(i);
	    if(mybs.isBound()) {
		BindingSite mybspartner = mybs.getPartner();
		//check to see if these bs's are on the right subunits 
		if(mybspartner.getSubunitID() == s.getid()) { 
		    bss.add(mybs);
		    bss.add(mybspartner);
		}
	    }
	}
	return bss;
    }

    /** 
     * Returns true if this Subunit is not bound to any other Subunit, 
     * false otherwise
     *
     * @return boolean
     */   
    public boolean isUnbound() {
        return (mypartners.size() == 0);
    }

    /** 
     * Add Subunit s to this Subunit's list of bound partner Subunits
     *
     * @param s Subunit to add
     */
    public void addPartner(Subunit s) {
	mypartners.add(s);
    }

    /** 
     * Removes the specified Subunit from the vector of bound partner Subunits.
     * Returns true if the removal was sucessful, false otherwise
     *     
     * @param s Subunit to be removed
     * @return boolean
     */    
    public boolean removePartner(Subunit s) {
        int id = s.getid();
        for(int i=0; i<mypartners.size(); ++i) {
            if(id == ((Subunit)(mypartners.get(i))).getid()) {
                mypartners.remove(i);
                return true;
            }
        }
        return false;
    }

    /**
     * Returns this Subunit's ID
     * 
     * @return int
     */
    public int getid() {
	return myid;
    }

    /**
     * Sets this Subunit's ID to i
     * 
     * @param i
     *            int
     */
    public void setid(int i) {
	myid = i;
    }

    /**
     * Returns true if this Subunit is Subunit s
     * 
     * @param s
     *            Subunit
     * @return boolean
     */
    public boolean equals(Subunit s) {
        return (myid == s.getid());
    }

    /**
     * Returns a vector pointing in the direction of bs in realtion to 
     * the parent assembly
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     */
    public Vector3d getBSOrientation(BindingSite bs) {
	//getting information to do rotation with quaternions
	Quat4d bsor = bs.getRot();
	Quat4d subunitor = new Quat4d(myrot);
	Quat4d startpos = new Quat4d(0.0, 1.0, 0.0, 0.0); //yaxis

	//do the rotation: finalor = (subunitor*bsor)*startpos*((subunitor*bsor)conjugate)
	Quat4d product = new Quat4d();
	product.mul(subunitor, bsor);

	Quat4d finalor = new Quat4d();
	finalor.mul(product, startpos);
	finalor.mulInverse(product);	
	finalor.normalize();

	double[] doubleor = new double[4];
	finalor.get(doubleor);
	Vector3d vectoror = new Vector3d(doubleor[0], doubleor[1], doubleor[2]);
	vectoror.scale((bs.getPos()).length() + (bs.length()/2.0)); //doesn't work if bs's aren't normal to the sphere
	

	return vectoror;
    }
    
    /**
     * Returns a Vector of current Conformations (one for each Domain)
     *
     * @return Vector
     */
    public Vector getConfs() {
        return myConfs;
    }

    /**
     * Returns a Vector of the Subunit's current BindingSites and resets the 
     * BindingSites' Domain ID's
     *
     * @return Vector
     */
    public Vector getBindSites() {
	Vector bsites = new Vector();
	for(int i=0; i<myConfs.size(); i++) {
	    Vector cursites = new Vector();
	    cursites = ((Conformation) myConfs.elementAt(i)).getBindSites();
	    for(int j=0; j<cursites.size(); j++) {
		BindingSite curbs = (BindingSite) cursites.elementAt(j);
		curbs.setDomainID(j);
		bsites.add(curbs);
	    }
	}	
	return bsites;
    }

    /**
     * Returns a Vector of the positions of this Subunit's current 
     * BindingSites
     *
     * @return Vector
     */
    public Vector getBindSitePoses() {
	Vector bsposes = new Vector();
	for(int i=0; i<myConfs.size(); i++) {
	    Vector cursites = new Vector();
	    Conformation curconf = (Conformation) myConfs.elementAt(i);
	    cursites = (curconf).getBindSites();
	    for(int j=0; j<cursites.size(); j++) {
		BindingSite curbs = (BindingSite) cursites.elementAt(j);
		Vector3d curpos = curbs.getPos();
		bsposes.add(curpos);
	    }
	}	
	return bsposes;
    }

    /**
     * Returns a Vector of the rotations of this Subunit's current 
     * BindingSites
     *
     * @return Vector
     */
    public Vector getBindSiteOrs() {
	Vector bsors = new Vector();
	for(int i=0; i<myConfs.size(); i++) {
	    Vector cursites = new Vector();
	    Conformation curconf = (Conformation) myConfs.elementAt(i);
	    cursites = (curconf).getBindSites();
	    for(int j=0; j<cursites.size(); j++) {
		BindingSite curbs = (BindingSite) cursites.elementAt(j);
		Quat4d curor = curbs.getRot();
		bsors.add(curor);
	    }
	}	
	return bsors;
    }

    /**
     * Returns the transform of bs in relation to the parent Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Transform3D
     */
    public Transform3D getBStoAsmTransform(BindingSite bs) {
	 Transform3D bstrans = bs.getTransform();
	 Transform3D subtrans = mygraphic.getTransform();
	 Transform3D product = new Transform3D();
	 
	 AxisAngle4d tmpa = new AxisAngle4d();
	 Quat4d subtransq = new Quat4d();
	 Vector3d subtransv = new Vector3d();
	 subtrans.get(subtransq, subtransv);
	 tmpa.set(subtransq);
	
	 Quat4d bstransq = new Quat4d();
	 Vector3d bstransv = new Vector3d();
	 bstrans.get(bstransq, bstransv);
	
	 product.mul(subtrans, bstrans);
	 return product;
    }

    /**
     * Returns the transform of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite
     * @return Transform3D
     */
    public Transform3D getBStoSubTransform(BindingSite bs) {
	 return bs.getTransform();
    }

    /**
     * Returns the up vector of bs in relation to this Subunit
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     */
    public Vector3d getBSUptoSub(BindingSite bs) {
	return bs.getUp();
    }

    /**
     * Returns the up vector of bs in relation to the parent Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     */
    public Vector3d getBSUptoAsm(BindingSite bs) {
	//getting information to do rotation with quaternions
	Vector3d bsup = new Vector3d(bs.getUp());
	bsup.normalize();
	double[] bsupd = new double[3];
	bsup.get(bsupd);
	Quat4d bsupq = new Quat4d(bsupd[0], bsupd[1], bsupd[2], 0.0);

	Quat4d subunitor = new Quat4d(myrot);
	subunitor.normalize();

	//do the rotation: finalor = subunitor*bsupq*(subunitor conjugate)
	Quat4d finalor = new Quat4d();
	finalor.mul(subunitor, bsupq);
	finalor.mulInverse(subunitor);	

	double[] finalord = new double[4];
	finalor.get(finalord);
	Vector3d finalorv = new Vector3d(finalord[0], finalord[1], finalord[2]);
	finalorv.normalize();	

	return finalorv;
    }

    /**
     * Returns this Subunit's transform in relation to the parent Assembly
     * 
     * @return Transform3D
     */
    public Transform3D getTransform() {
	return mygraphic.getTransform();
    }

    /**
     * Translates this Subunit by t
     * 
     * @param t
     *            Vector3d
     */
    public void translate(Vector3d t) {
	mypos.add(t);
	mygraphic.updateLocation(myrot, mypos);
    }

    /**
     * Returns the position of bs in relation to the parent Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Vector3d
     */
    public Vector3d getBindingSitePos(BindingSite bs) {
	return mygraphic.getBindingSitePos(bs);
    }

    /**
     * Returns the rotation of bs in relation to the parent Assembly
     * 
     * @param bs
     *            BindingSite
     * @return Quat4d
     */
    public Quat4d getBindingSiteRot(BindingSite bs) {
	return mygraphic.getBindingSiteRot(bs);
    }

    /**
     * Returns a String version of this Subunit
     * 
     * @return String
     */
    public String toString() {
	String mystring = new String("Subunit ID: " + myid
				     + "\nSubunitType Name: " + mySType.getName()
				     + "\nCurrent Conformation:");
	for(int i=0; i<myConfs.size(); i++) {
	    mystring = mystring + "\nSubSubUnit " + i 
		+ ":\n\tCurrent Conformation: " 
		+ ((Conformation) myConfs.elementAt(i)) + "\n";            
	}
	return mystring;
    }
}
