import java.util.Vector;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;
import java.util.Random;
import java.io.*;
import javax.vecmath.*;

/**
 * Simulation holds all Assembly information and actually runs the
 * queue-based discrete-assembly algorightm
 *  
 * @author Sue Yi Chew
 * @author Rori Rohlfs
 * @version 1.0
 *  
 */

public class Simulation {
    int asnum;

    double curtime;

    Solution mysoln;

    private HashMap assemblies;

    private PriorityQueue pq;

    private SimulatorGraphic graphic;

    /* To store average event times */
    private HashMap bondTimes; /* HashMap of HashMap 
                                   String bstNames |  |  |  |  |  |  | 
                                    |
                                   |  |-> double times[4]. (times[0]-->bindingTime, 
                                   |  |                     times[1]-->breakingTime,
				   |  |                     times[2]-->intraAssemblyBindingTime)
                                   |  |
                                   |  |
                                   |__| (Another HashMap, keys are also String bstNames)
                                */

    private HashMap confTimes; /* HashMap of HashMap 
                                  String confNames |  |  |  |  |  |  | 
                                   |
                                  |  |-> double times[1]. (times[0]-->confChangeTime)
                                  |  |                 
                                  |  |
                                  |  |
                                  |  |
                                  |__| (Another HashMap, keys are also String confNames)
                                */

    Random rand;

    /**
     * Creates a Simulation with no assemblies/subunits in it.
     */
    public Simulation() {
        asnum = 0;
        curtime = 0.0;
        mysoln = new Solution();
        assemblies = new HashMap();
        pq = new BinaryHeap();
        rand = new Random();

        //Places initial events in PriorityQueue pq
        prepEventQ();
    }

    /**
     * Creates a Simulation with specified Solution s, Vector of Assemblies a,
     * double current time, and HashMap of bonding times (detailed in private 
     * variable definitions), and HashMap of conformation changing times
     * (detailed in private variable definitions)
     * 
     * @param s
     *            Solution the properities of environment in which the
     *            simulation occurs
     * @param a
     *            Vector of Assemblies, 
     * @param ct
     *            double current time
     * @param bondTimes
     *            HashMap of HashMaps 
     *                              String bstNames |  |  |  |  |  |  | 
     *                               |
     *                              |  |-> double times[4]. (times[0]-->bindingTime, 
     *                              |  |                     times[1]-->breakingTime,
     *   			    |  |                     times[2]-->intraAssemblyBindingTime)
     *                              |  |
     *                              |  |
     *                              |__| (Another HashMap, keys are also String bstNames)
     *                           
     * @param confTimes
     *             HashMap of HashMaps 
     *                             String confNames |  |  |  |  |  |  | 
     *                              |
     *                             |  |-> double times[1]. (times[0]-->confChangeTime)
     *                             |  |                 
     *                             |  |
     *                             |  |
     *                             |  |
     *                             |__| (Another HashMap, keys are also String confNames)
     *
     */
    public Simulation(Solution s, Vector a, double ct, HashMap bondTimes,
            HashMap confTimes) {
        asnum = 0;
        mysoln = s;
        assemblies = new HashMap();
        for (int i = 0; i < a.size(); ++i) {
            addAssembly((Assembly) (a.get(i)));
        }
        curtime = ct;
        this.bondTimes = bondTimes;
        this.confTimes = confTimes;
        pq = new BinaryHeap();
        rand = new Random();

        //Places initial events in Priority pq based on BinaryHeap
        prepEventQ();

    }

    /**
     * Prepares the empty priority queue by sampling events 
     * and placing them in the priority queue 
     */
    private void prepEventQ() {
        String[] assems = new String[assemblies.size()];
        int count = 0;
        Iterator it = assemblies.keySet().iterator();

        while (it.hasNext()) {
            Object o = it.next();
            assems[count] = (String) (o);

            count++;
        }

        getNewEvents(assems); //all Assembly names
    }

    /**
     * Processes the given event.  That is, all changes on the Simulation
     * caused by ev are carried out and new events are sampled
     * 
     * @param ev
     *            Event
     */
    private void processEvent(Event ev) {
        System.out.println("--Processing Event: " + ev + "----\n");

        if (ev instanceof ConfChangeEvent) {
            ConfChangeEvent e = (ConfChangeEvent) ev;
            e.getSubunit().changeConf(e.getDomainID(), e.getNewConf());
            //should add in relax when relavent
        } 
	else if (ev instanceof BreakBondEvent) {
            BreakBondEvent e = (BreakBondEvent) ev;
            Assembly oldasm = (Assembly) assemblies.get((e.getBS())
                    .getAssemName());
            Assembly newasm = oldasm.splitAssembly(e, asnum);

            //if the break made a new assembly
            if (newasm != null) {
                // add new assembly
                assemblies.put("Assembly#" + (asnum), newasm);
                graphic.addAssemblytoGraphic(newasm);

                // update event's assembly list to reflect what changed i.e.
                // which assemblies to sample from next
                e.setAssemNamesInvolved(new String[] { oldasm.getName(),
                        "Assembly#" + (asnum) });
                asnum++;
            }
        } 
	else if (ev instanceof FormBondEvent) { //e instanceof FormBondEvent
            FormBondEvent e = (FormBondEvent) ev;
            Assembly asm1 = (Assembly) assemblies.get((e.getBS())
                    .getAssemName());
            Assembly asm2 = (Assembly) assemblies.get((e.getPartner())
                    .getAssemName());
            Subunit bsSub = asm1.getSubunit(e.getBS());
            Subunit partnerSub = asm2.getSubunit(e.getPartner());

            Assembly tmp = asm1.bindAssemblies(asm2, e, bsSub, partnerSub);
            //if assemblies couldn't be bound...
            if (tmp != null) {
                assemblies.remove((String) asm2.getName());
                addAssembly(tmp);
            }
            //if asm1 and asm2 were different assemblies, remove old assembly
            else if (!(asm1.getName()).equals(asm2.getName())) {
                assemblies.remove((String) asm2.getName());
            }

            //update freesubs (remove relevent bindingsites)

            // Update event's assembly list to know what to sample from next
            ev.setAssemNamesInvolved(new String[] { asm1.getName() });
            //System.out.println("sample from next: "+asm1.getName());
        } else {
            System.out.println("Error in usage of processEvent() function.");
            System.exit(-1);
        }

        curtime = ev.getEndTime();

    }

    /**
     * Samples new events for the Assemblies named in assems and 
     * adds the soonest event for each Assembly named in assems 
     * to the priority queue.  Assembly given to this method should
     * not have a valid nextEvent.
     * 
     * @param assems
     *            Array of Strings, Assembly names, specifying which Assemblies
     *            to sample new events for.
     */
    private void getNewEvents(String[] assems) {
        // Reset the minTime for all assemblies 
        Iterator it = assemblies.values().iterator();
        while (it.hasNext()) {
            ((Assembly) (it.next())).setMinTime(Double.MAX_VALUE);
        }

        Vector freebss = new Vector(); //Vector of free bindingsites in current
        // Assembly
        Vector allfreebss = new Vector(); //Vector of all free bindingsites in
        // Simulation
        Vector boundbss = new Vector(); //Vector of bound bindingsites in current Assembly

        /**
         * Walks through all Assemblies in this Simulation, finds and saves all
         * free BindingSites by checking if each one is bound.
         */
        it = assemblies.values().iterator();
        Assembly asm;
        Vector tmpsubs;
        Vector tmpbsv;
        BindingSite tmpbs;
        while (it.hasNext()) {
            asm = (Assembly) it.next();
            tmpsubs = asm.getSubunits();
            for (int i = 0; i < tmpsubs.size(); ++i) {
                tmpbsv = ((Subunit) tmpsubs.get(i)).getBindSites();
                for (int j = 0; j < tmpbsv.size(); ++j) {
                    tmpbs = (BindingSite) (tmpbsv.get(j));
                    if (!tmpbs.isBound()) {
                        allfreebss.add(tmpbs);
                    }
                }
            }
        }

        //looking for events
        for (int i = 0; i < assems.length; ++i) {
            if (assems[i] == null) {
                continue;
            }
            freebss.clear();
            boundbss.clear();
	    Event minTimeEvent = new Event();

            Assembly assem = (Assembly) assemblies.get(assems[i]);
	    //update assem's validTime
            assem.setValidTime(curtime);

            Vector subs = assem.getSubunits();
            for (int j = 0; j < subs.size(); ++j) {
                Subunit sub = (Subunit) subs.get(j);

                //For each domain: a possible conformation change
                Vector curConfs = sub.getConfs();
                
                //Vector of Vector of Conformations
		Vector confSets = sub.getST().getConfSets();

		//go through all possible conformations
                for (int k = 0; k < confSets.size(); ++k) {
                    for (int m = 0; m < ((Vector) confSets.get(k)).size(); ++m) {
                        Conformation curconf = (Conformation) curConfs.get(k);
                        Conformation confb = (Conformation) ((Vector) confSets
                                .get(k)).get(m);

			//skip non-conformation change
                        if (curconf.isEqual(confb))
			    continue;

			double[] transTimeArr = (double[]) (((HashMap) 
							     (confTimes.get(curconf.getName())))
							    .get(confb.getName()));
			double transitionTime = getExp(transTimeArr[0]);

			double endTime = curtime + transitionTime;

			//if this is the fastest Event so far, store it
			if(assem.minTime() > endTime) {
			    assem.setMinTime(endTime);

			    String[] assemName = {assem.getName()};
			    minTimeEvent = new ConfChangeEvent(curtime, endTime, assemName, sub, 
							       curconf.getDomainID(), confb);
			}
		    }//conformations loop
                }//domains loop

                //add all the binding sites in this subunit into boundbss or freebss
                Vector bindsites = sub.getBindSites();
                for (int k = 0; k < bindsites.size(); ++k) {
                    BindingSite bs = (BindingSite) bindsites.get(k);
                    if (bs.isBound()) {
                        boundbss.add(bs);
                    } 
		    else {
                        freebss.add(bs);
                    }
                }
            }//subs in the asm loop

            BindingSite bs, partner;
            double duration, endTime;
            Assembly assemBS = new Assembly();
            Assembly assemPartner = new Assembly();
            double[] bTimes;

            CheckPairSeen cps = new CheckPairSeen();

            //go thru boundsubs, looking at all possible bond breaking events
            for (int j = 0; j < boundbss.size(); j++) {
                bs = (BindingSite) boundbss.get(j);
                partner = bs.getPartner();

                
                if (cps.marked(bs.getid(), partner.getid()))
                    continue;
                //else mark it seen now
                cps.addPair(bs.getid(), partner.getid());

                bTimes = (double[]) (((HashMap) (bondTimes.get(bs.getBST()
                        .getName()))).get(partner.getBST().getName()));
                duration = getExp(bTimes[1]);

                endTime = curtime + duration;

		//if this is the fastest Event so far, store it
		assemBS = (Assembly) assemblies.get(bs.getAssemName());
		if(assemBS.minTime() > endTime) {
		    assemBS.setMinTime(endTime);

		    String[] assemNames = {assemBS.getName(), null};
		    minTimeEvent = new BreakBondEvent(curtime, endTime,
						      assemNames, bs, partner);
		}
            }

            //go thru freebss, looking at all possible bond forming events
            for (int k = 0; k < freebss.size(); k++) {

                bs = (BindingSite) freebss.get(k);
                for (int j = 0; j < allfreebss.size(); j++) {
                    partner = (BindingSite) allfreebss.get(j);

                    if (! bs.canBindTo(partner))  //check for compatable bindingsiteTypes
			continue;
		    assemBS = (Assembly) assemblies.get(bs.getAssemName());
		    assemPartner = (Assembly) assemblies.get(partner
							     .getAssemName());

		    //below only allows monomers to bind to assemblies
		    /*
		    if (!(((assemBS.getSubunit(bs)).getBoundSubunits()).isEmpty()
			  || ((assemPartner.getSubunit(partner)).getBoundSubunits()).isEmpty())) 
			continue;
		    */

		    //no subunit binding to itself
		    if (bs.getSubunitID() == partner.getSubunitID()) 
			continue;
		    

		    //subs in the same assembly can only bind if bs's
		    // are close to each other
		    if ((bs.getAssemName()).equals(partner
						   .getAssemName())) {
			Subunit sub1 = assemBS.getSubunit(bs);
			Subunit sub2 = assemPartner.getSubunit(partner);

			Vector3d bspos1 = sub1.getBindingSitePos(bs);
			Vector3d bspos2 = sub2
			    .getBindingSitePos(partner);

			Vector3d bsrot1 = sub1.getBSOrientation(bs);
			Vector3d bsrot2 = sub2
			    .getBSOrientation(partner);

			Vector3d bsup1 = bs.getUp();
			Vector3d bsup2 = partner.getUp();

			if (!(withinTranslationalBounds(bspos1, bspos2)
			      && withinAngularBounds(bsrot1, bsrot2) 
			      && withinTorsionalBounds(bsup1, bsup2)))
			    continue;
		    }

		    bTimes = (double[]) (((HashMap) (bondTimes.get(bs.getBST().getName()))).get(partner.getBST().getName()));
		    if ((bs.getAssemName()).equals(partner.getAssemName()))
			duration = getExp(bTimes[2]);
		    else
			duration = getExp(bTimes[0]);

		    endTime = curtime + duration;

		    assemBS = (Assembly) assemblies.get(bs.getAssemName());
		    assemPartner = (Assembly) assemblies.get(partner.getAssemName());

		    //if this is the fastest Event so far, store it
		    assemBS = (Assembly) assemblies.get(bs.getAssemName());
		    if(assemBS.minTime() > endTime && assemPartner.nextEventEndTime() > endTime) {
			assemBS.setMinTime(endTime);
			    
			String[] assemNames = {assemBS.getName(), assemPartner.getName()};
			minTimeEvent = new FormBondEvent(curtime, endTime, assemNames, bs, partner);
		    }

                }//allfreebss loop
            }//freebss loop
	    if(minTimeEvent.getEndTime() != -1.0) {
		String[] asms = minTimeEvent.getAssemNamesInvolved();
		((Assembly) assemblies.get(asms[0])).setNextEvent(minTimeEvent);
		if(asms[1] != null) 
		    ((Assembly) assemblies.get(asms[1])).setNextEvent(minTimeEvent);

		pq.add(minTimeEvent);
		
		//reset minTimes for Assemblies involved
		((Assembly) assemblies.get(asms[0])).setMinTime(Double.MAX_VALUE);
		if(asms[1] != null) 
		    ((Assembly) assemblies.get(asms[1])).setMinTime(Double.MAX_VALUE);
	    }

        }//assembly loop
    }

   
    /**
     * Returns true if a and b are "close" to each other
     * @param a Vector3d
     * @param b Vector3d
     * @return boolean
     */
    private boolean withinTranslationalBounds(Vector3d a, Vector3d b) {
        Vector3d diff = new Vector3d();
        diff.sub(a, b);

        if (diff.length() < 0.01) {//used 0.6 for icos
            return true;
        }
        return false;
    }

    /**
     * Returns true if a and b are reasonably anti-parallel to each other
     * @param a Vector3d
     * @param b Vector3d
     * @return boolean
     */
    private boolean withinAngularBounds(Vector3d a, Vector3d b) {
        double angle = a.angle(b); //in radians 0 to pi

        if (angle > (4.0 * Math.PI / 5.0)) {
            return true;
        }
        return false;
    }

  
    /**
     * Returns true if Vector3d a and  Vector3d b's up vectors are reasonably parallel
     * @param a Vector3d
     * @param b Vector3d
     * @return boolean
     */
    private boolean withinTorsionalBounds(Vector3d a, Vector3d b) {
        double angle = a.angle(b); //within 0 to pi

        if (angle < (Math.PI / 5.0)) {
            return true;
        }
        return false;
    }

    /**
     * Gets the next Event from the priority queue, processes the Event, 
     * relaxes the Assemblies involved and samples new events.
     */
    public void step() {
        //For debugging
        System.out.println("***next iteration of queue, curTime is " + curtime + "**");
        Event ev = (Event) pq.remove();
        double[] validTimes = { -1, -1 };
        String[] assemNames;
        int[] assemIDs = new int[2];
       
        //String Array of size 2
        assemNames = ev.getAssemNamesInvolved();

	//check if one or both Assemblies no longer exist
	Assembly asm1 = (Assembly) assemblies.get(assemNames[0]);
	if(assemNames[1] == null) {
	    //only one Assembly involved and it no longer exists
	    if(asm1==null) {
		System.out.println("invalid event (assembly no longer exists)");
		return;
	    }
	}
	else if(assemNames[1] != null) { //two Assemblies involved
	    Assembly asm2 = (Assembly) assemblies.get(assemNames[1]);

	    //both Assemblies no longer exist
	    if(asm1==null && asm2==null) {
		System.out.println("invalid event (assembly no longer exists)");
		return;
	    }

	    //only asm1 no longer exists
	    if(asm1==null && asm2!=null) {
		System.out.println("invalid event (assembly no longer exists)");

		//sample new events for asm2 before returning
		String[] tmp = { assemNames[1] };
		getNewEvents(tmp);
		return;
	    }

	    //only asm2 no longer exists
	    if(asm1!=null && asm2==null) {
		System.out.println("invalid event (assembly no longer exists)");

		//sample new events for asm1 before returning
		String[] tmp = { assemNames[0] };
		getNewEvents(tmp);
		return;
	    }
	}

	//track validTimes
	validTimes[0] = asm1.validTime();
	if(assemNames[1] != null)
	    validTimes[1] = ((Assembly) assemblies.get(assemNames[1])).validTime();

        /**
         * If event is valid, processes it, generates new events for next
         * iteration. else picks another event (try again)
         */
        if (ev.isValid(validTimes)) {

            processEvent(ev);

            assemNames = ev.getAssemNamesInvolved();

            /**
             * generates new events for queue based on new current state, only
             * need to update nextEvent for assemblies that were involved in
             * "ev".if only one assembly is involved, second element is null.
             */
            getNewEvents(assemNames); 

        } 
	else {
            System.out.println("invalid event");
	    getNewEvents(assemNames);
	    return;
        }

    }

    /**
     * Runs this simulation by calling step() repeatedly until a
     * certain number of events have been processed. User can hardcode his
     * number here and print the intermediate result to files
     */
    public void run() {
	int iterations_per_run = 1000;    //USER SHOULD MODIFY IF DESIRED
	int events_per_printout = 500;    //USER SHOULD MODIFY IF DESIRED
        int i = 0;

        while (!pq.isEmpty() && i < iterations_per_run) {
            this.step();
	    
            //System.out.println(i + " events have been processed.");
            if (i % events_per_printout == 0) {
		//print the number of Subunits in each Assembly to file
                printAssemblies("result_" + i);
            }
            i++;
        }
    }

    /**
     * Puts Assembly a in a HashMap by using its name as the key
     * 
     * @param a
     *            Assembly
     */
    public void addAssembly(Assembly a) {
	assemblies.put(a.getName(), a);
    }

    /*
     * Returns an random number from the exponential distribution with average
     * avg (1/lambda)
     *   
     * @param avg
     *            double
     * @return double
     */
    private double getExp(double avg) {
        if (rand == null) {
            System.out.println("rand is null");
        }
        return -avg * Math.log(rand.nextDouble());
    }

    /**
     * Returns a random double within [0 , 1)
     * 
     * @return double
     */
    public double getRand() {
        return rand.nextDouble();
    }

    /**
     * Returns a String version of this Simulation
     * 
     * @return String
     */
    public String toString() {
        String mystring = new String("*******Simulation: Current Time: "
                + curtime + "*******\n");
        Collection values = assemblies.values();
        for (Iterator i = values.iterator(); i.hasNext(); mystring += ((Assembly) (i
                .next()) + "\n\n")) {

        }

        mystring += "*****end******";
        return mystring;
    }

    /**
     * Used for debugging. Gets each Assembly in this Simulation and calls
     * relaxAssemblyLots().
     */
    public void relaxAssembliesLots() {
        Iterator it = assemblies.values().iterator();
        while (it.hasNext()) {
            Assembly temp = (Assembly) it.next();
            temp.relaxAssemblyLots();
        }
    }

    /**
     * Used for debugging. Gets each Assembly in this Simulation and calls
     * forxeRelaxAssembly() to force_relax them.
     */
    public void forceRelaxAssemblies() {
        Iterator it = assemblies.values().iterator();
        while (it.hasNext()) {
            Assembly temp = (Assembly) it.next();
            temp.forceRelaxAssembly();
        }
    }

    /**
     * Relaxes each Assembly in this Simulation
     */
    public void relaxAssemblies() {
        Iterator it = assemblies.values().iterator();
        while (it.hasNext()) {
            Assembly temp = (Assembly) it.next();
            temp.relaxAssembly();
        }

    }

    /**
     * Returns a Vector of all the Assemblies in this Simulation
     * 
     * @return Vector of Assemblies
     */
    public Vector getAssemblies() {
        Iterator it = assemblies.values().iterator();
        Vector v = new Vector();

        while (it.hasNext()) {
            Assembly temp = (Assembly) it.next();
            v.add(temp);
        }
        return v;
    }

    /**
     * Creates a SimulatorGraphic based on this Simulation and returns it.
     * 
     * @return SimulatorGraphic
     */
    public SimulatorGraphic startGraphic() {
        graphic = new SimulatorGraphic(this);
        return graphic;
    }

    /**
     * Prints the size of each Assembly in current Simulation by Walking through
     * HashMap assemblies and geting the size. The result is saved in string
     * fileName
     * 
     * @param fileName
     *            String the name of file to which the result is saved
     */
    private void printAssemblies(String fileName) {

        //used for debugging
        //      System.out.println("-------------Start of
        // printAssemblies(names)------------");
        Iterator it = assemblies.values().iterator();
        int i = 0;
        int assembliesSize = assemblies.values().size();
        PrintWriter fileOut = null;

        try {
            fileOut = new PrintWriter(new FileWriter(fileName));

            while (i < assembliesSize) {
                i++;
                Assembly tem = (Assembly) it.next();
                fileOut.println(tem.numSubunits());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        if (fileOut != null)
            fileOut.close();
    }

    /**
     * CheckPairSeen use a private Vector of int[2] to hold pairs of Integer,
     * user can add Integer pair by using addPair(int i, int j) and check if
     * some Inter pair has been added by using marked(int i, int j), the order
     * of Inter in any pairs will not affect the returned value.
     */
    private class CheckPairSeen {

        private Vector seen; //vector of double[2], which are the seen pairs

        public CheckPairSeen() {
            seen = new Vector();
        }

        public void addPair(int i, int j) {
            seen.add(new double[] { i, j });
        }

        public boolean marked(int i, int j) {
            double[] tmp;
            for (int k = 0; k < seen.size(); ++k) {
                tmp = (double[]) (seen.get(k));
                if ((tmp[0] == i && tmp[1] == j)
                        || (tmp[0] == j && tmp[1] == i)) {
                    return true;
                }
            }
            return false;
        }

    }

}

