import java.util.Vector;
import java.util.HashMap;
import java.util.Random;
import javax.vecmath.*;

/**
 * Icosahedral sets up a viral capsid-like Simulation, where there is one 
 * type of Subunit with a single Conformation and Domain and three binding 
 * sites arranged at angles of 120, 120, and 109 degrees apart (allowing
 * an icosahedral-shaped superstructure).  
 * The initial number of Subunits, and the binding and breaking times
 * control the assembly of Icosahedral
 * 
 * @author Rori Rohlfs
 * @version 1.0
 */

public class Icosahedral {
    private static Simulation sim;
    private static Solution soln;
    private static Vector assemblies;
    private static int bsid = 0;      //BindingSite ID
    private static int dmid = 0;      //Domain ID
    private static int stid = 0;      //SubunitType ID
    private static int suid = 0;      //Subunit ID

    //binding and breaking rates set here:
    //average time for a binding event between two compatible BindingSites 
    private double bindTime = 2.0;   
    //average time for a breaking event between two compatible BindingSites
    private double breakTime = 200.0;
    //average time for a bonding event between two compatible BindingSites,
    //positioned to bind, in the same Assembly
    private double fastbindTime = 0.01;

    //Initial number of Subunits are set here.
    private static int numSubunits = 81;
  
    /**
     * Constructs an Icosahedral Simulation.  
     */
    public Icosahedral() {
	soln = new Solution();
	initAssembly();

	//now need to use "last" simulation constructor so it works
	double[] bindbreak = new double[3];
	bindbreak[0] = bindTime;
	bindbreak[1] = breakTime;
	bindbreak[2] = fastbindTime;
	double[] bindbreak2 = new double[3];
	bindbreak2[0] = bindTime;
	bindbreak2[1] = breakTime;
	bindbreak2[2] = fastbindTime;
	double[] bindbreak3 = new double[3];
	bindbreak3[0] = bindTime;
	bindbreak3[1] = breakTime;
	bindbreak3[2] = fastbindTime;
	double[] nobindbreak = new double[3];
	nobindbreak[0] = 99999999999999.0;
	nobindbreak[1] = 0.00000000000001;
	nobindbreak[2] = 99999999999999.0;
	


	HashMap bst1hash = new HashMap();
	bst1hash.put(new String("bsta"), bindbreak);
	bst1hash.put(new String("bstb"), nobindbreak);
	bst1hash.put(new String("bstc"), nobindbreak);

	HashMap bst2hash = new HashMap();
	bst2hash.put(new String("bsta"), (double[]) nobindbreak.clone());
	bst2hash.put(new String("bstb"), (double[]) nobindbreak.clone());
	bst2hash.put(new String("bstc"), (double[]) bindbreak3.clone());

	HashMap bst3hash = new HashMap();
	bst3hash.put(new String("bsta"), (double[]) nobindbreak.clone());
	bst3hash.put(new String("bstb"), (double[]) bindbreak2.clone());
	bst3hash.put(new String("bstc"), (double[]) nobindbreak.clone());

	HashMap bondtimes = new HashMap();
	bondtimes.put(new String("bsta"), bst1hash);
	bondtimes.put(new String("bstb"), bst2hash);
	bondtimes.put(new String("bstc"), bst3hash);

	HashMap conftimes = new HashMap();

	sim = new Simulation(soln, assemblies, 0.0, bondtimes, conftimes);
    }

    /**
     * Returns Simulation of Icosahedral 
     * 
     * @return Simulation
     *  
     */
    public static Simulation getSim() {
	return sim;
    }
    
    /**
     * Creates all Subunits with appropriate bsID, domainID, 
     * ConformationID and stID.  Adds all Assemblies to Vector assemblies
     * (which is later used in the Simulation construction)
     */
    private static void initAssembly() {
	assemblies = new Vector();

	//bindingsitetype definitions
        double[] tolerances = {0.2, 0.1, 0.1};
        Vector partnersa = new Vector();   
	Vector partnersb = new Vector();
	Vector partnersc = new Vector();
        partnersa.add("bsta");
        BindingSiteType bsta = new BindingSiteType(tolerances, "bsta", partnersa);
        partnersb.add("bstc");
        BindingSiteType bstb = new BindingSiteType(tolerances, "bstb", partnersb);
	partnersc.add("bstb");
	BindingSiteType bstc = new BindingSiteType(tolerances, "bstc", partnersc);


      	//get rot/pos/up for bs1    
	AxisAngle4d orbs1a = new AxisAngle4d(1.0, 0.0, 0.0, 0.553574);
	Quat4d orbs1 = new Quat4d();
	orbs1.set(orbs1a);

	Vector3d posbs1 = new Vector3d(0.0, 0.850650808*0.08, 0.525731112*0.08);

	Vector3d upbs1 = new Vector3d(0.0, -0.525731, 0.850651);
	
	//get rot/pos/up for bs2 
	AxisAngle4d orbs2a = new AxisAngle4d(0.0, 0.0, 1.0, 126.0*(Math.PI/180.0));
	Quat4d orbs2 = new Quat4d();
	orbs2.set(orbs2a);

	Vector3d posbs2 = new Vector3d(-0.809016994*0.08, -0.587785252*0.08, 0.0);
	
	Vector3d upbs2 = new Vector3d(0.0, 0.0, 1.0);

	//get rot/pos/up for bs3     
	AxisAngle4d orbs3a = new AxisAngle4d(0.0, 0.0, 1.0, -126.0*(Math.PI/180.0));
	Quat4d orbs3 = new Quat4d();
	orbs3.set(orbs3a);

	Vector3d posbs3 = new Vector3d(0.809016994*0.08, -0.587785252*0.08, 0.0);

	Vector3d upbs3 = new Vector3d(0.0, 0.0, 1.0);


	//loop for making subunittypes
	Vector subts = new Vector();
	for(int i=0; i<numSubunits; i++) {
	    String asmname = "assem"+i;
	    BindingSite bsa = new BindingSite(bsta, bsid++, dmid, suid, asmname, posbs1, orbs1, upbs1);
	    BindingSite bsb = new BindingSite(bstb, bsid++, dmid, suid, asmname, posbs2, orbs2, upbs2);
	    BindingSite bsc = new BindingSite(bstc, bsid++, dmid, suid, asmname, posbs3, orbs3, upbs3);
         
	    /**
             * Adds BindingSites of one Subunit to Vector bss to create
             * Conformation
             */
	    Vector bss = new Vector();
	    bss.add(bsa);
	    bss.add(bsb);
	    bss.add(bsc);
	    
	    /**
             * Adds one Conformation of a Subunit to Vector confs to create
             * Domain
             */
	    Vector confs = new Vector(); //vector of conformations for the 1 domain
	    Conformation conf = new Conformation(bss, 6.6, "subt1only");
	    confs.add(conf); 
	    
            /**
             * Adds one Domain of a Subunit to Vector Confsets to create
             * SubunitType
             */
	    Domain dom = new Domain("only", confs, new Vector3d(), dmid);
	    Vector confSets = new Vector();
	    confSets.add(dom);

	    SubunitType subt = new SubunitType(confSets, "only", soln);

	    subts.add(subt);


	    /**
	     * set initial position, rotation, velocity, and rotational velocity
	     */
	    Vector3d sPos = new Vector3d();
	    Quat4d sRot = new Quat4d(0.0,0.0,0.0,1.0);
	    Vector3d sVel = new Vector3d();
	    Quat4d srVel = new Quat4d(0.0,0.0,0.0,1.0);

	    Subunit subunit = new Subunit(subt, confs, soln, suid++, sPos, sRot, sVel, srVel, 3.4, 0.08);

	    Vector asmsubs = new Vector();
	    asmsubs.add(subunit); 
	    Assembly asm = new Assembly(asmsubs, asmname);
	    //System.out.println("a2 " + assembly2.toString());
	    
	    assemblies.add(asm);
	}

	
    }










}
