import java.util.Vector;
import java.util.Random;
import javax.vecmath.*;
import javax.media.j3d.*;
import com.sun.j3d.utils.*;

/**
 * Assembly holds an Assembly of Subunits and the AssemblyGraphic.  It also 
 * tracks the next Event for this Assembly and its validity.  Assemblies can
 * be bonded, split, or relaxed.
 * 
 * @author Rori Rohlfs
 * @author Sue Yi Chew
 * @version 1.0
 *  
 */


public class Assembly {
    //private variables ---------------------------------------
    private String myname; 
    private Vector mysubs;   //Vector of Subunits
    private double k = 5.0;  //arbitrary spring constant, replace this later
    
    private double minTime; //stores minTime of event with minimum-Time seen so far 
    private Event nextEvent; //minimum-time event seen so far
    private double validTime; //the last time new events was sampled for this assembly

    AssemblyGraphic mygraphic;  //holds a graphics tree to all the different SubunitGraphics in this assembly
    Random gen = new Random();  //to produce random numbers

    //constructors ---------------------------------------------
    /**
     * Constructs an default Assembly with no Subunits and no valid Events
     */
    public Assembly() {
	mysubs = new Vector();
        minTime = Double.MAX_VALUE;
        validTime = -1.0;
    }
  
    /**
     * Constructs an Assembly named name with Subunits from s
     * 
     * @param s
     *            Vector, all the Subunits in this Assembly
     * @param name
     *            String, the name of this Assembly
     */
    public Assembly(Vector s, String name) {
	mysubs = new Vector();
	for(int i=0; i<s.size(); i++) {
	    mysubs.add((Subunit) s.elementAt(i));
	}

	myname = name;
        minTime = Double.MAX_VALUE;
        validTime = -1.0;
	
	mygraphic = new AssemblyGraphic(this);
    }

    //private methods --------------------------------------------------

    /**
     * Calculates intra-assembly forces and updates subunit positions 
     * and velocities over a timestep of dt
     * 
     * @param dt
     *            double, time step
     */
    private void recompute(double dt) {
	//vectors to keep track of each subunit's updated pos/rot/v as the subunits are still being traversed
	Vector newposes = new Vector();
	Vector newvs = new Vector();
	Vector newrots = new Vector();
	Vector newrotvs = new Vector();
	
	//divide assemblies into neighboring groups so repelling force doesn't have to be computed for every pair
	NeighborGroups ng = new NeighborGroups(this);
      	
	//go through all subunits to calculate intra-assembly forces on each
	for(int i=0; i<mysubs.size(); i++) {
	    //get current Subunit and relavent information
	    Subunit cursub = (Subunit) mysubs.elementAt(i);
	    Vector3d curvi = new Vector3d(cursub.getV());
	    Vector3d curvf = (Vector3d) curvi.clone(); 
	    Vector3d curposi = cursub.getPos();
	    Quat4d currvi = cursub.getRotV();
	    Quat4d currvf = (Quat4d) currvi.clone();
	    Quat4d curroti = cursub.getRot();
	    Quat4d currotf = (Quat4d) curroti.clone();
	    double mass = cursub.getMass();
	    double radius = cursub.getRadius();

	    //caculate forces between this subunit and each other near-by subunit
	    Vector myneighbors = ng.getNeighborGroups(cursub);

	    for(int j=0; j<myneighbors.size(); j++) {
		Subunit othersub = (Subunit) myneighbors.elementAt(j);
		
		//check to make sure cursub and othersub aren't the same
		if(cursub.getid() == othersub.getid()) 
		    continue;

		Vector3d otherposi = othersub.getPos();
		    
		//get vector between the centers of the subunits
		Vector3d dx = new Vector3d();
		dx.sub(otherposi, curposi);
		double r = dx.length();
		Vector3d olddx = (Vector3d) dx.clone(); //to be used later after dx is modified
		    
		//calculate repelling force and  the change it causes in velocity 
		//update the final velocity accordingly
		Vector3d repelling_force = (Vector3d) dx.clone();
		repelling_force.scale(-0.05/(r*r*r));
		Vector3d repel_delta_v = (Vector3d) repelling_force.clone();
		repel_delta_v.scale(dt/mass);
		curvf.add(repel_delta_v);
		
		//if othersub is bound to cursub, spring/torsion forces should be accounted for
		if(othersub.isBoundSubunit(cursub)) { 
		    //go through all bindingsite pairs bound between cur and othersub
		    Vector bss = cursub.getConnectingBSs(othersub);
		    for(int l=0; l<bss.size(); l+=2) {
			BindingSite curbs = (BindingSite) bss.get(l);
			BindingSite boundbs = (BindingSite) bss.get(l+1);
			    
			//get binding site vectors
			Vector3d curbsv = cursub.getBSOrientation(curbs);
			Vector3d boundbsv = othersub.getBSOrientation(boundbs);
			   			    
			//update dx to be between the binding site tips
			dx.sub(curbsv);
			dx.add(boundbsv);
			r = dx.length();
		
			//calculate translational force and  the change it causes in velocity 
			//update the final velocity accordingly
			Vector3d spring_force = (Vector3d) dx.clone(); 
			spring_force.scale(k);
			Vector3d spring_delta_v = (Vector3d) spring_force.clone();
			spring_delta_v.scale(dt/mass);
			curvf.add(spring_delta_v);
			   
			//calculate rotational (anti-paralleling) force and  the change 
			//it causes in rotatinal velocity. 
			Vector3d torque = new Vector3d();
			torque.cross(curbsv, spring_force);
			Vector3d delta_L = (Vector3d) torque.clone();
			delta_L.scale(dt);
			Vector3d delta_rotv = (Vector3d) delta_L.clone();
			double moment_inertia = (2.0/5.0)*mass*Math.pow(cursub.getRadius(), 2);
			delta_rotv.scale(1.0/moment_inertia); 
			
			//update the final rotational velocity by the delta_rotv found above
			double theata = delta_rotv.length();  
			if(theata != 0.0) { //check for no change in rotation
			    delta_rotv.normalize();
			    AxisAngle4d transaa = new AxisAngle4d(delta_rotv, theata);
				
			    Quat4d delta_rotvq = new Quat4d(delta_rotv.x*Math.sin(theata/2.0),
							    delta_rotv.y*Math.sin(theata/2.0),
							    delta_rotv.z*Math.sin(theata/2.0),
							    Math.cos(theata/2.0));
							
			    //update currvf by multiplying with the new delta rotation quaternion
			    currvf.mul(delta_rotvq);
			}

			//get information relevant to torsional forces
			Vector3d curup = cursub.getBSUptoAsm(curbs);
			Vector3d boundup = othersub.getBSUptoAsm(boundbs); 
			Vector3d normcurori = new Vector3d(curbsv);
			normcurori.normalize();
			
			//calculate scaling factor for force    
			Vector3d scalev = new Vector3d();
			scalev.cross(curup, boundup);
			double scale = scalev.dot(curbsv);

			//calculate torsional torque, L, and change in rotational velocity
			Vector3d tortorque = new Vector3d(normcurori);
			tortorque.scale(scale);//note, should also scale by tortional k constant.  
			Vector3d tordelta_L = (Vector3d) tortorque.clone();
			tordelta_L.scale(dt);
			Vector3d tordelta_rotv = (Vector3d) tordelta_L.clone();
			double tormoment_inertia = (2.0/5.0)*mass*Math.pow(cursub.getRadius(), 2);
			tordelta_rotv.scale(1.0/moment_inertia); 

			//apply rotational velocity change to Subunit
			double tortheata = tordelta_rotv.length();  
			if(tortheata != 0.0) { //check for no change in rotation
			    tordelta_rotv.normalize();
			    AxisAngle4d transaa = new AxisAngle4d(tordelta_rotv, tortheata);
				
			    Quat4d delta_rotvq = new Quat4d(tordelta_rotv.x*Math.sin(tortheata/2.0),
							    tordelta_rotv.y*Math.sin(tortheata/2.0),
							    tordelta_rotv.z*Math.sin(tortheata/2.0),
							    Math.cos(tortheata/2.0));
							
			    //update currvf by multiplying with the new delta rotation quaternion
			    currvf.mul(delta_rotvq);
			}
		    }//end for all connecting bss		
		}//end if bound
	    }//end neighbors forloop
	    
	    //apply damping to translational velocity
	    curvf.scale(Math.exp(-2.0*dt*Math.pow(mass, -1.0/3.0)));
	    
	    //apply damping to orientational velocity
	    double moment_inertia = (2.0/5.0)*mass*Math.pow(cursub.getRadius(), 2.0);
	    AxisAngle4d tempaa = new AxisAngle4d();
	    tempaa.set(currvf); 
	    double[] tempd = new double[4];
	    tempaa.get(tempd);
	    tempd[3] *= Math.exp(-3.0*dt*Math.pow(moment_inertia, -1.0/3.0));
	    tempaa.set(tempd);
	    currvf.set(tempaa);

	    //update position
	    Vector3d curxf = new Vector3d();
	    curxf.scale(dt, curvf);
	    curxf.add(curposi);	    
	    
	    //update orientation
	    AxisAngle4d delta_rotaa = new AxisAngle4d();
	    delta_rotaa.set(currvf); 
	    double[] delta_rot_doubles = new double[4];
	    delta_rotaa.get(delta_rot_doubles);
	    delta_rot_doubles[3] *= dt;
	    delta_rotaa.set(delta_rot_doubles);
		
	    Quat4d delta_rot_quat = new Quat4d(); 
	    delta_rot_quat.set(delta_rotaa);
	    currotf.mul(delta_rot_quat);
	    	    
	    //store new positions and velocities (but don't replace yet)
	    newposes.add(curxf);
	    newvs.add(curvf);
	    newrots.add(currotf);
	    newrotvs.add(currvf);
	}//end cursub forloop
	
	//now actually update pos/velocity for all subunits
	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.elementAt(i);
	
	    cursub.setPos((Vector3d) newposes.get(i));
	    cursub.setV((Vector3d) newvs.get(i));
	    cursub.setRot((Quat4d) newrots.get(i));
	    cursub.setRotV((Quat4d) newrotvs.get(i));
	}
    }



    /**
     * Returns true if the average velocity of Subunits in the Assembly is low
     * (assembly is in a low-energy state)
     * 
     * @return boolean
     */
    private boolean isStable() {
	double vsum = 0.0;
	double rvsum = 0.0;

	//get total velocities
	for(int i=0; i<mysubs.size(); i++) {
	    vsum += Math.pow((((Subunit) mysubs.get(i)).getV()).length(), 2.0);
	    
	    double w = (((Subunit) mysubs.get(i)).getRotV()).w;
	    rvsum += Math.pow(2.0*Math.acos(w), 2.0);
	}

	//calculate average velocities
	double vavg = Math.pow(vsum, 0.5);
	double rvavg = Math.pow(rvsum, 0.5);

	//this is arbitrary, should make better limits at some point
	if(vavg < .03 && rvavg < .03) { 
	    return true;
	}
	return false;
    }

    /**
     * Returns true if the average velocity of Subunits in the Assembly is VERY low
     * (assembly is in a VERY low-energy state)
     * used mostly for debugging and trials
     * 
     * @return boolean
     */
    private boolean isStableLots() {
	double vsum = 0.0;
	double rvsum = 0.0;

	//get total velocities
	for(int i=0; i<mysubs.size(); i++) {
	    vsum += Math.pow((((Subunit) mysubs.get(i)).getV()).length(), 2.0);
	    
	    double w = (((Subunit) mysubs.get(i)).getRotV()).w;
	    rvsum += Math.pow(2.0*Math.acos(w), 2.0);
	}

	//calculate average velocities
	double vavg = Math.pow(vsum, 0.5);
	double rvavg = Math.pow(rvsum, 0.5);

	//this is arbitrary, should make better limits at some point
	if(vavg < .0005 && rvavg < .0005) { 
	    return true;
	}
	return false;
    }

 
    /**
     * returns true if Vector3d's a and b are anti-parallel, or close to it
     * 
     * @param a
     *            Vector3d
     * @param b
     *            Vector3d
     *
     * @return boolean
     */
    private boolean antiparallel(Vector3d a, Vector3d b) {
	double angle = a.angle(b);

	if(angle > Math.PI - 0.01) 
	    return true;
	return false;
    }

    /**
     * returns true if Vector3d's a and b are parallel, or close to it
     * 
     * @param a
     *            Vector3d
     * @param b
     *            Vector3d
     *
     * @return boolean
     */
    private boolean parallel(Vector3d a, Vector3d b) {
	double angle = a.angle(b);

	if(angle < 0.01) 
	    return true;
	return false;
    }

    /**
     * given a rotation (Quat4d), returns a rotation around the same axis, 
     * but in the opposite direction (another Quat4d)
     * 
     * @param forwards
     *            Quat4d
     *
     * @return Quat4d
     */
    private Quat4d oppQuatRot(Quat4d forwards) {
	double[] rotd = new double[4];
	forwards.get(rotd);
	rotd[0] *= -1.0;
	rotd[1] *= -1.0;
	rotd[2] *= -1.0;

	Quat4d opprotq = new Quat4d(rotd);
	return opprotq;
    }

    /**
     * translates the subunits in this assembly so that the assembly's 
     * <0, 0, 0> is the center of sub.  (the assembly is centered 
     * around sub)
     * 
     * @param sub
     *            Subunit
     */
     private void centerAssembly(Subunit sub) {
	//get translation need for each subunit to recenter assembly
	Transform3D asmtosub = sub.getTransform();
	Quat4d asmtosubq = new Quat4d();
	Vector3d asmtosubv = new Vector3d();
	asmtosub.get(asmtosubq, asmtosubv);
	
	//get translation from sub to the assembly
	Vector3d subtoasmv = new Vector3d(asmtosubv);
	subtoasmv.scale(-1.0);

	//translate every subunit
	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.get(i);
	    cursub.translate(subtoasmv);
	}
    }

    /**
     * Returns a unit vector rotated by Quat4d q
     * 
     * @param q
     *            Quat4d
     *
     * @return Vector3d
     */
    private Vector3d quatToVector(Quat4d q) {
	Quat4d vq = new Quat4d(q);
	vq.mul(new Quat4d(0.0, 1.0, 0.0, 0.0));
	vq.mulInverse(q);
	
	double[] vd = new double[4];
	vq.get(vd);
	Vector3d v = new Vector3d(vd[0], vd[1], vd[2]);

	return v;
    }

    /**
     * returns true if there is steric hindrance in this assembly (ie: 
     * at least two subunits are overlapping)
     *
     * @return boolean
     */
    private boolean stericHindrance() {
	for(int i=0; i<mysubs.size(); i++) {	
	    Subunit sub1 = (Subunit) mysubs.get(i);
	    
	    //divide assemblies into neighboring groups
	    NeighborGroups ng = new NeighborGroups(this);

	    Vector myneighbors = ng.getNeighborGroups(sub1);
	    for(int j=0; j<myneighbors.size(); j++) {
		Subunit sub2 = (Subunit) myneighbors.get(j);
		if(sub1.equals(sub2))
		    continue;

		Vector3d pos1 = sub1.getPos();
		Vector3d pos2 = sub2.getPos();
		
		Vector3d diff = new Vector3d();
		diff.sub(pos1, pos2);
		double distance = diff.length();

		if(distance < sub1.getRadius() + sub2.getRadius())
		    return true;
	    }
	}
	return false;
    }

    /**
     * Rotates bindasm so that the bs's indicated in new and oldasmtobsq
     * are anti-parallel (pointing at each other)
     *
     * @param bindasm
     *            Assembly, the Assembly to be rotated
     * @param newasmtobsq
     *            Quat4d, rotation from the bindasm to its bs in question
     * @param oldasmtobsq
     *            Quat4d, rotation from the this Assembly to its bs in question
     * @param dxv
     *            Vector3d, vector between the two bs's
     * @param oldup
     *            the up vector of the BindingSite in question on this Assembly
     */
    private void arrangeBSs(Assembly bindasm, Quat4d newasmtobsq, Quat4d oldasmtobsq, 
			    Vector3d dxv, Vector3d oldup) {
	//both assemblies start with their "ups" pointing Up

	//line up newbs and oldasm's "up"
	Quat4d newbstoasmq = oppQuatRot(newasmtobsq);
	bindasm.rotate(newbstoasmq);  

	//line up bs's
	bindasm.rotate(oldasmtobsq);

	//make axisangle rot about old up vector (which is perpendicular to bs's) for pi radians
	AxisAngle4d flipnewasma = new AxisAngle4d(oldup, Math.PI);
	Quat4d flipnewasmq = new Quat4d();
	flipnewasmq.set(flipnewasma);
	
	bindasm.rotate(flipnewasmq);
    }

    /**
     * Rotates bindasm so that the up's of the bs's indicated in 
     * new and oldasmtobsq are parallel 
     *
     * @param bindasm
     *            Assembly, the Assembly to be rotated
     * @param oldup
     *            Vector3d, up for the bs in question on this Assembly 
     *            (given in relation to the real axis, not the subunit)
     * @param newup
     *            Vector3d, up for the bs in question on bindasm
     *            (given in relation to the real axis, not the subunit)
     * @param dxv
     *            Vector3d, vector between the two bs's
     * @param newasmtobsv
     *            Vector3d, vector from bindasm to the bs in question
     */
    private void arrangeUps(Assembly bindasm, Vector3d oldup, Vector3d newup, Vector3d dxv, Vector3d newasmtobsv) {
	//if up's are anti-parallel, crossprod is 0, so it's backwards.  thus: check
	if(antiparallel(oldup, newup)) { //bs's are anti parallel, so rot pi around newbs
	    AxisAngle4d uptoup_rota = new AxisAngle4d(newasmtobsv, Math.PI);
	    Quat4d uptoup_rotq = new Quat4d();
	    uptoup_rotq.set(uptoup_rota);
	    
	    bindasm.rotate(uptoup_rotq);
	}		
	else if(!parallel(oldup, newup)) { //up's aren't antiparallel or parallel
	    double ang = newup.angle(oldup);
	    Vector3d crossprod = new Vector3d();
	    crossprod.cross(newup, oldup);
	    crossprod.normalize();
	
	    AxisAngle4d uptoup_rota = new AxisAngle4d(crossprod, ang);//i think that's ok...
	    Quat4d uptoup_rotq = new Quat4d();
	    uptoup_rotq.set(uptoup_rota);

	    bindasm.rotate(uptoup_rotq);
	}
    }

    //public methods --------------------------------------------------
    /**
     * Returns the Subunit in this Assembly that holds bs, if that's not found
     * returns a new blank Subunit
     * 
     * @param bs
     *            BindingSite
     *
     * @return Subunit
     */
    public Subunit getSubunit(BindingSite bs) {
	int subid = bs.getSubunitID();

	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.get(i); 

	    Vector bss = cursub.getBindSites();
	    for(int j=0; j<bss.size(); j++) {
		if(((BindingSite) bss.get(j)).equals(bs))
		    return cursub;
	    }
	}
	return new Subunit();
    }

    /**
     * Actually bind Subunit newsub to this Assembly through a bond between newbs and mybs.  
     * (ie: change id's, names, and binding information appropriately)
     * Subunit could be bound directly to this Assembly, bound indirectly as a 
     * remote part of the new Assembly, or part of the this Assembly binding to itself 
     * (making a loop).  
     *
     * @param newsub
     *            Subunit, the Subunit that is being added to this Assembly
     * @param newbs
     *            BindingSite, the BindingSite on the new Assembly that forms the Assembly-linking bond
     * @param mybs
     *            BindingSite, the BindingSite on the this Assembly that forms the Assembly-linking bond
     * @param loop
     *            boolean, true if bond forms a loop in a single Assembly, 
     *            instead of linking two distinct Assemblies
     *
     * @return Subunit
     */
    public void addSubunit(Subunit newsub, BindingSite newbs, BindingSite mybs, boolean loop) {
	//if newsub contains newbs, actually bind on bs and subunit level
	if(newsub.getid() == newbs.getSubunitID()) {
	    //bind on bindingsite level
	    Debug.checkFalse(newbs.bindTo(mybs), "addSubunit bind in Assembly.java didn't work");
	    
	    //should check for conformation changes here?, will that be taken care of in the event queue?
	    
	    //bind bindingsubunit on subunit level
	    Subunit mysub = getSubunit(mybs);
	    mysub.addPartner(newsub);
	    newsub.addPartner(mysub);
	}
	
	//if it's two new assemblies binding, update all newsub's bindinsite assembly names
	//and add it to mysubs
	if(!loop) {
	    Vector tmpbs = newsub.getBindSites();
	    for(int i=0; i<tmpbs.size(); i++) {
		((BindingSite)tmpbs.get(i)).setAssemName(myname);
	    }	   
	    mysubs.add(newsub);
	}

	//add the subunit to the graphic (even if it's a loop, the graphic was detached, so must be reattached)
	mygraphic.addSubunit(newsub);
    }

    /**
     * Used to split this Assembly when it encounters steric hindrance.  Returns
     * the newly split-off Assembly
     *
     * @param newasmname
     *            String, name of the Assembly to be split off
     * @param newbs
     *            BindingSite, the BindingSite on the new Assembly that forms the Assembly-linking bond
     * @param oldsub
     *            Subunit, the Subunit on this Assembly that holds the BindingSite whose bond will break
     * @param newsub
     *            Subunit, the Subunit on this Assembly that holds newbs
     *
     * @return Assembly
     */
    private Assembly reSplitAssembly(String newasmname, BindingSite newbs, Subunit oldsub, Subunit newsub) {
	//break the bond (on BindingSite level)
	Debug.checkFalse(newbs.breakBond(), "processEvent: breakBond() was unsuccessful in Assembly.java: reSplitAssembly");
	
	//break the bond (on Subunit level)
	Debug.checkFalse(oldsub.removePartner(newsub), 
			 "tried to removePartner on Subunit that isn't a partner on oldsub in Assembly.java: reSplitAssembly");
	Debug.checkFalse(newsub.removePartner(oldsub), 
			 "tried to removePartner on Subunit that isn't a partner on newsub in Assembly.java: reSplitAssembly");

	//old assembly (this assembly) -- has all subunits connected to oldsub (including oldsub)
	CheckConnected cc = new CheckConnected();   
	Vector keepsubunits = cc.getConnected(oldsub);
	    
	//new assembly
	CheckConnected cc2 = new CheckConnected();
	Vector detachsubunits = cc2.getConnected(newsub);
	    
	//detach each SubunitGraphic and update BindingSite Assembly names 
	Vector tmpbs;
	for(int j=0; j<detachsubunits.size(); j++) {  
	    Subunit cursub = (Subunit) detachsubunits.get(j);
	    mysubs.remove(cursub);
	    cursub.detachGraphic();
	    tmpbs = cursub.getBindSites();
	    for(int i=0; i<tmpbs.size(); i++) {
		((BindingSite)tmpbs.get(i)).setAssemName(newasmname);
	    }
	}

	//actually make the new detached Assembly
	Assembly detachassembly = new Assembly(detachsubunits, newasmname);
	
	//relax the assemblies
	this.relaxAssembly();
	detachassembly.relaxAssembly();    
	
	return detachassembly;
    }

    /**
     * Used to split this Assembly when indicated by a BreakBondEvent.  Returns the newly
     * split off Assembly, or null if the split didn't make a new Assembly (rather, 
     * it broke a loop)
     *
     * @param e
     *            BreakBondEvent, the event that causes the split
     * @param newasmnum
     *            int, the number of the Assembly to be split off
     * @param bsSub1
     *            Subunit, the Subunit on this Assembly that holds the BindingSite whose bond will break
     * @param bsSub2
     *            Subunit, the Subunit on this Assembly that holds newbs
     *
     * @return Assembly
     */
    public Assembly splitAssembly(BreakBondEvent e, int newasmnum) {
	BindingSite bs1 = e.getBS();
	BindingSite bs2 = e.getPartner();
	Subunit bsSub1 = getSubunit(bs1);
	Subunit bsSub2 = getSubunit(bs2);

	//break the bond (on BindingSite level)
	Debug.checkFalse(bs1.breakBond(), "processEvent: breakBond() was unsuccessful in Assembly.java: splitAssembly");
	
	//break the bond (on Subunit level)
	Debug.checkFalse(bsSub1.removePartner(bsSub2), 
			 "tried to removePartner on Subunit that isn't a partner with bsSub1 in Assembly.java: splitAssembly");
	Debug.checkFalse(bsSub2.removePartner(bsSub1), 
			 "tried to removePartner on Subunit that isn't a partner with bsSub2 in Assembly.java: splitAssembly");

	//if bond break actually made 2 disconnected assemblies, separate them
	CheckConnected cc = new CheckConnected();   
	if(!(cc.isConnected(bsSub1, bsSub2))) {
	    //first assembly (this assembly) -- has all subunits connected to bsSub1 including bsSub1
	    Vector keepsubunits = cc.getConnected(bsSub1);
	    
	    //second assembly
	    CheckConnected cc2 = new CheckConnected();
	    Vector detachsubunits = cc2.getConnected(bsSub2);

	    //detach each SubunitGraphic and update BindingSite Assembly names 
	    Vector tmpbs;
	    for(int j=0; j<detachsubunits.size(); j++) {  
		Subunit cursub = (Subunit) detachsubunits.get(j);
		mysubs.remove(cursub);
		cursub.detachGraphic();
		tmpbs = cursub.getBindSites();
		for(int i=0; i<tmpbs.size(); i++) {
		    ((BindingSite)tmpbs.get(i)).setAssemName("Assembly#" + (newasmnum));
		}
	    }         

	    //make the new Assembly
	    Assembly detachassembly = new Assembly(detachsubunits, "Assembly#" + (newasmnum));

	    //relax the assemblies
	    this.relaxAssembly();
	    detachassembly.relaxAssembly();    
	    
	    return detachassembly;
	}

	//if bond break didn't make 2 assemblies, return null
	return null;
    }

    /**
     * Carries out the FormBondEvent e by binding bindasm to this through a bond between 
     * bsSub1 in this Assembly and bsSub2 in bindasm.  If the binding can not occur
     * (because of steric hindrance), returns a modified version of bindasm, otherwise
     * returns null. 
     *
     * @param bindasm
     *            Assembly, assembly to be bound
     * @param e
     *            FormBondEvent, the event that causes the binding
     * @param bsSub1
     *            Subunit, the Subunit on this Assembly that holds the BindingSite which forms the bond
     * @param bsSub2
     *            Subunit, the Subunit on bindasm that holds the BindingSite which forms the bond
     *
     * @return Assembly
     */
    public Assembly bindAssemblies(Assembly bindasm, FormBondEvent e, Subunit bsSub1, Subunit bsSub2) {
	//note: oldasm = this    newasm = bindasm
	BindingSite oldbs = e.getBS();
	BindingSite newbs = e.getPartner();

	// if the subunits came from different assemblies, "add" the second assembly to the first
        if(! e.getBS().getAssemName().equals(e.getPartner().getAssemName())) {
	    //center assemblies around subunits in question
	    bindasm.centerAssembly(bsSub2);
	    this.centerAssembly(bsSub1);

	    //find oldasm pos/rot
	    Vector3d oldasmposv = getPos();
	    Quat4d oldasmrotq = getRot();

	    //find newasm name
	    String newasmname = bindasm.getName();

	    //find newasmtosub pos
	    Vector3d newasmtosubv = bsSub2.getPos();
	    Vector3d newsubtoasmv = new Vector3d(newasmtosubv);
	    newsubtoasmv.scale(-1.0);
	    
	    //find asmtobs rots/poss for both
	    Transform3D oldasmtobst = bsSub1.getBStoAsmTransform(oldbs);
	    Quat4d oldasmtobsq = new Quat4d();
	    Vector3d oldasmtobsv = new Vector3d();//this is aligned w/ binding site
	    oldasmtobst.get(oldasmtobsq, oldasmtobsv);

	    Transform3D newasmtobst = bsSub2.getBStoAsmTransform(newbs);
	    Quat4d newasmtobsq = new Quat4d();
	    Vector3d newasmtobsv = new Vector3d();
	    newasmtobst.get(newasmtobsq, newasmtobsv);

	    //find newasm transform
	    Transform3D newasmt = bindasm.getTransform();
	    Vector3d newasmpos = new Vector3d();
	    Quat4d newasmrot = new Quat4d();
	    newasmt.get(newasmrot, newasmpos);
	    
 	    //find vector between the 2 binding sites
	    Vector3d dxv = new Vector3d(0.0, 1.0, 0.0);
	    oldasmtobst.transform(dxv);
	    dxv.normalize();
	    dxv.scale(0.3);
	    
	    //translate newassembly to the origin for rotating stuff
	    bindasm.updateLocation(new Quat4d(0.0, 0.0, 0.0, 1.0), new Vector3d(0.0, 0.0, 0.0));
	    this.updateLocation(new Quat4d(0.0, 0.0, 0.0, 1.0), new Vector3d(0.0, 0.0, 0.0));
	    
	    //get upvectors for each bindingsite
	    Vector3d newup = bindasm.getSubBSUp(bsSub2, newbs);	 //this is relative to newasm
	    Vector3d oldup = this.getSubBSUp(bsSub1, oldbs);  //this is relative to oldasm
	    
	    //rotate appropriately to line up bs's
	    arrangeBSs(bindasm, newasmtobsq, oldasmtobsq, dxv, oldup);
	    
	    Vector3d newupreXYZ = bindasm.getSubBSUpReAXIS(bsSub2, newbs);	 //this is relative to XYZ axis
	    Vector3d oldupreXYZ = this.getSubBSUpReAXIS(bsSub1, oldbs);  //this is relative to XYZ axis
	    
	    //now to line up bs 'up' vectors
	    arrangeUps(bindasm, oldupreXYZ, newupreXYZ, dxv, newasmtobsv);
	    
	    //right pos for newasm will be oldasm (origin) to oldbs + dx + newbstoasm
	    Vector3d oldasmtonewasm = new Vector3d(oldasmtobsv);
	    oldasmtonewasm.add(dxv);
	    
	    //this little run around instead of just using newasmtobs because it's relative to newasm, not oldasm
	    double newbsl = newasmtobsv.length();
	    Vector3d newbstooldasmv = new Vector3d(oldasmtonewasm);
	    newbstooldasmv.normalize();
	    newbstooldasmv.scale(newbsl);
	    oldasmtonewasm.add(newbstooldasmv);
	    
	    //now actually move the newasm into place
	    bindasm.setPos(oldasmtonewasm);

	    //compound bindasm so the subunits hold the values for pos/rot in relation to newasm
	    bindasm.compoundTransforms();
	    
	    //go through bindasm's subs binding each to this asm
	    Vector bindsubs = bindasm.getSubunits();
	    for(int i=0; i<bindsubs.size(); i++) {
		Subunit cursub = (Subunit) bindsubs.get(i);
		
		this.addSubunit(cursub, newbs, oldbs, false);
	    }
	
	    //now rotate the whole thing back to original orienation
	    rotate(oldasmrotq);
	    
	    //now check for steric problems
	    if(this.stericHindrance()) {
		Assembly retasm = this.reSplitAssembly(newasmname, newbs, bsSub1, bsSub2);
		return retasm;
	    }
	    else {
		//return this combind assembly to original position
		this.updateLocation(getRot(), oldasmposv);
		this.relaxAssembly();
		return null;
	    }
	} //if subs from diff asm
	else { //else the 2 bindingsites were already on the same assembly
	    //actually make the new bond
	    Vector bindsubs = bindasm.getSubunits();
	    for(int i=0; i<bindsubs.size(); i++) {
		Subunit cursub = (Subunit) bindsubs.get(i);

		this.addSubunit(cursub, newbs, oldbs, true);
	    } 
	    this.relaxAssembly();

	    return null;
	}
    }

    /**
     * Composes Assembly and Subunit transforms so that the Subunits hold all 
     * the significant location information and the Assembly location information
     * is all the identity function
     */
    public void compoundTransforms() {
	//get assembly transform
	Transform3D asmtr = mygraphic.getTransform();


	//go through all subunits, composing transform
	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.get(i);
	    Transform3D subtr = cursub.getTransform();

	    Transform3D product = new Transform3D();
	    product.mul(asmtr, subtr);

	    cursub.setTrans(product);
	}

	//set assembly transform to identity
	updateLocation(new Quat4d(0.0,0.0,0.0,1.0), new Vector3d(0.0,0.0,0.0));
    }

    /**
     * Returns the up vector for bs in relation to the real axis system
     *
     * @param sub
     *            Subunit, holds bs
     * @param bs
     *            BindingSite, bs in question
     *
     * @return Vector3d
     */
    public Vector3d getSubBSUpReAXIS(Subunit sub, BindingSite bs) {
	Vector3d upreasm = sub.getBSUptoAsm(bs);
	Quat4d asmrot = mygraphic.getRot();
	
	return rotate(upreasm, asmrot);
    }

    /**
     * Returns the Vector3d v rotated by Quat4d q
     *
     * @param v
     *            Vector3d
     * @param q
     *            Quat4d
     *
     * @return Vector3d
     */
    private Vector3d rotate(Vector3d v, Quat4d q) {
	double[] vd = new double[3];
	v.get(vd);
	Quat4d vq = new Quat4d(vd[0], vd[1], vd[2], 0.0);

	Quat4d rotvq = new Quat4d();
	rotvq.mul(q, vq);
	rotvq.mulInverse(q);

	double[] rotvd = new double[4];
	rotvq.get(rotvd);
	Vector3d rotv = new Vector3d(rotvd[0], rotvd[1], rotvd[2]);
	return rotv;
    }

    /**
     * Returns the up vector for bs in relation to the assembly
     *
     * @param sub
     *            Subunit, holds bs
     * @param bs
     *            BindingSite
     *
     * @return Vector3d
     */
    public Vector3d getSubBSUp(Subunit sub, BindingSite bs) {
	return sub.getBSUptoAsm(bs);
    }

    /**
     * Rotates this Assembly by rot
     *
     * @param rot
     *            Quat4d
     */
    //rotates this assembly by rot
    public void rotate(Quat4d rot) {
	mygraphic.rotate(rot);
    }

    /**
     * Returns the position of the binding site relative to the real axis system
     *
     * @param sub
     *            Subunit, holds bs
     * @param bs
     *            BindingSite
     *
     * @return Vector3d
     */
    public Vector3d getBindingSitePos(Subunit sub, BindingSite bs) {
	return mygraphic.getBindingSitePos(sub, bs);	
    }

    /**
     * Returns the rotation of the binding site relative to the real axis system
     *
     * @param sub
     *            Subunit, holds bs
     * @param bs
     *            BindingSite
     *
     * @return Quat4d
     */
    public Quat4d getBindingSiteRot(Subunit sub, BindingSite bs) {
	return mygraphic.getBindingSiteRot(sub, bs);	
    }

    /**
     * Relaxes this Assembly into a low-energy state
     */
    public void relaxAssembly() {
	double dt = 0.10;
	int j=0;
	double ddt = 0.1;

	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.elementAt(i);
	    cursub.setV(new Vector3d(0.0, 0.0, 0.0));
	    cursub.setRotV(new Quat4d(0.0, 0.0, 0.0, 1.0));
	}

	//for some accuracy, divide dt into 10 equal sections and update times
	do {
	    recompute((dt/ddt));
	    j++;
	    ddt += 0.01;
	} while(!isStable() && j<100000);

    }

    /**
     * Relaxes this Assembly into a VERY low-energy state
     * (used mostly for debugging and trials)
     */
    public void relaxAssemblyLots() {
	double dt = 0.10;
	int j=0;
	double ddt = 0.1;

	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.elementAt(i);
	    cursub.setV(new Vector3d(0.0, 0.0, 0.0));
	    cursub.setRotV(new Quat4d(0.0, 0.0, 0.0, 1.0));
	}

	//for some accuracy, divide dt into 10 equal sections and update times
	for(int i=0; i<400; i++) {
	    recompute(dt);
	}
	do {
	    recompute((dt/ddt));
	    j++;
	    ddt += 0.01;
	} while((!isStableLots()) && j<100000);
    }

    /**
     * Forces the Assembly to relax a number of times, 
     * even if it's in a low-energy state to start with
     * (used mostly for debugging and trials)
     */
    public void forceRelaxAssembly() {
	double dt = 0.10;

	for(int i=0; i<mysubs.size(); i++) {
	    Subunit cursub = (Subunit) mysubs.elementAt(i);
	    cursub.setV(new Vector3d(0.0, 0.0, 0.0)); 
	    cursub.setRotV(new Quat4d(0.0, 0.0, 0.0, 1.0));
	}

	//for some accuracy, divide dt into 10 equal sections and update times
	for(int i=0; i<2009; i++) {
	    recompute(dt/10.0);
	} 
    }

    /**
     * Returns this Assembly's AssemblyGraphic
     *
     * @return AssemblyGraphic
     */
    public AssemblyGraphic getAssemblyGraphic() {
	return mygraphic;
    }

    /**
     * Returns this Assembly's transform
     *
     * @return Transform3D
     */
    public Transform3D getTransform() {
	return mygraphic.getTransform();
    }

    /**
     * Returns this Assembly's rotation
     *
     * @return Quat4d
     */
    public Quat4d getRot() {
	return mygraphic.getRot();
    }
  
    /**
     * Returns this Assembly's position
     *
     * @return Vector3d
     */
    public Vector3d getPos() {
	return mygraphic.getPos();
    }
    
    /**
     * Returns this Assembly's name
     *
     * @return String
     */
    public String getName() {
        return myname;
    }

    /**
     * Sets this Assembly's name to name
     *
     * @param name
     *            String
     */
    public void setName(String name) {
	myname = name;
    }  

    /**
     * Returns a Vector of this Assembly's Subunits
     *
     * @return Vector
     */
    public Vector getSubunits() {
	return mysubs;
    }
    
    /**
     * Sets this Assembly's rotation to q
     *
     * @param q
     *            Quat4d
     */
    public void setRot(Quat4d q) {
	mygraphic.setRot(q);
    }
    
    /**
     * Sets this Assembly's position to v
     *
     * @param v
     *            Vector3d
     */
    public void setPos(Vector3d v) {
	mygraphic.setPos(v);
    }
    
    /**
     * Sets this Assembly's position to v and rotation to q
     *     
     * @param q
     *            Quat4d
     * @param v
     *            Vector3d
     */
    public void updateLocation(Quat4d q, Vector3d v) {
	mygraphic.updateLocation(q, v);
    }
    
    /**
     * Sets this Assembly's transform to tr
     *
     * @param tr
     *            Transform3D
     */
    public void updateLocation(Transform3D tr) {
	mygraphic.updateLocation(tr);
    }

    /**
     * Returns true if s is in this Assembly
     *
     * @param s
     *            Subunit
     *
     * @return boolean
     */
    public boolean containsSubunit(Subunit s) {
	for(int i=0; i<mysubs.size(); i++) {
	    if(mysubs.elementAt(i).equals(s))
		return true;
	}
	return false;
    }

    /**
     * Returns number of subunits in this Assembly
     *
     * @return int
     */
    public int numSubunits() {
	return mysubs.size();
    }
    
    /**
     * Returns a String version of this Assembly
     *
     * @return String
     */
    public String toString() {
	String mystring = new String("Assembly name: " + myname
				     + "\nSubunit IDs:\n");
	for(int i=0; i<mysubs.size(); i++) {
	    //mystring = mystring + "\t" + ((Subunit) mysubs.elementAt(i)).getid();
            mystring += "\t" + (Subunit)mysubs.get(i);
        }
	return mystring;
    }


    //event-processing functions -------------------------------------
    /**
     * Sets the min time of this Assembly to min
     *
     * @param min
     *            double
     */
    public void setMinTime(double min) {
        minTime = min;
    }
    
    /**
     * Returns the min time of this Assembly
     *
     * @return double
     */
    public double minTime() {
        return minTime;
    }
    
    /**
     * Sets the next event for this Assembly to ev
     *
     * @param ev
     *            Event
     */
    public void setNextEvent(Event ev) {
        nextEvent = ev;
    }
    
    /**
     * Returns this Assembly's next Event
     *
     * @return Event
     */
    public Event nextEvent() {
        return nextEvent;
    }

    /**
     * Returns this Assembly's next Event's end time
     *
     * @return double
     */
    //note: throws pointer exception if event isn't set
    public double nextEventEndTime() {
	if(nextEvent != null)
	    return nextEvent.getEndTime();
	return Double.MAX_VALUE;
    }
    
    /**
     * Sets the valid time for this Assembly to time
     *
     * @param time
     *            double
     */
    public void setValidTime(double time) {
        validTime = time;
    }

    /**
     * Returns this Assembly's valid time
     *
     * @return double
     */
    public double validTime() {
        return validTime;
    }
    
    /**
     * Returns this true if this Assembly is the same object as Assembly a
     *     
     * @param a
     *            Assembly
     *     
     * @return boolean
     */
    public boolean equals(Assembly a) {
        return myname.equals(a.getName());
    }
}
