
import java.util.Vector;
import java.util.HashMap;
import java.util.Random;
import javax.vecmath.*;

/**
 * Actin sets up a filament-like scenerio, where there is one type of 
 * Subunit with a single Conformation and Domain and two binding sites 
 * on opposite ends.  
 * The initial number of Subunits, and the binding and breaking times
 * control the assembly of Actin
 * 
 * @author Rori Rohlfs
 * @version 1.0
 *  
 */
public class Actin {
    private static Simulation sim;

    private static Solution soln;

    private static Vector assemblies;

    private static int bsID = 0; //BindingSite ID

    private static int domainID = 0; //Domain ID

    private static int stID = 0; //SubunitType ID

    private static int suID = 0; //SubunitID

    //binding and breaking rates are set here:
    private double bindTime = 2.0;    //average time for a binding event

    private double breakTime = 2.0;  //average time for a breaking event

    private double fastbindtime = 0.01; //not applicable in an Actin simulation (no loops are formed)

    //Initial number of Subunits are set here.
    private static int numSubunits = 50;

    /**
     * Constructs an Actin Simulation.  
     *  
     */
    public Actin() {
        soln = new Solution();
        initAssembly();
        
        double[] bindbreak1 = new double[3];
        bindbreak1[0] = bindTime;
        bindbreak1[1] = breakTime;
        bindbreak1[2] = fastbindtime;
        double[] bindbreak2 = new double[3];
        bindbreak2[0] = bindTime;
        bindbreak2[1] = breakTime;
        bindbreak2[2] = fastbindtime;

        double[] nobindbreak = new double[3];
    	nobindbreak[0] = 99999999999999.0;
    	nobindbreak[1] = 0.00000000000001;
    	nobindbreak[2] = 99999999999999.0;

        HashMap bst1HashMap = new HashMap();
        bst1HashMap.put(new String("bsta"), nobindbreak);
        bst1HashMap.put(new String("bstb"), bindbreak1);

        HashMap bst2HashMap = new HashMap();
        bst2HashMap.put(new String("bsta"), bindbreak2);
        bst2HashMap.put(new String("bstb"), (double[]) nobindbreak.clone());

        HashMap bondtimes = new HashMap();
        bondtimes.put(new String("bsta"), bst1HashMap);
        bondtimes.put(new String("bstb"), bst2HashMap);

        HashMap conftimes = new HashMap();

        sim = new Simulation(soln, assemblies, 0.0, bondtimes, conftimes);
    }

    /**
     * Returns Simulation of Actin
     * 
     * @return Simulation
     */
    public static Simulation getSim() {
        return sim;
    }

    /**
     * 
     * Creates all Subunits with appropriate bsID, domainID, ConformationID and stID.  
     * Adds all Assembly to Vector Assemblies
     *  
     */
    private static void initAssembly() {
        assemblies = new Vector();

        //Creates two BindingSiteType objects with ID 0 and 1
        double[] tolerances = { 0.2, 0.1, 0.1 };
        Vector partners1 = new Vector();
        Vector partners2 = new Vector();
        partners1.add("bstb");
        BindingSiteType bsta = new BindingSiteType(tolerances, "bsta", partners1);
        partners2.add("bsta");
        BindingSiteType bstb = new BindingSiteType(tolerances, "bstb", partners2);

        //make pos and rot arrays for binding sites
        Vector3d posD1 = new Vector3d(-0.08, 0.0, 0.0);//get from radius
        Vector3d posD2 = new Vector3d(0.08, 0.0, 0.0);
        Quat4d orD = new Quat4d(0.0, 0.0, -1.0 / Math.pow(2, 1 / 2), -1.0
                / Math.pow(2, 1 / 2)); //points left
        Quat4d orD2 = new Quat4d(0.0, 0.0, 1.0 / Math.pow(2, 1 / 2), -1.0
                / Math.pow(2, 1 / 2)); //points right


        //loop for making subunittypes
        Vector subts = new Vector();
        for (int i = 0; i < numSubunits; i++) {
            String asmname = "assem" + i;
            BindingSite bsa = new BindingSite(bsta, bsID++, domainID, suID,
                    asmname, posD1, orD, new Vector3d(0.0, 0.0, 1.0));
            BindingSite bsb = new BindingSite(bstb, bsID++, domainID, suID,
                    asmname, posD2, orD2, new Vector3d(0.0, 0.0, 1.0));

            /**
             * Adds two BindingSites of one Subunit to Vector bss to create
             * Conformation
             */
            Vector bss = new Vector();
            bss.add(bsa);
            bss.add(bsb);

            /**
             * Adds one Conformation of a Subunit to Vector confs to create
             * Domain
             */
            Vector confs = new Vector();
            Conformation conf = new Conformation(bss, 6.6, "subt1only");
            confs.add(conf);

            /**
             * Adds one Domain of a Subunit to Vector Confsets to create
             * SubunitType
             */
            Domain dom = new Domain("only", confs, new Vector3d(), domainID);
            Vector confSets = new Vector();
            confSets.add(dom);

            SubunitType subt = new SubunitType(confSets, "only", soln);

            subts.add(subt);

	    /**
	     * set initial position, rotation, velocity, and rotational velocity
	     */
            Vector3d sPos = new Vector3d();
            Quat4d sRot = new Quat4d(0.0, 0.0, 0.0, 1.0);
            Vector3d sVel = new Vector3d();
            Quat4d srVel = new Quat4d(0.0, 0.0, 0.0, 1.0);
         
            Subunit subunit = new Subunit(subt, confs, soln, suID++, sPos,
                    sRot, sVel, srVel, 3.4, 0.08);

            Vector asmsubs = new Vector();
            asmsubs.add(subunit);
            Assembly asm = new Assembly(asmsubs, asmname);

            assemblies.add(asm);
        }

    }

}
