/*
 * wm97xx-battery.c  --  Battery monitor for
 *                         Wolfson WM97xx AC97 Codecs.
 *
 * Copyright 2004 Wolfson Microelectronics PLC.
 * Author: Liam Girdwood
 *         liam.girdwood@wolfsonmicro.com or linux@wolfsonmicro.com
 * Parts Copyright : Ian Molton <spyro@f2s.com>
 *                   Andrew Zabolotny <zap@homelink.ru>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Notes:
 *     This code uses an API exposed by ac97_plugin_wm97xx to capture battery
 *     data.
 * 
 *  Features:
 *       - codecs supported:- WM9712, WM9713
 *
 *  Revision history
 *    20th Sep 2004   Initial version.
 * 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/arch/pxa-regs.h>

#include "wm97xx.h"

#define TS_NAME			"wm97xx_battery"
#define WM_TS_VERSION		"0.2"

/*
 * Battery sampling period jiffies.
 */
static int period = 2;
module_param(period, int, 0);
MODULE_PARM_DESC(period, "Battery sampling period (secs)");

static struct wm97xx_device *dev;

#ifdef CONFIG_PXA27x
/*
 * Sometimes the PXA27x cannot reset sticky codec GPIO's. In this case we
 * also need to reset the GPIO in codec space.
 */
static int pxa27x_clear_sticky(int status)
{
	volatile u32 *reg_addr;
	u32 gsr;

	reg_addr = (u32 *)&PAC_REG_BASE + (0x54 >> 1);
	*reg_addr = (status & ~WM97XX_GPIO_13) << 1;
		
	udelay(50);
    gsr = GSR;
    GSR = gsr & GSR_CDONE;
 
	if (!(gsr & GSR_CDONE)) {
		printk(KERN_CRIT "%s: write codec register timeout.\n", __FUNCTION__);
        return 0;
	}
	return 1;
}
#endif


/*
 * PXA arch pen down handler.
 */
static int pxa_handler(int status)
{
	u32 gsr;
#ifdef CONFIG_PXA27x
	pxa27x_clear_sticky(status);
#endif
	gsr = GSR;
	GSR = gsr & GSR_GSCI;
	return 0;
}

static int battery_dead_handler(int status)
{
	printk("dead\n");
	pxa_handler(status);
	return 0;
}

static int battery_low_handler(int status)
{
	printk("low\n");
	pxa_handler(status);
	return 0;
}

static void battery_read (wm97xx_aux_conv_t ac, int stage)
{
	int val = 0;

	if (stage) {
		val = wm97xx_get_auxconv(dev, WM97XX_AC_BMON);
		printk("got %d\n", val & 0xfff);
	}
}

static int pxa_wm97xx_probe(void)
{
	/* register the battery alarm handlers */
	switch (wm97xx_id & 0xffff) {
		case WM9705_ID2:
			return -ENODEV;
		case WM9712_ID2:
		case WM9713_ID2:
			/* use the virtual PEN_DOWN GPIO to send irq over AC97 */
			wm97xx_set_codec_irq(dev, IRQ_AC97, 1);
			wm97xx_request_codec_event(dev, WM97XX_GPIO_14, battery_low_handler);
			wm97xx_request_codec_event(dev, WM97XX_GPIO_15, battery_dead_handler);
			GCR |= GCR_GIE;
			break;
	}
	
	/* register battery reader */
	wm97xx_request_auxconv(dev, WM97XX_AC_BMON, period * HZ, battery_read);
	return 0;
}

static int pxa_wm97xx_remove(void)
{
	wm97xx_free_auxconv(dev, WM97XX_AC_BMON);
	
	/* codec specific deconfig */
	switch (wm97xx_id & 0xffff) {
		case WM9712_ID2:
		case WM9713_ID2:
			/* disable interrupt */
			if (wm97xx_clear_codec_irq(dev) == 0)
				GCR &= ~GCR_GIE;
			wm97xx_free_codec_event(dev, WM97XX_GPIO_14);
			wm97xx_free_codec_event(dev, WM97XX_GPIO_15);
			break;
	}
	return 0;
}

#if defined(CONFIG_PM)
static int pxa_wm97xx_suspend(u32 state, u32 level)
{
	return 0;
}

static int pxa_wm97xx_resume(u32 level)
{
	return 0;
}
#else
#define pxa_wm97xx_suspend NULL
#define pxa_wm97xx_resume NULL
#endif

static struct wm97xx_device pxa_wm97xx_batt_driver = {
        .name   = "pxa-wm97xx-battery",
        .probe  = pxa_wm97xx_probe,
        .remove = pxa_wm97xx_remove,
        .suspend= pxa_wm97xx_suspend,
        .resume = pxa_wm97xx_resume,
		.cont_read = NULL,
};

static int __init pxa_wm97xx_battery_init(void)
{
	dev = &pxa_wm97xx_batt_driver;
	return wm97xx_driver_register(dev);
}

static void __exit pxa_wm97xx_battery_exit(void)
{
	wm97xx_driver_unregister(dev);
	dev = NULL;
}

/* Module information */
MODULE_AUTHOR("Liam Girdwood <liam.girdwood@wolfsonmicro.com>");
MODULE_DESCRIPTION("PXA WM97xx Battery Monitor");
MODULE_LICENSE("GPL");

module_init(pxa_wm97xx_battery_init);
module_exit(pxa_wm97xx_battery_exit);
