/*
 *  linux/drivers/sound/pxa-ac97.c -- AC97 interface for the Cotula chip
 *
 *  Author:	Nicolas Pitre
 *  Created:	Aug 15, 2001
 *  Copyright:	MontaVista Software Inc.
 *
 *  Forward ported to 2.6 by Ian Molton 15/09/2003
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  Sep 1st 2004: Liam Girdwood <liam.girdwood@wolfsonmicro.com>
 *                Changed codec initialisation sequence to perform a cold reset 
 *                followed by a warm reset. This allows a greater range of 
 *                codecs to be used on the XScale.
 *                Added codec init failure error detection, fast codec GPIO
 *                for PXA25x and PXA27x, new PXA27x AC97 read/write and
 *                shared the AC97 interrupt for use with codec specifig GPIO's.
 *
 *  Mar/Apr 2005: Matthias Ihmig <m.ihmig@mytum.de>
 *                Modified for Glencoe and backport of some patches from later versions
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/completion.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/ac97_codec.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/semaphore.h>
#include <asm/dma.h>
#include <asm/arch/pxa-regs.h>
#include "pxa-audio.h"

#ifdef CONFIG_MACH_MAINSTONE
	#include <asm/arch/mainstone.h>
#endif
#ifdef CONFIG_MACH_GLENCOE
	#include <asm/arch/glencoe.h>
#endif


/* We need a dummy irq tag to request/free our shared irq handler */
#define AC97_IRQ_TAG	(void*)0xac97c0de

//#define DEBUG 1

static DECLARE_MUTEX(CAR_mutex);

/*
 * PXA27x has a different AC97 IO flow than PXA25x.
 */
#ifdef CONFIG_PXA27x

static int pxa27x_read_wait(void)
{
	/* XLLP: timeout in reading and writing codec registers through AC link is 200us */
        /* a PXA27x AC97 read takes 4 AC link frames to complete */
	int timeout = 20;
	u32 gsr;

	while ( (!(GSR & GSR_SDONE)) && (timeout--) ) {
		udelay(10);
	}

	gsr = GSR;
	GSR = gsr | GSR_SDONE | GSR_CDONE;

	if (timeout <= 0) {
		printk(KERN_CRIT "%s: read codec register timeout.\n", __FUNCTION__);
		return 0;
	}
#ifdef DEBUG
	else {
		printk("\n<%s>: finish in %d us. \n",__FUNCTION__, (20-timeout)*10);
	}
#endif

	return 1;
}

static int pxa27x_write_wait(void)
{
	u32 gsr;
	/* XLLP: timeout in reading and writing codec registers through AC link is 200us */
	int timeout = 20;

	while ( (!(GSR & GSR_CDONE)) && (timeout--) ) {
		udelay(10);
	}

	/* a PXA27x AC97 write takes 2 AC link frames to complete */
	udelay (50);
	gsr = GSR;
	GSR = gsr | GSR_CDONE;

	if (timeout <= 0) {
		printk(KERN_CRIT "%s: write codec register timeout.\n", __FUNCTION__);
		return 0;
	}		
#ifdef DEBUG
	else{
		printk("\n<%s>: finish in %d us. \n",__FUNCTION__, (20-timeout)*10);
	}
#endif

	return 1;
}

static u16 pxa_ac97_read(struct ac97_codec *codec, u8 reg)
{
	u16 val =-1;
	volatile u32 *reg_addr;
	
	down(&CAR_mutex);

	/* set up primary or secondary space */
	if (codec->id & 0x1)
		reg_addr = (u32 *)&SAC_REG_BASE + (reg >> 1);
	else
		reg_addr = (u32 *)&PAC_REG_BASE + (reg >> 1);
		
	/* No fast codec GPIO reads on PXA27x */
	(void)*reg_addr;			// dummy read
	if (pxa27x_read_wait()) {
		val = *reg_addr;
		if (!pxa27x_read_wait())
			val = -1;
	}
		
	up(&CAR_mutex);
	return val;
}

static void pxa_ac97_write(struct ac97_codec *codec, u8 reg, u16 val)
{
	volatile u32 *reg_addr;
	down(&CAR_mutex);
		
	/* Status reg (0x54) writes are fast, others need to wait */ 
	if (reg == AC97_GPIO_STATUS) {
		
		/* set up primary or secondary modem space */
		if (codec->id & 0x1)
			reg_addr = (u32 *)&SMC_REG_BASE + (reg >> 1);
		else
			reg_addr = (u32 *)&PMC_REG_BASE + (reg >> 1);
		
		*reg_addr = val;
		udelay(50);
	} else {
		/* set up primary or secondary codec space */
		if (codec->id & 0x1)
			reg_addr = (u32 *)&SAC_REG_BASE + (reg >> 1);
		else
			reg_addr = (u32 *)&PAC_REG_BASE + (reg >> 1);
			
		*reg_addr = val;
		pxa27x_write_wait();
	}
	up(&CAR_mutex);
}

#else

static struct completion CAR_completion;
static int waitingForMask;

static u16 pxa_ac97_read(struct ac97_codec *codec, u8 reg)
{
	u16 val = -1;

	down(&CAR_mutex);
	if (!(CAR & CAR_CAIP)) {
		volatile u32 *reg_addr;

		/* Status reg (0x54) reads are fast, others need to wait */ 
		if (reg == AC97_GPIO_STATUS) {
			/* set up primary or secondary modem space */
			if (codec->id & 0x1)
				reg_addr = (u32 *)&SMC_REG_BASE + (reg >> 1);
			else
				reg_addr = (u32 *)&PMC_REG_BASE + (reg >> 1);
				
			val = *reg_addr;
			up(&CAR_mutex);
			return val;
		}

		/* set up primary or secondary codec space */
		if (codec->id & 0x1)
			reg_addr = (u32 *)&SAC_REG_BASE + (reg >> 1);
		else
			reg_addr = (u32 *)&PAC_REG_BASE + (reg >> 1);
			
		waitingForMask=GSR_SDONE;
		init_completion(&CAR_completion);
		(void)*reg_addr;			//start read access across the ac97 link
		wait_for_completion(&CAR_completion);

		if (GSR & GSR_RDCS) {
			GSR |= GSR_RDCS;		//write a 1 to clear
			printk(KERN_CRIT "%s: read codec register timeout.\n", __FUNCTION__);
		}

		init_completion(&CAR_completion);
		val = *reg_addr;			//valid data now but we've just started another cycle...
		wait_for_completion(&CAR_completion);

	} else {
		printk(KERN_CRIT"%s: CAR_CAIP already set\n", __FUNCTION__);
	}
	up(&CAR_mutex);

	return val;
}

static void pxa_ac97_write(struct ac97_codec *codec, u8 reg, u16 val)
{
	down(&CAR_mutex);
	if (!(CAR & CAR_CAIP)) {
		volatile u32 *reg_addr;
		
		/* Status reg (0x54) writes are fast, others need to wait  */ 
		if (reg == AC97_GPIO_STATUS) {
			/* set up primary or secondary modem space */
			if (codec->id & 0x1)
				reg_addr = (u32 *)&SMC_REG_BASE + (reg >> 1);
			else
				reg_addr = (u32 *)&PMC_REG_BASE + (reg >> 1);
				
			*reg_addr = val;
			up(&CAR_mutex);
			return;
		}

		/* set up primary or secondary codec space */
		if (codec->id & 0x1)
			reg_addr = (u32 *)&SAC_REG_BASE + (reg >> 1);
		else
			reg_addr = (u32 *)&PAC_REG_BASE + (reg >> 1);
			
		waitingForMask=GSR_CDONE;
		init_completion(&CAR_completion);
		*reg_addr = val;
		wait_for_completion(&CAR_completion);
	} else {
		printk(KERN_CRIT "%s: CAR_CAIP already set\n", __FUNCTION__);
	}
	up(&CAR_mutex);
}

static irqreturn_t pxa_ac97_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	int gsr = GSR;
	
	/* we only handle CDONE and SDONE */
	if (gsr & (GSR_SDONE|GSR_CDONE)) {
		GSR = gsr & (GSR_SDONE|GSR_CDONE);		//write a 1 to clear
		if (gsr & waitingForMask)
			complete(&CAR_completion);
		return IRQ_HANDLED;
	} else
		return IRQ_NONE;
}
#endif

static struct ac97_codec pxa_ac97_codec = {
	codec_read:		pxa_ac97_read,
	codec_write:	pxa_ac97_write,
};

static int pxa_ac97_cold_reset(void)
{
	int timeout = 50;

	GCR = 0;
	udelay(100);	/* XLLP: holdtime for keeping nReset signal active(low) in AC link is 100us */
#ifdef CONFIG_PXA27x
	GCR = GCR_COLD_RST;
#else
	GCR = GCR_COLD_RST|GCR_CDONE_IE|GCR_SDONE_IE;
#endif
	/* link should be up after 10 uS */
	udelay(10);
	/* XLLP: timeout in waiting for codec's ready signal during setup process is 5ms */
	while (!(GSR & GSR_PCR) && timeout--) {
		udelay(100);
	}
	if (timeout <= 0) {
		printk(KERN_CRIT "%s: cold reset codec timeout.\n", __FUNCTION__);
		return 0;
	}
#ifdef DEBUG
	else{
		printk("\n<%s>: finish in %d us. \n",__FUNCTION__, (50-timeout)*10);
	}
#endif

	return 1;
}

static int pxa_ac97_warm_reset(void)
{
	int timeout = 50;

#ifdef CONFIG_PXA27x
	GCR |= GCR_WARM_RST;
#else
	GCR |= GCR_WARM_RST|GCR_CDONE_IE|GCR_SDONE_IE;
#endif
	udelay(500);

	/* XLLP: timeout in waiting for codec's ready signal during setup process is 5ms */
	while (!(GSR & GSR_PCR) && timeout--) {
		udelay(100);
	}
	if (timeout < 0) {
		printk(KERN_CRIT "%s: warm reset codec timeout.\n", __FUNCTION__);
		return 0;
	}
#ifdef DEBUG
	else{
		printk("\n<%s>: finish in %d us. \n",__FUNCTION__, (50-timeout)*10);
	}
#endif

	return 1;
}

static DECLARE_MUTEX(pxa_ac97_mutex);
static int pxa_ac97_refcount;

int pxa_ac97_get(struct ac97_codec **codec)
{
	int ret;

	*codec = NULL;
	down(&pxa_ac97_mutex);

	if (!pxa_ac97_refcount) {	
#ifndef CONFIG_PXA27x
		ret = request_irq(IRQ_AC97, pxa_ac97_irq, SA_SHIRQ, "AC97", AC97_IRQ_TAG);
		if (ret)
			return ret;
#endif			
		pxa_set_cken(CKEN2_AC97, 1);

		/*
		 * On Mainstone, we enable the on board audio amp and
		 * route AC97_SYSCLK via GPIO45 to the audio daughter card
		 */
#ifdef CONFIG_MACH_MAINSTONE
		MST_MSCWR2 &= 0xffff000b;
		pxa_gpio_mode(GPIO45_SYSCLK_AC97_MD);
#endif

#ifdef CONFIG_MACH_GLENCOE	
		pxa_gpio_mode(GLN_GPIO_nAUDIO_PWR_DWN | GPIO_OUT);
		GPIO_set(GLN_GPIO_nAUDIO_PWR_DWN);
		
		pxa_gpio_mode(GLN_GPIO_nAC97_INT | GPIO_IN);
		set_irq_type(GLENCOE_nAC97_IRQ, IRQT_FALLING);
#endif

		pxa_gpio_mode(GPIO31_SYNC_AC97_MD);
		pxa_gpio_mode(GPIO30_SDATA_OUT_AC97_MD);
		pxa_gpio_mode(GPIO28_BITCLK_AC97_MD);
		pxa_gpio_mode(GPIO29_SDATA_IN_AC97_MD);

		/*
		 * Use GPIO 113 as AC97 Reset on Bulverde
		 */
#ifdef CONFIG_PXA27x
		pxa_gpio_mode(GPIO113_AC97_RESET_N_MD);
#endif

		/* 
		 * We try a cold reset followed by a warm reset. If the codec doesn't
		 * respond after this then something is wrong.
		 */
		if (pxa_ac97_cold_reset() == 0)
		 	if (pxa_ac97_warm_reset() == 0)
		 		goto ac97_fail;
		 		
		ret = ac97_probe_codec(&pxa_ac97_codec);
		if (ret != 1) 
			goto ac97_fail;
	}

	pxa_ac97_refcount++;
	up(&pxa_ac97_mutex);
	*codec = &pxa_ac97_codec;
	return 0;
	
ac97_fail:
#ifdef CONFIG_MACH_MAINSTONE
	MST_MSCWR2 &= ~0xc0;
	pxa_gpio_mode(GPIO45_BTRTS_MD);
#endif
#ifdef CONFIG_MACH_GLENCOE
	GPIO_clr(GLN_GPIO_nAUDIO_PWR_DWN);
#endif

#ifndef CONFIG_PXA27x
	free_irq(IRQ_AC97, (void*)AC97_IRQ_TAG);
#endif

	GCR = GCR_ACLINK_OFF;
	pxa_set_cken(CKEN2_AC97, 0);
	printk(KERN_ERR "could not initialise AC97 codec.\n");
	return -ENODEV;
}

void pxa_ac97_put(void)
{
	down(&pxa_ac97_mutex);
	pxa_ac97_refcount--;
	if (!pxa_ac97_refcount) {
#ifdef CONFIG_MACH_MAINSTONE
		MST_MSCWR2 &= ~0xc0;
		pxa_gpio_mode(GPIO45_BTRTS_MD);
#endif
#ifdef CONFIG_MACH_GLENCOE
		GPIO_clr(GLN_GPIO_nAUDIO_PWR_DWN);
#endif

		GCR = GCR_ACLINK_OFF;
		pxa_set_cken(CKEN2_AC97, 0);
#ifndef CONFIG_PXA27x 
		free_irq(IRQ_AC97, AC97_IRQ_TAG);
#endif
	}
	up(&pxa_ac97_mutex);
}

EXPORT_SYMBOL(pxa_ac97_get);
EXPORT_SYMBOL(pxa_ac97_put);


/*
 * Audio Mixer stuff
 */

static audio_state_t ac97_audio_state;
static audio_stream_t ac97_audio_in;

static int mixer_ioctl( struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg)
{
	int ret;

	ret = pxa_ac97_codec.mixer_ioctl(&pxa_ac97_codec, cmd, arg);
	if (ret)
		return ret;

	/* We must snoop for some commands to provide our own extra processing */
	switch (cmd) {
	case SOUND_MIXER_WRITE_RECSRC:
		/*
		 * According to the PXA250 spec, mic-in should use different
		 * DRCMR and different AC97 FIFO.
		 * Unfortunately current UCB1400 versions (up to ver 2A) don't
		 * produce slot 6 for the audio input frame, therefore the PXA
		 * AC97 mic-in FIFO is always starved.
		 */
#if 0
		ret = get_user(val, (int *)arg);
		if (ret)
			return ret;
		pxa_audio_clear_buf(&ac97_audio_in);
		*ac97_audio_in.drcmr = 0;
		if (val & (1 << SOUND_MIXER_MIC)) {
			ac97_audio_in.dcmd = DCMD_RXMCDR;
			ac97_audio_in.drcmr = &DRCMRRXMCDR;
			ac97_audio_in.dev_addr = __PREG(MCDR);
		} else {
			ac97_audio_in.dcmd = DCMD_RXPCDR;
			ac97_audio_in.drcmr = &DRCMRRXPCDR;
			ac97_audio_in.dev_addr = __PREG(PCDR);
		}
		if (ac97_audio_state.rd_ref)
			*ac97_audio_in.drcmr =
				ac97_audio_in.dma_ch | DRCMR_MAPVLD;
#endif
		break;
	}
	return 0;
}

static struct file_operations mixer_fops = {
	ioctl:		mixer_ioctl,
	llseek:		no_llseek,
	owner:		THIS_MODULE
};

/*
 * AC97 codec ioctls
 */

static int codec_adc_rate = 48000;
static int codec_dac_rate = 48000;

static int ac97_ioctl(struct inode *inode, struct file *file,
		      unsigned int cmd, unsigned long arg)
{
	int ret;
	long val = 0;

	switch(cmd) {
	case SNDCTL_DSP_STEREO:
		ret = get_user(val, (int *) arg);
		if (ret)
			return ret;
		/* FIXME: do we support mono? */
		ret = (val == 0) ? -EINVAL : 1;
		return put_user(ret, (int *) arg);

	case SNDCTL_DSP_CHANNELS:
	case SOUND_PCM_READ_CHANNELS:
		/* FIXME: do we support mono? */
		return put_user(2, (long *) arg);

	case SNDCTL_DSP_SPEED:
		ret = get_user(val, (long *) arg);
		if (ret)
			return ret;
		if (file->f_mode & FMODE_READ)
			codec_adc_rate = ac97_set_adc_rate(&pxa_ac97_codec, val);
		if (file->f_mode & FMODE_WRITE)
			codec_dac_rate = ac97_set_dac_rate(&pxa_ac97_codec, val);
		/* fall through */
	case SOUND_PCM_READ_RATE:
		if (file->f_mode & FMODE_READ)
			val = codec_adc_rate;
		if (file->f_mode & FMODE_WRITE)
			val = codec_dac_rate;
		return put_user(val, (long *) arg);

	case SNDCTL_DSP_SETFMT:
	case SNDCTL_DSP_GETFMTS:
		/* FIXME: can we do other fmts? */
		return put_user(AFMT_S16_LE, (long *) arg);

	default:
		/* Maybe this is meant for the mixer (As per OSS Docs) */
		return mixer_ioctl(inode, file, cmd, arg);
	}
	return 0;
}


/*
 * Audio stuff
 */

static audio_stream_t ac97_audio_out = {
	name:			"AC97 audio out",
	dcmd:			DCMD_TXPCDR,
	drcmr:			&DRCMRTXPCDR,
	dev_addr:		__PREG(PCDR),
};

static audio_stream_t ac97_audio_in = {
	name:			"AC97 audio in",
	dcmd:			DCMD_RXPCDR,
	drcmr:			&DRCMRRXPCDR,
	dev_addr:		__PREG(PCDR),
};

static audio_state_t ac97_audio_state = {
	output_stream:		&ac97_audio_out,
	input_stream:		&ac97_audio_in,
	client_ioctl:		ac97_ioctl,
	sem:			__MUTEX_INITIALIZER(ac97_audio_state.sem),
};

static int ac97_audio_open(struct inode *inode, struct file *file)
{
	return pxa_audio_attach(inode, file, &ac97_audio_state);
}

static int pxa_ac97_suspend(struct device * dev, u32 state, u32 level)
{
	u16 reg, pwrmode;
	int ret, i = 0;

	if ( level != SUSPEND_POWER_DOWN )
		return 0;

	switch (state) {
		case 1: pwrmode = AC97_PWR_D1; break;
		case 2: pwrmode = AC97_PWR_D2; break;
		case 3: pwrmode = AC97_PWR_D3; break;
		default: return 1;
	}

	ret = ac97_suspend(&pxa_ac97_codec, state);
	if (ret)
		return ret;

	/* now gracefully power down the codec and AC link */
	reg = pxa_ac97_read(&pxa_ac97_codec, AC97_POWER_CONTROL);
	reg |= (pwrmode & ~(AC97_PWR_PR6|AC97_PWR_PR5|AC97_PWR_PR4));
	pxa_ac97_write(&pxa_ac97_codec, AC97_POWER_CONTROL, reg);
	udelay(500);

	reg = pxa_ac97_read(&pxa_ac97_codec, AC97_POWER_CONTROL);
	reg |= pwrmode;
	pxa_ac97_write(&pxa_ac97_codec, AC97_POWER_CONTROL, reg);

	GCR |= GCR_ACLINK_OFF;
	pxa_set_cken(CKEN31_AC97, 1);
	while (!(GSR & (1 << 3)) && i < 99999)
		i++;
	pxa_set_cken(CKEN31_AC97, 0);
	udelay(20); 
	pxa_set_cken(CKEN2_AC97, 0);

	return 0;
}

static int pxa_ac97_resume(struct device * dev, u32 level)
{
	if ( level != RESUME_POWER_ON ) 
		return 0;

	pxa_set_cken(CKEN2_AC97, 1);
#ifdef CONFIG_MACH_MAINSTONE
		MST_MSCWR2 &= 0xffff000b;
#endif		
	if (pxa_ac97_warm_reset() < 0) {
		/* ok, warm reset didn't wake the codec, so we have to do 
		 * a cold reset and restore the codec state 
		 */
		ac97_probe_codec(&pxa_ac97_codec);
	} 
	/* XLLP: timeout in waiting for codec's ready signal during setup process is 500us */
	int timeout=50;
	while (!(GSR & GSR_PCR) && timeout--) {
		udelay(10);
	}

	if (timeout < 0) {
		printk(KERN_CRIT "%s: resume codec timeout.\n", __FUNCTION__);
	}
#ifdef DEBUG
	else{
		printk("\n<%s>: finish in %d us. \n",__FUNCTION__, (50-timeout)*10);
	}
#endif

	/* make sure that everything we need is powered up */
	pxa_ac97_write(&pxa_ac97_codec, AC97_POWER_CONTROL, AC97_PWR_D0);
	ac97_resume(&pxa_ac97_codec);

	return 0;
}

/*
 * Missing fields of this structure will be patched with the call
 * to pxa_audio_attach().
 */

static struct file_operations ac97_audio_fops = {
	open:		ac97_audio_open,
	owner:		THIS_MODULE
};


static int __devinit pxa_ac97_probe(struct device *dev)
{
	int ret;
	struct ac97_codec *dummy;

	ret = pxa_ac97_get(&dummy);
	if (ret)
		return ret;

	ac97_audio_state.dev_dsp = register_sound_dsp(&ac97_audio_fops, -1);
	pxa_ac97_codec.dev_mixer = register_sound_mixer(&mixer_fops, -1);
	ac97_audio_state.output_stream->dev = dev;
	ac97_audio_state.input_stream->dev = dev;
	
	return 0;
}

static int __devexit pxa_ac97_remove(struct device *dev)
{
	unregister_sound_dsp(ac97_audio_state.dev_dsp);
	unregister_sound_mixer(pxa_ac97_codec.dev_mixer);
	pxa_ac97_put();
	ac97_audio_state.output_stream->dev = NULL;
	ac97_audio_state.input_stream->dev = NULL;
	return 0;
}

static struct device_driver pxa_ac97_driver = {
	.name           = "pxa-ac97",
	.bus            = &platform_bus_type,
	.probe          = pxa_ac97_probe,
	.remove         = pxa_ac97_remove,
	.suspend        = pxa_ac97_suspend,
	.resume         = pxa_ac97_resume,
	.shutdown       = NULL,
};
 
static int __init pxa_ac97_init(void)
{
	return driver_register(&pxa_ac97_driver);
}

static void __exit pxa_ac97_exit(void)
{
	driver_unregister(&pxa_ac97_driver);
}

module_init(pxa_ac97_init);
module_exit(pxa_ac97_exit);

MODULE_AUTHOR("Nicolas Pitre");
MODULE_DESCRIPTION("AC97 interface for the Cotula chip");
MODULE_LICENSE("GPL");
