/*
 * linux/drivers/pcmcia/pxa2xx_glencoe.c
 *
 * Glencoe CompactFlash specific routines,
 *  based on pxa2xx_mainstone.c by Nicolas Pitre
 *  adapted for CompactFlash by Matthias Ihmig <m.ihmig@mytum.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/device.h>

#include <pcmcia/ss.h>

#include <asm/hardware.h>
#include <asm/irq.h>

#include <asm/arch/pxa-regs.h>
#include <asm/arch/glencoe.h>

#include "soc_common.h"


static struct pcmcia_irqs irqs[] = {
	{ 0, GLENCOE_S0_CD_IRQ, "CF0 CD" },
	{ 1, GLENCOE_S1_CD_IRQ, "CF1 CD" },
};

static int gln_pcmcia_hw_init(struct soc_pcmcia_socket *skt)
{
	/*
	 * Setup default state of GPIO outputs
	 * before we enable them as outputs.
	 */
	GPSR(GPIO48_nPOE) =
		GPIO_bit(GPIO48_nPOE) |
		GPIO_bit(GPIO49_nPWE) |
		GPIO_bit(GPIO50_nPIOR) |
		GPIO_bit(GPIO51_nPIOW) |
		GPIO_bit(GPIO85_nPCE_1) |
		GPIO_bit(GPIO54_nPCE_2);

	pxa_gpio_mode(GPIO48_nPOE_MD);
	pxa_gpio_mode(GPIO49_nPWE_MD);
	pxa_gpio_mode(GPIO50_nPIOR_MD);
	pxa_gpio_mode(GPIO51_nPIOW_MD);
	pxa_gpio_mode(GPIO85_nPCE_1_MD);
	pxa_gpio_mode(GPIO54_nPCE_2_MD);
	pxa_gpio_mode(GPIO79_pSKTSEL_MD);
	pxa_gpio_mode(GPIO55_nPREG_MD);
	pxa_gpio_mode(GPIO56_nPWAIT_MD);
	pxa_gpio_mode(GPIO57_nIOIS16_MD);
	
	GPIO_clr(GLN_GPIO_CF_PWR_ON);
	
	pxa_gpio_mode(GLN_GPIO_nCF1_IRQ | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nCF2_IRQ | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nCF1_CARD_DET | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nCF2_CARD_DET | GPIO_IN);
	
	pxa_gpio_mode(GLN_GPIO_CF_PWR_ON | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_CF1_RST | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_CF2_RST | GPIO_OUT);
	
	GPIO_set(GLN_GPIO_CF_PWR_ON);

	skt->irq = (skt->nr == 0) ? GLENCOE_S0_IRQ : GLENCOE_S1_IRQ;
	return soc_pcmcia_request_irqs(skt, irqs, ARRAY_SIZE(irqs));
}

static void gln_pcmcia_hw_shutdown(struct soc_pcmcia_socket *skt)
{
	soc_pcmcia_free_irqs(skt, irqs, ARRAY_SIZE(irqs));
	GPIO_clr(GLN_GPIO_CF_PWR_ON);
}

static void gln_pcmcia_socket_state(struct soc_pcmcia_socket *skt,
				    struct pcmcia_state *state)
{
	if (skt->nr == 0) {
		state->detect = (GPIO_lev(GLN_GPIO_nCF1_CARD_DET)) ? 0 : 1;
		state->ready  = (GPIO_lev(GLN_GPIO_nCF1_IRQ))      ? 1 : 0;
	} else {
		state->detect = (GPIO_lev(GLN_GPIO_nCF2_CARD_DET)) ? 0 : 1;
		state->ready  = (GPIO_lev(GLN_GPIO_nCF2_IRQ))      ? 1 : 0;
	}
	state->bvd1   = 1;
	state->bvd2   = 1;
	state->vs_3v  = 1;
	state->vs_Xv  = 0;
	state->wrprot = 0;  /* not available */
}

static int gln_pcmcia_configure_socket(struct soc_pcmcia_socket *skt,
				       const socket_state_t *state)
{
	int ret = 0;
	
	GPIO_set(GLN_GPIO_CF_PWR_ON); /* no need to set voltage -> just switch power on */

	printk(KERN_NOTICE "%s(): configuring sockekt %u, Vcc=%u, Vpp=%u\n",
					  __FUNCTION__, skt->nr, state->Vcc, state->Vpp);
	
	ret = -1;
	if (skt->nr == 0) {
		if (state->flags & SS_RESET)
			GPIO_set(GLN_GPIO_CF1_RST);
		else 
			GPIO_clr(GLN_GPIO_CF1_RST);
		ret = 0;
	};
	if (skt->nr == 1) {
		if (state->flags & SS_RESET)
			GPIO_set(GLN_GPIO_CF2_RST);
		else 
			GPIO_clr(GLN_GPIO_CF2_RST);
		ret = 0;
	}

	return ret;
}

static void gln_pcmcia_socket_init(struct soc_pcmcia_socket *skt)
{
}

static void gln_pcmcia_socket_suspend(struct soc_pcmcia_socket *skt)
{
}

static struct pcmcia_low_level gln_pcmcia_ops = {
	.owner			= THIS_MODULE,
	.hw_init		= gln_pcmcia_hw_init,
	.hw_shutdown		= gln_pcmcia_hw_shutdown,
	.socket_state		= gln_pcmcia_socket_state,
	.configure_socket	= gln_pcmcia_configure_socket,
	.socket_init		= gln_pcmcia_socket_init,
	.socket_suspend		= gln_pcmcia_socket_suspend,
	.nr			= 2,
};

static struct platform_device *gln_pcmcia_device;

static int __init gln_pcmcia_init(void)
{
	int ret;

	gln_pcmcia_device = kmalloc(sizeof(*gln_pcmcia_device), GFP_KERNEL);
	if (!gln_pcmcia_device)
		return -ENOMEM;
	memset(gln_pcmcia_device, 0, sizeof(*gln_pcmcia_device));
	gln_pcmcia_device->name = "pxa2xx-pcmcia";
	gln_pcmcia_device->dev.platform_data = &gln_pcmcia_ops;

	ret = platform_device_register(gln_pcmcia_device);
	if (ret)
		kfree(gln_pcmcia_device);

	return ret;
}

static void __exit gln_pcmcia_exit(void)
{
	/*
	 * This call is supposed to free our gln_pcmcia_device.
	 * Unfortunately platform_device don't have a free method, and
	 * we can't assume it's free of any reference at this point so we
	 * can't free it either.
	 */
	platform_device_unregister(gln_pcmcia_device);
}

module_init(gln_pcmcia_init);
module_exit(gln_pcmcia_exit);

MODULE_AUTHOR("Matthias Ihmig <m.ihmig@mytum.de>");
MODULE_LICENSE("GPL");
